/*
 * Simple Virtual Machine - A versatile and robust architecture to
 * easily write applications.
 * Copyright (C) 2021  Julien BRUGUIER
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

class SVM_SequenceDiagram_Participant
{
	constructor(id,svm_name,svm_position)
	{
		// Theorical
		this.id = id;
		this.svm_position = svm_position;
		this.svm_name = svm_name;
		this.svm_waiting_events_left = new Map();
		this.svm_waiting_events_right = new Map();

		// Virtual
		var c = document.createElement("canvas");
		var cc = c.getContext("2d");
		this.svm_name_size = cc.measureText(svm_name).width;

		this.svm_name_vertical = null;
		this.svm_horizontal_position = null;
		this.svm_horizontal_size_left = this.svm_name_size/2+1;
		this.svm_horizontal_size_right = this.svm_name_size/2+1;
		this.svm_vertical_start = null;
		this.svm_vertical_end = null;

		this.svm_participant_width = null;
	}
	compute_virtual(svm_horizontal_end,svm_horizontal_space,svm_participant_width,svm_vertical_start,svm_vertical_end,svm_name_vertical)
	{
		this.svm_name_vertical = svm_name_vertical;
		this.svm_participant_width = svm_participant_width;
		this.svm_vertical_start = svm_vertical_start;
		this.svm_vertical_end = svm_vertical_end;
		this.svm_horizontal_position = svm_horizontal_end+svm_horizontal_space/2+this.svm_horizontal_size_left;
		return svm_horizontal_end+svm_horizontal_space+this.svm_horizontal_size_left+this.svm_horizontal_size_right;
	}
	adapt_left(svm_size)
	{
		if(svm_size>this.svm_horizontal_size_left)
		{
			this.svm_horizontal_size_left = svm_size;
			return true;
		}
		return false;
	}
	adapt_right(svm_size)
	{
		if(svm_size>this.svm_horizontal_size_right)
		{
			this.svm_horizontal_size_right = svm_size;
			return true;
		}
		return false;
	}
	update_vertical_end(svm_vertical_end)
	{
		this.svm_vertical_end = svm_vertical_end;
	}
	add_waiting_left(svm_event)
	{
		var svm_end_wait = this.get_waiting_left()+1;
		this.svm_waiting_events_left.set(svm_event.id,svm_end_wait);
		return svm_end_wait;
	}
	add_waiting_right(svm_event)
	{
		var svm_end_wait = this.get_waiting_right()+1;
		this.svm_waiting_events_right.set(svm_event.id,svm_end_wait);
		return svm_end_wait;
	}
	get_waiting_left()
	{
		var svm_wait = 0;
		for(var [ i , e ] of this.svm_waiting_events_left)
		{
			if(e>svm_wait)
			{
				svm_wait = e;
			}
		}
		return svm_wait;
	}
	get_waiting_right()
	{
		var svm_wait = 0;
		for(var [ i , e ] of this.svm_waiting_events_right)
		{
			if(e>svm_wait)
			{
				svm_wait = e;
			}
		}
		return svm_wait;
	}
	remove_waiting_left(svm_event)
	{
		this.svm_waiting_events_left.delete(svm_event.id);
	}
	remove_waiting_right(svm_event)
	{
		this.svm_waiting_events_right.delete(svm_event.id);
	}
	draw(ctx)
	{
		ctx.lineWidth = this.svm_participant_width;
		ctx.beginPath();
		ctx.moveTo(this.svm_horizontal_position,this.svm_vertical_start);
		ctx.lineTo(this.svm_horizontal_position,this.svm_vertical_end);
		ctx.stroke();
	}
	draw_name(ctx)
	{
		ctx.fillText(this.svm_name,this.svm_horizontal_position-this.svm_name_size/2,this.svm_name_vertical);
	}
};

class SVM_SequenceDiagram_Event_Operation
{
	constructor(op,horizontal,vertical)
	{
		this.op = op;
		this.horizontal = horizontal;
		this.vertical = vertical;
	}
	draw(ctx)
	{
		if(this.op)
		{
			ctx.lineTo(this.horizontal,this.vertical);
		}
		else
		{
			ctx.moveTo(this.horizontal,this.vertical);
		}
	}
};

class SVM_SequenceDiagram_Event
{
	constructor(id,svm_event,svm_start_index,svm_start_participant,svm_end_participant,svm_start_position)
	{
		// Theorical
		this.id = id;
		this.svm_event = svm_event;
		this.svm_start_index = svm_start_index;
		this.svm_start_participant = svm_start_participant;
		this.svm_end_index = null;
		this.svm_end_participant = svm_end_participant;
		this.svm_cancelled = false;
		if(this.svm_start_participant.svm_position>=this.svm_end_participant.svm_position)
		{
			this.svm_start_wait = this.svm_start_participant.get_waiting_right();
			this.svm_end_wait = this.svm_end_participant.add_waiting_left(this);
		}
		else
		{
			this.svm_start_wait = this.svm_start_participant.get_waiting_left();
			this.svm_end_wait = this.svm_end_participant.add_waiting_right(this);
		}

		// Virtual
		var c = document.createElement("canvas");
		var cc = c.getContext("2d");
		this.svm_event_size = cc.measureText(this.svm_event).width;

		this.svm_vertical_start = null;
		this.svm_vertical_end = null;

		this.svm_line = [];
		this.svm_text_position_horizontal = null;
		this.svm_text_position_vertical = null;
		this.svm_start_position = svm_start_position;

		this.svm_event_width = null;
	}
	complete(svm_end_index,svm_cancelled)
	{
		this.svm_end_index = svm_end_index;
		this.svm_cancelled = svm_cancelled;
		if(this.svm_start_participant.svm_position>=this.svm_end_participant.svm_position)
		{
			this.svm_end_participant.remove_waiting_left(this);
		}
		else
		{
			this.svm_end_participant.remove_waiting_right(this);
		}
	}
	compute_virtual(svm_event_width,svm_participant_width,svm_vertical_start,svm_vertical_space,svm_event_space,svm_event_arrow_size,svm_event_wait_space)
	{
		this.svm_event_width = svm_event_width;
		var direction = -1;
		if(this.svm_start_participant.svm_horizontal_position<=this.svm_end_participant.svm_horizontal_position)
		{
			direction = 1;
		}
		if(this.svm_end_index == null)
		{
			this.svm_line = [
				new SVM_SequenceDiagram_Event_Operation(false,this.svm_start_participant.svm_horizontal_position+direction*(svm_participant_width/2+svm_event_space),svm_vertical_start+svm_vertical_space*this.svm_start_index),
				new SVM_SequenceDiagram_Event_Operation(true,this.svm_end_participant.svm_horizontal_position-direction*(svm_participant_width/2+svm_event_space+svm_event_wait_space*this.svm_end_wait),svm_vertical_start+svm_vertical_space*this.svm_start_index)
				];
		}
		else 
		{
			if(this.svm_start_index==this.svm_end_index)
			{
				this.svm_line = [
					new SVM_SequenceDiagram_Event_Operation(false,this.svm_start_participant.svm_horizontal_position+direction*(svm_participant_width/2+svm_event_space),svm_vertical_start+svm_vertical_space*this.svm_start_index),
					new SVM_SequenceDiagram_Event_Operation(true,this.svm_end_participant.svm_horizontal_position-direction*(svm_participant_width/2+svm_event_space),svm_vertical_start+svm_vertical_space*this.svm_end_index)
					];
			}
			else
			{
				this.svm_line = [
					new SVM_SequenceDiagram_Event_Operation(false,this.svm_start_participant.svm_horizontal_position+direction*(svm_participant_width/2+svm_event_space),svm_vertical_start+svm_vertical_space*this.svm_start_index)
					];
				if(this.svm_start_participant.svm_position == this.svm_end_participant.svm_position)
				{
					direction = -direction;
				}
				this.svm_line.push(new SVM_SequenceDiagram_Event_Operation(true,this.svm_end_participant.svm_horizontal_position-direction*(svm_participant_width/2+svm_event_space+svm_event_wait_space*this.svm_end_wait),svm_vertical_start+svm_vertical_space*this.svm_start_index));
				this.svm_line.push(new SVM_SequenceDiagram_Event_Operation(true,this.svm_end_participant.svm_horizontal_position-direction*(svm_participant_width/2+svm_event_space+svm_event_wait_space*this.svm_end_wait),svm_vertical_start+svm_vertical_space*this.svm_end_index));
				if(!this.svm_cancelled)
				{
					this.svm_line.push(new SVM_SequenceDiagram_Event_Operation(true,this.svm_end_participant.svm_horizontal_position-direction*(svm_participant_width/2+svm_event_space),svm_vertical_start+svm_vertical_space*this.svm_end_index));
				}
			}
			var svm_point = this.svm_line[this.svm_line.length-1];
			if(this.svm_cancelled)
			{
				this.svm_line.push(new SVM_SequenceDiagram_Event_Operation(false,svm_point.horizontal-svm_event_arrow_size,svm_point.vertical-svm_event_arrow_size));
				this.svm_line.push(new SVM_SequenceDiagram_Event_Operation(true,svm_point.horizontal+svm_event_arrow_size,svm_point.vertical+svm_event_arrow_size));
				this.svm_line.push(new SVM_SequenceDiagram_Event_Operation(false,svm_point.horizontal-svm_event_arrow_size,svm_point.vertical+svm_event_arrow_size));
				this.svm_line.push(new SVM_SequenceDiagram_Event_Operation(true,svm_point.horizontal+svm_event_arrow_size,svm_point.vertical-svm_event_arrow_size));
			}
			else
			{
				this.svm_line.push(new SVM_SequenceDiagram_Event_Operation(false,svm_point.horizontal-direction*svm_event_arrow_size,svm_point.vertical-svm_event_arrow_size));
				this.svm_line.push(new SVM_SequenceDiagram_Event_Operation(true,svm_point.horizontal,svm_point.vertical));
				this.svm_line.push(new SVM_SequenceDiagram_Event_Operation(true,svm_point.horizontal-direction*svm_event_arrow_size,svm_point.vertical+svm_event_arrow_size));
			}
		}
		if(this.svm_start_participant.svm_position == this.svm_end_participant.svm_position)
		{
			direction = -direction;
		}
		this.svm_text_position_vertical = svm_vertical_start+svm_vertical_space*this.svm_start_index-svm_event_space;
		var svm_recompute_virtual_all = false;
		var svm_text_shift = (svm_participant_width/2+svm_event_space)+this.svm_start_wait*svm_event_wait_space+svm_event_space;
		if(direction>0)
		{
			this.svm_text_position_horizontal = this.svm_start_participant.svm_horizontal_position+svm_text_shift;
			svm_recompute_virtual_all = this.svm_start_participant.adapt_right(this.svm_event_size+svm_event_arrow_size+3*svm_event_space);
		}
		else
		{
			this.svm_text_position_horizontal = this.svm_start_participant.svm_horizontal_position-svm_text_shift-this.svm_event_size;
			svm_recompute_virtual_all = this.svm_start_participant.adapt_left(this.svm_event_size+svm_event_arrow_size+3*svm_event_space);
		}
		this.svm_vertical_start = svm_vertical_start+svm_vertical_space*(this.svm_start_index-1);
		this.svm_vertical_end = svm_vertical_start+svm_vertical_space*(this.svm_end_index+1);
		return svm_recompute_virtual_all;
	}
	draw(ctx)
	{
		ctx.lineWidth = this.svm_event_width;
		ctx.beginPath();
		for(var o of this.svm_line)
		{
			o.draw(ctx);
		}
		ctx.stroke();
		ctx.fillText(this.svm_event,this.svm_text_position_horizontal,this.svm_text_position_vertical);
	}
};

class SVM_SequenceDiagram_Diagram
{
	constructor()
	{
		// Theorical
		this.svm_participants = new Map();
		this.svm_events = new Map();
		this.svm_events_start = [];
		this.svm_events_end = [];
		
		// Virtual
		this.svm_horizontal_start = 0;
		this.svm_horizontal_end = 0;
		this.svm_vertical_start = 20;
		this.svm_vertical_end = 20;

		this.svm_vertical_index = 0;
		this.svm_last_received_event_id = null;

		this.svm_marker_color = "#800000";
		this.svm_marker_participant = new Map();
		this.svm_marker_event = new Map();

		// Virtual: Config
		this.svm_participant_name_vertical = 12;
		this.svm_horizontal_space = 20;
		this.svm_vertical_space = 20;
		this.svm_vertical_future = 10;
		this.svm_event_width = 1;
		this.svm_participant_width = 3;
		this.svm_event_space = 3;
		this.svm_event_arrow_size = 5;
		this.svm_event_wait_space = 20;
		this.svm_diagram_color = "#000000";
		this.svm_decoration_color = "#101010";
		this.svm_move_color = "#FFFFFF";
		this.svm_move_view_color = "#000000";
		this.svm_move_arrow_width = 5;

		// Real
		this.svm_view_horizontal = 0;
		this.svm_view_vertical = 0;
		this.svm_scale = 1;
		this.svm_move = false;
		this.svm_move_origin_horizontal = null;
		this.svm_move_origin_vertical = null;
		this.svm_move_speed_horizontal = null;
		this.svm_move_speed_vertical = null;
		this.svm_move_loop_function = null;

		// Real: Config
		this.svm_move_speed_initial = 2;
		this.svm_move_speed_boost = 100;
		this.svm_move_speed_interval = 100;
		this.svm_move_origin_size = 50;
	}
	compute_virtual_participant(svm_participant)
	{
		this.svm_horizontal_end = svm_participant.compute_virtual(this.svm_horizontal_end,this.svm_horizontal_space,this.svm_participant_width,this.svm_vertical_start,this.svm_vertical_end,this.svm_participant_name_vertical);
	}
	compute_virtual_event(svm_event)
	{
		return svm_event.compute_virtual(this.svm_event_width,this.svm_participant_width,this.svm_vertical_start,this.svm_vertical_space,this.svm_event_space,this.svm_event_arrow_size,this.svm_event_wait_space);
	}
	add_event(svm_event)
	{
		var svm_start_participant = this.svm_participants.get(svm_event.source.id);
		if(svm_start_participant == undefined)
		{
			svm_start_participant = new SVM_SequenceDiagram_Participant(svm_event.source.id,svm_event.source.name,this.svm_participants.size);
			this.svm_participants.set(svm_event.source.id,svm_start_participant);
			this.compute_virtual_participant(svm_start_participant);
		}
		var svm_end_participant = this.svm_participants.get(svm_event.target.id);
		if(svm_end_participant == undefined)
		{
			svm_end_participant = new SVM_SequenceDiagram_Participant(svm_event.target.id,svm_event.target.name,this.svm_participants.size);
			this.svm_participants.set(svm_event.target.id,svm_end_participant);
			this.compute_virtual_participant(svm_end_participant);
		}
		var svm_recompute_virtual_all = false;
		if(svm_event.type=="post")
		{
			this.svm_vertical_index++;
			var ev = new SVM_SequenceDiagram_Event(svm_event.id,svm_event.event,this.svm_vertical_index,svm_start_participant,svm_end_participant,this.svm_events_start.length);
			this.svm_events.set(svm_event.id,ev);
			this.svm_events_start.push(ev);
			this.svm_last_received_event_id = svm_event.id;
			svm_recompute_virtual_all = this.compute_virtual_event(ev);
		}
		else if((svm_event.type=="recv")||(svm_event.type=="cancel"))
		{
			if((svm_event.ref!=this.svm_last_received_event_id)||(svm_event.source.id==svm_event.target.id))
			{
				this.svm_vertical_index++;
			}
			this.svm_last_received_event_id = null;
			var ev = this.svm_events.get(svm_event.ref);
			if(ev != undefined)
			{
				ev.complete(this.svm_vertical_index,svm_event.type=="cancel");
				this.svm_events_end.push(ev);
				svm_recompute_virtual_all = this.compute_virtual_event(ev);
			}
		}
		if(svm_recompute_virtual_all)
		{
			this.compute_virtual();
		}
		this.svm_vertical_end = this.svm_vertical_start+this.svm_vertical_space*(this.svm_vertical_index+this.svm_vertical_future);
		for(var [ i, p ] of this.svm_participants)
		{
			p.update_vertical_end(this.svm_vertical_end);
		}
	}
	compute_virtual()
	{
		this.svm_horizontal_end = 0;
		for(var [ i, p ] of this.svm_participants)
		{
			this.compute_virtual_participant(p);
		}
		for(var [ i , e ] of this.svm_events)
		{
			this.compute_virtual_event(e);
		}
		
	}
	increase_scale()
	{
		this.svm_scale += 0.2;
	}
	decrease_scale()
	{
		this.svm_scale -= 0.2;
		if(this.svm_scale<0.1)
		{
			this.svm_scale = 0.2;
		}
	}
	reset_scale()
	{
		this.svm_scale = 1;
	}
	toggle_move(svm_origin_horizontal,svm_origin_vertical)
	{
		if(this.svm_move)
		{
			this.svm_move = false;
			return false;
		}
		this.svm_move = true;
		this.svm_move_origin_horizontal = svm_origin_horizontal;
		this.svm_move_origin_vertical = svm_origin_vertical;
		return true;
	}
	set_move_speed(svm_move_speed_horizontal,svm_move_speed_vertical,svm_view)
	{
		if(!this.svm_move)
		{
			return;
		}
		this.svm_move_speed_horizontal = 0;
		if(svm_move_speed_horizontal<this.svm_move_origin_horizontal-this.svm_move_origin_size)
		{
			var svm_cursor_distance = this.svm_move_origin_horizontal-this.svm_move_origin_size-svm_move_speed_horizontal;
			var svm_cursor_distance_max = Math.max(1,this.svm_move_origin_horizontal-this.svm_move_origin_size);
			var svm_cursor_ratio = svm_cursor_distance/svm_cursor_distance_max;
			var svm_size_ratio = Math.max(1,(this.svm_horizontal_end-this.svm_horizontal_start)/(svm_view.dom.clientWidth));
			this.svm_move_speed_horizontal = -this.svm_move_speed_initial*this.svm_move_speed_interval/1000*Math.pow(this.svm_move_speed_boost*svm_size_ratio,svm_cursor_ratio);
		}
		if(svm_move_speed_horizontal>this.svm_move_origin_horizontal+this.svm_move_origin_size)
		{
			var svm_cursor_distance = svm_move_speed_horizontal-(this.svm_move_origin_horizontal+this.svm_move_origin_size);
			var svm_cursor_distance_max = Math.max(1,svm_view.dom.clientWidth-(this.svm_move_origin_horizontal+this.svm_move_origin_size));
			var svm_cursor_ratio = svm_cursor_distance/svm_cursor_distance_max;
			var svm_size_ratio = Math.max(1,(this.svm_horizontal_end-this.svm_horizontal_start)/(svm_view.dom.clientWidth));
			this.svm_move_speed_horizontal = this.svm_move_speed_initial*this.svm_move_speed_interval/1000*Math.pow(this.svm_move_speed_boost*svm_size_ratio,svm_cursor_ratio);
		}
		this.svm_move_speed_vertical = 0;
		if(svm_move_speed_vertical<this.svm_move_origin_vertical-this.svm_move_origin_size)
		{
			var svm_cursor_distance = this.svm_move_origin_vertical-this.svm_move_origin_size-svm_move_speed_vertical;
			var svm_cursor_distance_max = Math.max(1,this.svm_move_origin_vertical-this.svm_move_origin_size);
			var svm_cursor_ratio = svm_cursor_distance/svm_cursor_distance_max;
			var svm_size_ratio = Math.max(1,(this.svm_vertical_end-this.svm_vertical_start)/(svm_view.dom.clientHeight));
			this.svm_move_speed_vertical = -this.svm_move_speed_initial*this.svm_move_speed_interval/1000*Math.pow(this.svm_move_speed_boost*svm_size_ratio,svm_cursor_ratio);
		}
		if(svm_move_speed_vertical>this.svm_move_origin_vertical+this.svm_move_origin_size)
		{
			var svm_cursor_distance = svm_move_speed_vertical-(this.svm_move_origin_vertical+this.svm_move_origin_size);
			var svm_cursor_distance_max = Math.max(1,svm_view.dom.clientHeight-(this.svm_move_origin_vertical+this.svm_move_origin_size));
			var svm_cursor_ratio = svm_cursor_distance/svm_cursor_distance_max;
			var svm_size_ratio = Math.max(1,(this.svm_vertical_end-this.svm_vertical_start)/(svm_view.dom.clientHeight));
			this.svm_move_speed_vertical = this.svm_move_speed_initial*this.svm_move_speed_interval/1000*Math.pow(this.svm_move_speed_boost*svm_size_ratio,svm_cursor_ratio);
		}
	}
	set_move_loop(svm_view)
	{
		this.svm_move_loop_function = window.setInterval(function(svm_diagram,svm_view) {
				if(svm_diagram.move(svm_view))
				{
					svm_view.draw();
				}
			},this.svm_move_speed_interval,this,svm_view);
	}
	reset_move_loop()
	{
		window.clearInterval(this.svm_move_loop_function);
	}
	move(svm_view)
	{
		if(!this.svm_move)
		{
			return false;
		}
		var svm_horizontal = this.svm_view_horizontal;
		var svm_vertical = this.svm_view_vertical;
		this.svm_view_horizontal += this.svm_move_speed_horizontal;
		if(this.svm_view_horizontal<0)
		{
			this.svm_view_horizontal = 0;
		}
		var svm_max = Math.max(0,this.svm_horizontal_end-this.svm_horizontal_start-svm_view.dom.clientWidth/this.svm_scale)
		if(this.svm_view_horizontal>svm_max)
		{
			this.svm_view_horizontal = svm_max;
		}
		this.svm_view_vertical += this.svm_move_speed_vertical;
		if(this.svm_view_vertical<0)
		{
			this.svm_view_vertical = 0;
		}
		svm_max = Math.max(0,this.svm_vertical_end-this.svm_vertical_start-svm_view.dom.clientHeight/this.svm_scale)
		if(this.svm_view_vertical>svm_max)
		{
			this.svm_view_vertical = svm_max;
		}
		return (svm_horizontal!=this.svm_view_horizontal)||(svm_vertical!=this.svm_view_vertical);
	}
	set_marker_color(color)
	{
		this.svm_marker_color = color;
	}
	reset_markers()
	{
		this.svm_marker_participant = new Map();
		this.svm_marker_event = new Map();
	}
	toggle_marker_participant(svm_participant_id)
	{
		var svm_marker = this.svm_marker_participant.get(svm_participant_id);
		if(svm_marker == undefined)
		{
			this.svm_marker_participant.set(svm_participant_id,this.svm_marker_color);
		}
		else
		{
			this.svm_marker_participant.delete(svm_participant_id);
		}
	}
	toggle_marker_event(svm_event_index)
	{
		var svm_marker = this.svm_marker_event.get(svm_event_index);
		if(svm_marker == undefined)
		{
			this.svm_marker_event.set(svm_event_index,this.svm_marker_color);
		}
		else
		{
			this.svm_marker_event.delete(svm_event_index);
		}
	}
	toggle_marker(svm_horizontal,svm_vertical)
	{
		if(svm_vertical<this.svm_vertical_start/this.svm_scale)
		{
			return;
		}
		var svm_horizontal_diagram = this.svm_view_horizontal+svm_horizontal/this.svm_scale;
		var svm_vertical_diagram = this.svm_view_vertical+svm_vertical/this.svm_scale;
		for(var [ i , p ] of this.svm_participants)
		{
			if(Math.abs(svm_horizontal_diagram-p.svm_horizontal_position)<this.svm_vertical_space/2)
			{
				this.toggle_marker_participant(i);
				return;
			}
		}
		var svm_event_index = Math.round((svm_vertical_diagram-this.svm_vertical_start)/this.svm_vertical_space);
		this.toggle_marker_event(svm_event_index);
	}
	draw(ctx,width,height)
	{
		ctx.resetTransform();
		ctx.clearRect(0, 0, width, height);
		ctx.scale(this.svm_scale,this.svm_scale);
		ctx.translate(-this.svm_view_horizontal,-this.svm_view_vertical);
		ctx.strokeStyle = this.svm_diagram_color;
		ctx.fillStyle = this.svm_diagram_color;
		for(var [ i, p ] of this.svm_participants)
		{
			var svm_marker = this.svm_marker_participant.get(i);
			if(svm_marker != undefined)
			{
				ctx.strokeStyle = svm_marker;
			}
			p.draw(ctx);
			ctx.strokeStyle = this.svm_diagram_color;
		}
		var svm_index_before = 0;
		var svm_index_after = this.svm_events_end.length;
		while(true)
		{
			if(svm_index_after-svm_index_before<=1)
			{
				svm_index = svm_index_before;
				break;
			}
			var svm_index = Math.floor((svm_index_before+svm_index_after)/2);
			if(this.svm_events_end[svm_index].svm_vertical_end<this.svm_view_vertical)
			{
				svm_index_before = svm_index;
			}
			else
			{
				svm_index_after = svm_index;
			}
		}
		var svm_event_index = this.svm_events_end[svm_index].svm_start_position;
		while(true)
		{
			var e = this.svm_events_start[svm_event_index];
			if(e == undefined)
			{
				break;
			}
			if(e.svm_vertical_start>(this.svm_view_vertical+height/this.svm_scale))
			{
				break;
			}
			var svm_marker = this.svm_marker_event.get(e.svm_start_index);
			if(svm_marker != undefined)
			{
				ctx.strokeStyle = svm_marker;
				ctx.fillStyle = svm_marker;
			}
			e.draw(ctx);
			ctx.strokeStyle = this.svm_diagram_color;
			ctx.fillStyle = this.svm_diagram_color;
			ctx.lineWidth = this.svm_event_width;
			svm_event_index++;
		}
		ctx.resetTransform();
		ctx.scale(this.svm_scale,this.svm_scale);
		ctx.clearRect(0,0,width/this.svm_scale,this.svm_vertical_start);
		ctx.strokeStyle = this.svm_decoration_color;
		ctx.fillStyle = this.svm_decoration_color;
		ctx.beginPath();
		ctx.moveTo(0,this.svm_vertical_start-1);
		ctx.lineTo(width/this.svm_scale,this.svm_vertical_start-1);
		ctx.stroke();
		ctx.strokeStyle = this.svm_diagram_color;
		ctx.fillStyle = this.svm_diagram_color;
		ctx.translate(-this.svm_view_horizontal,0);
		for(var [ i, p ] of this.svm_participants)
		{
			p.draw_name(ctx);
		}
		if(this.svm_move)
		{
			ctx.strokeStyle = this.svm_move_color;
			ctx.fillStyle = this.svm_move_color;
			ctx.resetTransform();
			ctx.fillRect(this.svm_move_origin_horizontal-this.svm_move_origin_size,this.svm_move_origin_vertical-this.svm_move_origin_size,this.svm_move_origin_size*2+1,this.svm_move_origin_size*2+1);
			ctx.strokeStyle = this.svm_move_view_color;
			ctx.fillStyle = this.svm_move_view_color;
			ctx.fillRect(this.svm_move_origin_horizontal-this.svm_move_origin_size+1+(2*this.svm_move_origin_size-1)*this.svm_view_horizontal/(this.svm_horizontal_end-this.svm_horizontal_start),this.svm_move_origin_vertical-this.svm_move_origin_size+1+(2*this.svm_move_origin_size-1)*this.svm_view_vertical/(this.svm_vertical_end-this.svm_vertical_start),(2*this.svm_move_origin_size-1)*Math.min(1,(width/this.svm_scale)/(this.svm_horizontal_end-this.svm_horizontal_start)),(2*this.svm_move_origin_size-1)*Math.min(1,(height/this.svm_scale)/(this.svm_vertical_end-this.svm_vertical_start)));
			ctx.strokeStyle = this.svm_move_color;
			ctx.fillStyle = this.svm_move_color;
			ctx.beginPath();
			ctx.moveTo(this.svm_move_origin_horizontal-this.svm_move_origin_size,0);
			ctx.lineTo(this.svm_move_origin_horizontal-this.svm_move_origin_size,height);
			ctx.moveTo(this.svm_move_origin_horizontal+this.svm_move_origin_size,0);
			ctx.lineTo(this.svm_move_origin_horizontal+this.svm_move_origin_size,height);
			ctx.moveTo(0,this.svm_move_origin_vertical-this.svm_move_origin_size);
			ctx.lineTo(width,this.svm_move_origin_vertical-this.svm_move_origin_size);
			ctx.moveTo(0,this.svm_move_origin_vertical+this.svm_move_origin_size);
			ctx.lineTo(width,this.svm_move_origin_vertical+this.svm_move_origin_size);
			ctx.stroke();
			ctx.lineWidth = this.svm_move_arrow_width;
			ctx.translate(this.svm_move_origin_horizontal,this.svm_move_origin_vertical);
			for(var i=0 ; i<8 ; i++)
			{
				ctx.beginPath();
				ctx.moveTo(0,1.5*this.svm_move_origin_size);
				ctx.lineTo(0,3*this.svm_move_origin_size);
				ctx.moveTo(this.svm_move_origin_size/3,2.3*this.svm_move_origin_size);
				ctx.lineTo(0,3*this.svm_move_origin_size);
				ctx.lineTo(-this.svm_move_origin_size/3,2.3*this.svm_move_origin_size);
				ctx.stroke();
				ctx.rotate(Math.PI/4);
			}
		}
	}
};

class SVM_SequenceDiagram_View
{
	constructor(dom,svm_diagram)
	{
		this.dom = dom;
		dom.style.cursor = "pointer";
		this.svm_diagram = svm_diagram;
		this.svm_trace = document.createElement("canvas");
		this.svm_trace.style.display = "none";
		this.svm_trace.width = dom.clientWidth;
		this.svm_trace.height = dom.clientHeight;
		dom.appendChild(this.svm_trace);
		this.svm_display = document.createElement("canvas");
		this.svm_display.width = dom.clientWidth;
		this.svm_display.height = dom.clientHeight;
		dom.appendChild(this.svm_display);
		dom.addEventListener('click',function(event) {
			if(event.ctrlKey)
			{
				this.svm_sequencediagram.toggle_marker(event.layerX,event.layerY);
				return;
			}
			if(this.svm_sequencediagram.toggle_move(event.layerX,event.layerY))
			{
				this.style.cursor = "move";
				this.svm_sequencediagram.set_move_loop();
			}
			else
			{
				this.style.cursor = "pointer";
				this.svm_sequencediagram.reset_move_loop();
			}
			});
		dom.addEventListener('mousemove',function(event) {
			this.svm_sequencediagram.set_move_speed(event.layerX,event.layerY);
			});
	}
	draw()
	{
		this.svm_diagram.draw(this.svm_trace.getContext("2d"),this.svm_trace.width,this.svm_trace.height);
		var temp = this.svm_trace;
		this.svm_trace = this.svm_display;
		this.svm_display = temp;
		this.svm_trace.style.display = "none";
		this.svm_display.style.display = "block";
	}
};

class SVM_SequenceDiagram
{
	constructor(dom,svm_event_list)
	{
		this.svm_diagram = new SVM_SequenceDiagram_Diagram();
		this.svm_view = new SVM_SequenceDiagram_View(dom,this.svm_diagram);
		dom.svm_sequencediagram = this;
		for(var svm_event of svm_event_list)
		{
			this.svm_diagram.add_event(svm_event);
		}
		this.svm_view.draw();
	}
	add_event(svm_event)
	{
		this.svm_diagram.add_event(svm_event);
		this.svm_view.draw();
	}
	increase_scale()
	{
		this.svm_diagram.increase_scale();
		this.svm_view.draw();
	}
	decrease_scale()
	{
		this.svm_diagram.decrease_scale();
		this.svm_view.draw();
	}
	reset_scale()
	{
		this.svm_diagram.reset_scale();
		this.svm_view.draw();
	}
	toggle_move(svm_origin_horizontal,svm_origin_vertical)
	{
		var svm_result = this.svm_diagram.toggle_move(svm_origin_horizontal,svm_origin_vertical);
		this.svm_view.draw();
		return svm_result;
	}
	set_move_speed(svm_move_speed_horizontal,svm_move_speed_vertical)
	{
		this.svm_diagram.set_move_speed(svm_move_speed_horizontal,svm_move_speed_vertical,this.svm_view);
	}
	set_move_loop()
	{
		this.svm_diagram.set_move_loop(this.svm_view);
	}
	reset_move_loop()
	{
		this.svm_diagram.reset_move_loop();
	}
	set_marker_color(color)
	{
		this.svm_diagram.set_marker_color(color);
	}
	reset_markers()
	{
		this.svm_diagram.reset_markers();
		this.svm_view.draw();
	}
	toggle_marker(svm_horizontal,svm_vertical)
	{
		this.svm_diagram.toggle_marker(svm_horizontal,svm_vertical);
		this.svm_view.draw();
	}
};
