/*
 * Simple Virtual Machine - A versatile and robust architecture to
 * easily write applications.
 * Copyright (C) 2021  Julien BRUGUIER
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#pragma once
#include <src/global/global.h>
namespace SVM
{
namespace Global
{
namespace Cache
{
	template<typename Type>
	struct Nettoyage
	{
		void operator() (Type& objet)
		{
			objet.clear();
		}
	};

	template<typename Type, size_t Eviction, typename Nettoyage> 
	struct Cache;

	template<typename Type>
	struct ElementCache;

	template<typename Type, size_t Eviction, typename Nettoyage = Nettoyage<Type> >
	struct Objet
	{
		friend struct Cache<Type,Eviction,Nettoyage>;
		Objet(Objet&& objet) = default;
		~Objet()
		{
			_cache->libere(*this);
		}
		operator Type& () { return _element_cache->_objet; }
		operator const Type& () const { return _element_cache->_objet; }
		Type* operator-> () { return &_element_cache->_objet; }
		const Type* operator-> () const { return &_element_cache->_objet; }
		Type& operator* () { return _element_cache->_objet; }
		const Type& operator* () const { return _element_cache->_objet; }
		private:
			Objet(ElementCache<Type> *element, const std::shared_ptr<Cache<Type,Eviction,Nettoyage> >&cache)
				:_element_cache(element), _cache(cache) {}
			ElementCache<Type>* _element_cache;
			std::shared_ptr<Cache<Type,Eviction,Nettoyage> > _cache;
	};

	template<typename Type>
	struct ElementCache
	{
		explicit ElementCache(const Type& objet)
			:_objet(objet) {}
		~ElementCache()
		{
			delete _suivant;
		}
		public:
			Type _objet;
			ElementCache<Type> *_precedent = nullptr;
			ElementCache<Type> *_suivant = nullptr;
			ElementCache<Type> *_libre = nullptr;
	};

	template<typename Type, size_t Eviction, typename Nettoyage = Nettoyage<Type> >
	struct Cache : public std::enable_shared_from_this<Cache<Type,Eviction,Nettoyage> >
	{
		typedef Objet<Type,Eviction,Nettoyage> TypeObjet;
		Cache() = default;
		Cache(const Cache&) = delete;
		~Cache()
		{
			delete _elements;
		}
		template<typename... Args>
		Objet<Type,Eviction,Nettoyage> reserve(Args&&... args)
		{
			if(_libres == nullptr)
			{
				ElementCache<Type> *nouveau = new ElementCache<Type>(Type(std::forward<Args...>(args)...));
				nouveau->_suivant = _elements;
				if((Eviction>0) and (_elements != nullptr))
				{
					_elements->_precedent = nouveau;
				}
				_elements = nouveau;
				if(Eviction>0) ++_nb_elements;
				//std::cerr << this << " ALLOCATION " << _nb_libres << "/" << _nb_elements << std::endl;
				return Objet<Type,Eviction,Nettoyage>(nouveau,this->shared_from_this());
			}
			ElementCache<Type> *libre = _libres;
			_libres = _libres->_libre;
			if(Eviction>0) --_nb_libres;
			//std::cerr << this << " REUTILISATION " << _nb_libres << "/" << _nb_elements << std::endl;
			return Objet<Type,Eviction,Nettoyage>(libre,this->shared_from_this());
		}
		void libere(const Objet<Type,Eviction,Nettoyage>& objet)
		{
			_nettoyage(objet._element_cache->_objet);
			if((Eviction==0) or (_nb_libres*(1+Eviction)<=_nb_elements))
			{
				objet._element_cache->_libre = _libres;
				_libres = objet._element_cache;
				if(Eviction>0) ++_nb_libres;
				//std::cerr << this << " LIBERATION " << _nb_libres << "/" << _nb_elements << std::endl;
			}
			else
			{
				if(objet._element_cache->_precedent != nullptr)
				{
					objet._element_cache->_precedent->_suivant = objet._element_cache->_suivant;
				}
				else
				{
					_elements = objet._element_cache->_suivant;
				}
				if(objet._element_cache->_suivant != nullptr)
				{
					objet._element_cache->_suivant->_precedent = objet._element_cache->_precedent;
				}
				objet._element_cache->_precedent = nullptr;
				objet._element_cache->_suivant = nullptr;
				delete objet._element_cache;
				if(Eviction>0) --_nb_elements;
				//std::cerr << this << " SUPPRESSION " << _nb_libres << "/" << _nb_elements << std::endl;
			}
		}
		private:
			ElementCache<Type> *_elements = nullptr;
			ElementCache<Type> *_libres = nullptr;
			size_t _nb_elements = 0;
			size_t _nb_libres = 0;
			Nettoyage _nettoyage;
	};
}
}
}
