.TH "Simple Virtual Machine - Plugin array" 7 "2023-11-15"
.SH SYNOPSIS
.B svm_plugin_array
Arrays enhancement
.SH DESCRIPTION
The virtual machine is able to manage arrays represented by pointeurs in a basic way.
.P
This plugin enhances array management.
.SH INVOKATION
This plugin can be added to the Simple Virtual Machine using one of these directives:
.SS Local
.nf

LOCAL PLUGIN "svmpluginarray/libsvmarray.so"

.fi
when compiled/installed locally and should be accessible from the application directory.
.SS Machine and System
.nf

PLUGIN "===PLUGINLIB==="

.fi
when installed with auto-tools and should be accessible from the Simple Virtual Machine installation directory or from system directories. (Recommended)
.SS Absolute path
.nf

PLUGIN "===PLUGINLIBDIR===/===PLUGINLIB==="

.fi
when installed with auto-tools and should be accessible from an absolute path.
.SH CONTENT
.SS OPTION array.wrap -w BLN
When this option is active, arrays indexes are always valid and wrap around array boundaries.
.P
In particular, -1 indicates the position after the last element, -2 the last element...
.SS INTERRUPTION array.out_of_range
This interruption is raised when an index would indicate a position outside an array.
.SS INSTRUCTION array.append MUTABLE PTR:array ( INT:size | { VALUE + }:values ) -> PTR
This instruction appends:
 - either empty addresses at the end of an array,
 - or values at the end of an array.

Then, it updates the pointer to the array.
.P
If the operation can not be done in-place, the array is moved to a new location.
.P
The instruction returns the pointer corresponding to the new addresses.
.SS INSTRUCTION array.insert MUTABLE PTR:array INT:index ( INT:size | { VALUE + }:values ) -> PTR
This instruction inserts elements into an array at the specified index.
When the index is invalid, an !array.out_of_range can be raised.
When the size is negative, a FAILURE interruption will be raised.
.P
The array is always relocated.
.P
When the values are not specified, the inserted addresses are left empty.
.P
The instruction returns the pointer corresponding to the new addresses.
.SS INSTRUCTION array.erase MUTABLE PTR:array INT:index INT:size
This instruction erases elements from an array at the specified index.
When the index is invalid, an !array.out_of_range can be raised.
The array is always relocated, unless the deleted size is 0.
.SS INSTRUCTION array.sub PTR:array INT:index INT:size -> PTR
This instruction returns a pointer referencing a sub-part of the array.
.P
When the index and the size are invalid, an !array.out_of_range interruption will be raised.
.P
When the size is negative, a FAILURE interruption will be raised.
.SS INSTRUCTION array.copy PTR:array -> PTR:copy
This instruction copies an array.
.SS INSTRUCTION array.print PTR:array -> STR
This instruction generates a string from an array.
.SS INSTRUCTION array.update PTR:array ( PEP:function INT:element ? { . * } | [ STR SYM ]:function { VALUE * } )
This instruction applies a function to each element of an array. The array is modified in place.
.P
The function can be:
 - A plugin function when the function is passed as a PEP: The array element is passed by default as first parameter in the function and is mutable, but the position of the element can be specified to another index,
 - A code function when the function is passed as a STR or a SYM: The array element is passed as second parameter - aka (P/1) - in the function, and (P/0) is set to a pointer to store local values.
.P
The values between { and } are extra parameters passed to the function.
.P
The instruction just launches the functions, and does not care about checking for eventual errors. It is up to the user to set correct interruption handlers to manage errors.
.SS INTERRUPTION array.invalid_return
This interruption is raised when a plugin function returns an invalid value.
.SS INSTRUCTION array.map PTR:array ( PEP:function INT:element ? { . * } | [ STR SYM ]:function { VALUE * } ) -> PTR
This instruction applies a function to each element of an array, and creates an array with the resulting values.
.P
The function can be:
 - A plugin function when the function is passed as a PEP: The array element is passed by default as first parameter in the function, but the position of the element can be specified to another index,
 - A code function when the function is passed as a STR or a SYM: The array element is passed as second parameter - aka (P/1) - in the function, and (P/0) is set to a pointer to store local values.
.P
The values between { and } are extra parameters passed to the function.
.P
The instruction just launches the functions, and does not care about checking for eventual errors. It is up to the user to set correct interruption handlers to manage errors.
.SS INSTRUCTION array.accumulate PTR:array ( PEP:function ( INT:element INT:accumulator ) ? { . * } VALUE:init | [ STR SYM ]:function { VALUE * } PTR:accumulator ) -> [ VALUE PTR ]
This instruction accumulates elements from an array into a value.
.P
This instruction has two execution modes:
 - With a plugin function: The initial value is passed as last parameter, and the accumulated value is returned by the instruction. Within the function, the array element is passed by default as first parameter, and the accumulator as second parameter. When the element and the accumulator indexes are specified, element and accumulator are placed at these indexes.
 - With a code function: The accumulator is passed as a pointer referencing the accumulated values. This pointer shall be initialised before the instruction call, and the result will be referenced by this pointer, returned by the instruction. Within the function, the first parameter is pointer usable for local allocations, the second parameter - (P/1) - is the array element, the third parameter - (P/2) - is the pointer to the accumulator.
.P
In both cases, parameters between { and } are extra parameters.
.P
The instruction raises an !array.invalid_return interruption if the plugin function does not return a value.
.SS INSTRUCTION array.filter PTR:array PEP:predicate INT:index ? { . * } -> PTR
This instruction applies a predicate on each array element, and copies in a new array elements passing the test of the predicate.
.P
Within the plugin function, the array element is passed by default as first parameter. Parameters between { and } are passed as extra parameters to the predicate function. When the index is specified, the element is put at this index.
.P
The instruction raises an !array.invalid_return interruption if the plugin function does not return a boolean value.
.SS INSTRUCTION array.sort PTR:array PEP:comparator ( INT:left INT:right ) ? { . * } -> PTR
This instruction applies a sort algorithm using a plugin function as comparator, and creates a new array with sorted values.
.P
Within the plugin function, the two array elements are passed by default as first and second parameter. Parameters between { and } are passed as extra parameters to the comparator function. When the left and the right index are specified, the left value and the right values will be placed at these index on the comparator parameters.
.P
The instruction raises an !array.invalid_return interruption if the plugin function does not return a boolean value.
.SH EXAMPLES
.SS Basic handling
.nf
#!/usr/bin/env svm
LOG
PLUGIN "===PLUGINLIB==="
PLUGIN "svmint.so"
PLUGIN "svmcom.so"
PROCESS "basic"
	CODE "main" INLINE
		:memory PTR/array, STR/s
		:memory INT*5 -> &array
		[ 0 , 1 , 2 , 3 , 4 ] -> @&array
		:array.append @&array { -1 -2 }
		:array.print @&array -> &s
		:com.message @&s
		:array.insert @&array 2 { 10 11 12 }                                                                                                                    
		:array.print @&array -> &s                                                                                                                              
		:com.message @&s                                                                                                                                        
		:array.erase @&array 4 3                                                                                                                                
		:array.print @&array -> &s                                                                                                                              
		:com.message @&s                      
		:array.update @&array "inc" { 100 }
		:array.print @&array -> &s
		:com.message @&s
		:shutdown
	:label inc
		:shift @(P/2) (P/1)
		:return
	END
END
.fi
.SS Transformations
.nf
#!/usr/bin/env svm
LOG
PLUGIN "===PLUGINLIB==="
PLUGIN "svmpluginop/libsvmop.so"
PLUGIN "svmcom.so"
PLUGIN "svmint.so"
PROCESS "transform"
	CODE "main" INLINE
		:memory PTR/array, STR/s
		:memory INT*5 -> &array
		[ 0 , 1 , 2 , 3 , 4 ] -> @&array
		:memory PTR/map, PTR/hypermap
		:array.map @&array op.add { 10 } -> &map
		:array.print @&map -> &s
		:com.message @&s
		:array.map @&array "incsqr" { 10 } -> &hypermap
		:array.print @&hypermap -> &s
		:com.message @&s
		:memory INT/sum, INT/product
		:array.accumulate @&array op.add { } 0 -> &sum
		:com.message @&sum
		[ 1 ] -> product
		:array.accumulate @&map "prod" { } product
		:com.message @&product
		:memory PTR/filter
		:array.filter @&hypermap op.odd { } -> &filter
		:array.print @&filter -> &s
		:com.message @&s
		:memory PTR/remap, PTR/sorted
		:array.map @&hypermap "selectdec" { 100 } -> &remap
		:array.sort @&remap op.cmp 0 2 { < } -> &sorted
		:array.print @&sorted -> &s
		:com.message @&s
		:shutdown
	:label incsqr
		:shift @(P/3) (P/1)
		:int.mul @(P/1) @(P/1) -> (P/2)
		:return
	:label prod
		:int.mul @&@(P/2) @(P/1) -> &@(P/2)
		:return
	:label selectdec
		@(P/1) -> (P/2)
		:memory BLN, INT -> &P
		:int.mod @(P/1) 2 -> (@&P/1)
		:int.cmp @(@&P/1) = 0 -> &@&P
		:return :unless @&@&P TRUE
		:int.sub @(P/1) @(P/3) -> (P/2)
		:return
	END
END
.fi
.P
This application uses this little plugin:
.nf
PLUGIN op

DEFINE

FUNCTION op.add INT INT -> INT
:{
	auto l = ARGV_VALUE(0,integer);
	auto r = ARGV_VALUE(1,integer);
	return NEW_VALUE(integer,l+r);
:}

code:
:{
template<typename T>
bool comp(const T& l, const T& r, const std::string& c)
{
	if(c=="<")
		return l<r;
	if(c=="<=")
		return l<=r;
	if(c=="=")
		return l==r;
	if(c=="=>")
		return l>=r;
	if(c==">")
		return l>r;
	if(c=="<>")
		return l!=r;
}
:}

FUNCTION op.cmp INT [ < <= = => > <> ] INT -> BLN
:{
	auto l = ARGV_VALUE(0,integer);
	auto c = ARGV_MARKER(1);
	auto r = ARGV_VALUE(2,integer);
	return NEW_VALUE(boolean,comp(l,r,c)?TRUE:FALSE);
:}

FUNCTION op.odd INT -> BLN
:{
	auto i = ARGV_VALUE(0,integer);
	return NEW_VALUE(boolean,(i%2==1)?TRUE:FALSE);
:}
.fi
.SH SEE ALSO
.BR svm (1)
for help on the Simple Virtual Machine.
.SH AUTHOR
This plugin has been written by Julien BRUGUIER and is maintained by Julien BRUGUIER <projet.svm@pappy.tf>.
