/*
 * Copyright (C) 2023-03-13  Julien BRUGUIER
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */


#include <string>
#include <sstream>

#line 75 "../plugins/bit/core/bit.svm_plugin"
#include <vector>
#include <string>
#line 26 "src/plugin.cpp"

#include <src/plugin.h>

extern "C"
{

void plugin_configure(void *plugin)
{
	::svm_plugin_configure(plugin,
		"PLUGIN bit \n"
		"DEFINE \n"
		"	TYPE bit.set \n"
		"	INTERRUPTION bit.size_mismatch \n"
		"	INTERRUPTION bit.out_of_range \n"
		"	INSTRUCTION bit.set [ INT STR ]:size_or_bits -> bit.set \n"
		"	INSTRUCTION bit.string bit.set -> STR \n"
		"	INSTRUCTION bit.format bit.set STR:false STR:true ( INT:block STR:separator ) ? -> STR \n"
		"	INSTRUCTION bit.check bit.set INT:index -> BLN \n"
		"	INSTRUCTION bit.modify MUTABLE bit.set INT:index [ BLN 'SWAP' ]:value \n"
		"	INSTRUCTION bit.pattern MUTABLE bit.set ( PTR | BLN + ) \n"
		"	INSTRUCTION bit.size bit.set -> INT \n"
		"	INSTRUCTION bit.count bit.set -> INT \n"
		"	INSTRUCTION bit.empty bit.set -> BLN \n"
		"	INSTRUCTION bit.index bit.set -> PTR \n"
		"	INSTRUCTION bit.shift bit.set INT:shift -> bit.set \n"
		"	INSTRUCTION bit.rotate bit.set INT:shift -> bit.set \n"
		"	INSTRUCTION bit.not bit.set -> bit.set \n"
		"	INSTRUCTION bit.all bit.set bit.set + -> bit.set \n"
		"	INSTRUCTION bit.any bit.set bit.set + -> bit.set \n"
		"	INSTRUCTION bit.operation [ = <> < <= > => ]:operation INT:threshold ( PTR | bit.set bit.set + ) -> bit.set \n"
		"	INSTRUCTION bit.cmp bit.set [ = <> ] bit.set -> BLN \n"
		"	FUNCTION bit.map MUTABLE bit.set PEP:function . *:parameters \n"
		"	INSTRUCTION bit.map MUTABLE bit.set PEP:function . *:parameters \n"
		,SVM_API_SIGNATURE,SVM_VERSION);
}

#ifndef ARGV_VALUE
#define ARGV_VALUE(index,type) ::svm_value_##type##_get(svm,::svm_parameter_value_get(svm,argv[(index)]))
#endif
#ifndef ARGV_PLUGIN
#define ARGV_PLUGIN(index,plugin,name) reinterpret_cast<type_##name*>(::svm_value_plugin_get_internal(svm,::svm_parameter_value_get(svm,argv[(index)])))
#endif
#ifndef ARGV_MARKER
#define ARGV_MARKER(index) std::string(::svm_parameter_marker_get(svm,argv[(index)]).string)
#endif
#ifndef ARGV_KEYWORD
#define ARGV_KEYWORD(index) std::string(::svm_parameter_keyword_get(svm,argv[(index)]).string)
#endif
#ifndef ARGV_STRUCT
#define ARGV_STRUCT(index,plugin,name) reinterpret_cast<struct_##name*>(::svm_structure_get_internal(svm,::svm_value_pluginentrypoint_new__raw(svm,#plugin,#name),::svm_parameter_structure_get(svm,argv[(index)])))
#endif
#ifndef ARGV_VARIABLE
#define ARGV_VARIABLE(index) ::svm_parameter_variable_get(svm,argv[(index)]);
#endif
#ifndef NEW_VALUE
#define NEW_VALUE(type,value) ::svm_value_##type##_new(svm,(value))
#endif
#ifndef NEW_PLUGIN
#define NEW_PLUGIN(plugin,name,value) ::svm_value_plugin_new(svm,::svm_value_pluginentrypoint_new__raw(svm,#plugin,#name),(value))
#endif
#ifndef NEW_STRUCT
#define NEW_STRUCT(plugin,name,value) ::svm_structure_new(svm,::svm_value_pluginentrypoint_new__raw(svm,#plugin,#name),(value))
#endif
#ifndef NEW_STRING
#define NEW_STRING(raw_string) ::svm_string_new(svm,raw_string.c_str(),raw_string.size())
#endif
#ifndef RAW_STRING
#define RAW_STRING(svm_string) std::string(svm_string.string,svm_string.size)
#endif
#ifndef NEW_BOOLEAN
#define NEW_BOOLEAN(boolean) ((boolean)?TRUE:FALSE)
#endif
#ifndef RAW_BOOLEAN
#define RAW_BOOLEAN(boolean) ((boolean)==TRUE)
#endif
#ifndef NEW_NULL_VALUE
#define NEW_NULL_VALUE(type) ::svm_value_##type##_new_null(svm)
#endif
#ifndef NEW_NULL_PLUGIN
#define NEW_NULL_PLUGIN(plugin,name) ::svm_value_plugin_new_null(svm,::svm_value_pluginentrypoint_new__raw(svm,#plugin,#name))
#endif
#ifndef NEW_NULL_STRUCT
#define NEW_NULL_STRUCT(plugin,name) ::svm_structure_new_null(svm,::svm_value_pluginentrypoint_new__raw(svm,#plugin,#name))
#endif
#ifndef ERROR_INTERNAL
#define ERROR_INTERNAL(irq,message) ::svm_processor_current_raise_error_internal__raw(svm,irq,message)
#endif
#ifndef ERROR_EXTERNAL
#define ERROR_EXTERNAL(plugin,name,message) ::svm_processor_current_raise_error_external__raw(svm,::svm_value_pluginentrypoint_new__raw(svm,#plugin,#name),message)
#endif
#ifndef CONST_PEP
#define CONST_PEP(plugin,name) ::svm_value_pluginentrypoint_new__raw(svm,#plugin,#name)
#endif
#ifndef CURRENT
#define CURRENT(object) ::svm_##object##_get_current(svm)
#endif
#ifndef RETURN
#define RETURN return nullptr
#endif
#ifndef VARIABLE_GLOBAL
#define VARIABLE_GLOBAL(variable) ::svm_variable_scope_set_global(svm,(variable))
#endif
#ifndef VARIABLE_LOCAL
#define VARIABLE_LOCAL(variable) ::svm_variable_scope_set_local(svm,(variable))
#endif
#ifndef VARIABLE_DELETE
#define VARIABLE_DELETE(variable) ::svm_variable_delete(svm,(variable))
#endif


/* TYPE bit.set */

struct type_set
{
#line 83 "../plugins/bit/core/bit.svm_plugin"
	size_t _size;
	std::vector<unsigned char> _bits;
	type_set(const size_t s)
	:_size(s), _bits(s/8+((s%8)?1:0),'\000') {}
	type_set(const type_set& s)
	:_size(s._size),_bits(s._bits) {}
	explicit type_set(const std::string& s)
	:_size(s.size()*8),_bits(s.begin(),s.end()) {}
	std::string format(const std::string& no, const std::string& yes, const size_t block, const std::string& separator) const
	{
		std::string s;
		for(size_t i=0 ; i<_size ; ++i)
		{
			if((block>0) and ((_size-i)%block==0) and (i>0) and (i<(_size-1)))
			{
				s.push_back(separator[0]);
			}
			if(_bits[i/8] bitand (1<<(7-i%8)))
			{
				s.push_back(yes[0]);
			}
			else
			{
				s.push_back(no[0]);
			}
		}
		return s;
	}
	operator std::string () const
	{
		return format(".","X",8," ");
	}
	std::string string() const
	{
		std::string s;
		for(const auto& c: _bits)
		{
			s.push_back(c);
		}
		return s;
	}
#line 183 "src/plugin.cpp"
};

void type_set_delete(const void *svm, void *handler)
{
	type_set * const object = reinterpret_cast<type_set*>(handler);
	{
#line 126 "../plugins/bit/core/bit.svm_plugin"

#line 192 "src/plugin.cpp"
	}
	delete object;
}

void* type_set_copy(const void *svm, const void *handler)
{
	const type_set *object = reinterpret_cast<const type_set*>(handler);
	type_set *copy = new type_set(*object);
	{
#line 127 "../plugins/bit/core/bit.svm_plugin"

#line 204 "src/plugin.cpp"
	}
	return copy;
}

void* type_set_constant(const void *svm, const SVM_String value)
{
	std::string string(value.string,value.size);
	type_set *object = new type_set(string);
	{
#line 128 "../plugins/bit/core/bit.svm_plugin"

#line 216 "src/plugin.cpp"
	}
	return object;
}

SVM_String type_set_print(const void *svm, const void *handler)
{
	const type_set *object = reinterpret_cast<const type_set*>(handler);
	std::string string = static_cast<std::string>(*object);
	{
#line 129 "../plugins/bit/core/bit.svm_plugin"

#line 228 "src/plugin.cpp"
	}
	return ::svm_string_new(svm,string.c_str(),string.size());
}


/* INTERRUPTION bit.size_mismatch */


/* INTERRUPTION bit.out_of_range */


/* INSTRUCTION bit.set [ INT STR ]:size_or_bits -> bit.set */

SVM_Value instruction_set(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 149 "../plugins/bit/core/bit.svm_plugin"
	SVM_Value value = ::svm_parameter_value_get(svm,argv[0]);
	if(::svm_value_type_is_integer(svm,value))
	{
		auto size = ::svm_value_integer_get(svm,value);
		if(size<0)
		{
			ERROR_INTERNAL(FAILURE,"Invalid size");
		}
		auto set = new type_set(size);
		return NEW_PLUGIN(bit,set,set);
	}
	else
	{
		SVM_String bit = ::svm_value_string_get(svm,value);
		auto set = new type_set(RAW_STRING(bit));
		return NEW_PLUGIN(bit,set,set);
	}
#line 262 "src/plugin.cpp"
}


/* INSTRUCTION bit.string bit.set -> STR */

SVM_Value instruction_string(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 179 "../plugins/bit/core/bit.svm_plugin"
	auto set = ARGV_PLUGIN(0,bit,set);
	if((set->_size%8)!=0)
	{
		ERROR_EXTERNAL(bit,size_mismatch,"Bit set size is not a multiple of 8.");
	}
	auto s = set->string();
	return NEW_VALUE(string,NEW_STRING(s));
#line 278 "src/plugin.cpp"
}


/* INSTRUCTION bit.format bit.set STR:false STR:true ( INT:block STR:separator ) ? -> STR */

SVM_Value instruction_format(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 195 "../plugins/bit/core/bit.svm_plugin"
	auto set = ARGV_PLUGIN(0,bit,set);
	auto f = ARGV_VALUE(1,string);
	auto t = ARGV_VALUE(2,string);
	if(f.size==0)
	{
		ERROR_INTERNAL(FAILURE,"Empty false string");
	}
	if(t.size==0)
	{
		ERROR_INTERNAL(FAILURE,"Empty true string");
	}
	size_t block = 0;
	std::string separator(" ");
	if(argc>3)
	{
		auto b = ARGV_VALUE(3,integer);
		auto s = ARGV_VALUE(4,string);
		if(b<1)
		{
			ERROR_INTERNAL(FAILURE,"Invalid block size");
		}
		if(s.size==0)
		{
			ERROR_INTERNAL(FAILURE,"Empty separator string");
		}
		block = b;
		separator = RAW_STRING(s);
	}
	std::string s = set->format(RAW_STRING(f),RAW_STRING(t),block,separator);
	return NEW_VALUE(string,NEW_STRING(s));
#line 317 "src/plugin.cpp"
}


/* INSTRUCTION bit.check bit.set INT:index -> BLN */

SVM_Value instruction_check(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 239 "../plugins/bit/core/bit.svm_plugin"
	auto set = ARGV_PLUGIN(0,bit,set);
	auto index = ARGV_VALUE(1,integer);
	if((index<0) or (index>=set->_size))
	{
		ERROR_EXTERNAL(bit,out_of_range,"Index out of range");
	}
	index = set->_size-index-1;
	unsigned char result = set->_bits[index/8] bitand (1<<(7-index%8));
	return NEW_VALUE(boolean,result?TRUE:FALSE);
#line 335 "src/plugin.cpp"
}


/* INSTRUCTION bit.modify MUTABLE bit.set INT:index [ BLN 'SWAP' ]:value */

SVM_Value instruction_modify(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 259 "../plugins/bit/core/bit.svm_plugin"
	auto set = ARGV_PLUGIN(0,bit,set);
	auto index = ARGV_VALUE(1,integer);
	if((index<0) or (index>=set->_size))
	{
		ERROR_EXTERNAL(bit,out_of_range,"Index out of range");
	}
	index = set->_size-index-1;
	unsigned char mask = 1<< (7-index%8);
	if(::svm_parameter_type_is_keyword(svm,argv[2]))
	{
		set->_bits[index/8] ^= mask;
	}
	else
	{
		SVM_Boolean value = ARGV_VALUE(2,boolean);
		if(value==TRUE)
		{
			set->_bits[index/8] |= mask;
		}
		else
		{
			set->_bits[index/8] &= ~mask;
		}
	}
#line 368 "src/plugin.cpp"
	return nullptr;
}


/* INSTRUCTION bit.pattern MUTABLE bit.set ( PTR | BLN + ) */

SVM_Value instruction_pattern(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 295 "../plugins/bit/core/bit.svm_plugin"
	auto set = ARGV_PLUGIN(0,bit,set);
	std::vector<bool> pattern;
	SVM_Value v = ::svm_parameter_value_get(svm,argv[1]);
	if(::svm_value_type_is_pointer(svm,v))
	{
		SVM_Address a = ::svm_value_pointer_get_address(svm,v);
		SVM_Size s = ::svm_value_pointer_get_size(svm,v);
		if(s==0)
		{
			ERROR_INTERNAL(FAILURE,"Invalid pattern size");
		}
		for(SVM_Address it=a ; it<(a+s) ; ++it)
		{
			SVM_Value_Boolean b = ::svm_memory_read_address_type_internal(svm,CURRENT(kernel),it,BOOLEAN);
			pattern.push_back(::svm_value_boolean_get(svm,b)==TRUE);
		}
	}
	else
	{
		for(SVM_Index i=1 ; i<argc ; ++i)
		{
			pattern.push_back(ARGV_VALUE(i,boolean)==TRUE);
		}
	}
	size_t p = pattern.size()-1;
	for(size_t i=0 ; i<set->_size ; ++i)
	{
		auto index = set->_size-i-1;
		unsigned char mask = 1<< (7-index%8);
		if(pattern[p])
		{
			set->_bits[index/8] |= mask;
		}
		else
		{
			set->_bits[index/8] &= ~mask;
		}
		if(p==0)
		{
			p = pattern.size()-1;
		}
		else
		{
			--p;
		}
	}
#line 424 "src/plugin.cpp"
	return nullptr;
}


/* INSTRUCTION bit.size bit.set -> INT */

SVM_Value instruction_size(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 365 "../plugins/bit/core/bit.svm_plugin"
	auto set = ARGV_PLUGIN(0,bit,set);
	return NEW_VALUE(integer,set->_size);
#line 436 "src/plugin.cpp"
}


/* INSTRUCTION bit.count bit.set -> INT */

SVM_Value instruction_count(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 375 "../plugins/bit/core/bit.svm_plugin"
	auto set = ARGV_PLUGIN(0,bit,set);
	size_t nb=0;
	for(size_t i=0 ; i<set->_size ; ++i)
	{
		auto index = set->_size-i-1;
		unsigned char mask = 1<< (7-index%8);
		if(set->_bits[index/8] bitand mask)
		{
			++nb;
		}
	}
	return NEW_VALUE(integer,nb);
#line 457 "src/plugin.cpp"
}


/* INSTRUCTION bit.empty bit.set -> BLN */

SVM_Value instruction_empty(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 395 "../plugins/bit/core/bit.svm_plugin"
	auto set = ARGV_PLUGIN(0,bit,set);
	size_t nb=0;
	for(size_t i=0 ; i<set->_size ; ++i)
	{
		auto index = set->_size-i-1;
		unsigned char mask = 1<< (7-index%8);
		if(set->_bits[index/8] bitand mask)
		{
			++nb;
		}
	}
	return NEW_VALUE(boolean,(nb==0)?TRUE:FALSE);
#line 478 "src/plugin.cpp"
}


/* INSTRUCTION bit.index bit.set -> PTR */

SVM_Value instruction_index(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 415 "../plugins/bit/core/bit.svm_plugin"
	auto set = ARGV_PLUGIN(0,bit,set);
	std::vector<size_t> positions;
	for(size_t i=0 ; i<set->_size ; ++i)
	{
		auto index = set->_size-i-1;
		unsigned char mask = 1<< (7-index%8);
		if(set->_bits[index/8] bitand mask)
		{
			positions.push_back(i);
		}
	}
	SVM_Memory_Zone zone = ::svm_memory_zone_new(svm);
	::svm_memory_zone_append_internal__raw(svm,zone,INTEGER,positions.size());
	SVM_Value_Pointer p = ::svm_memory_allocate(svm,CURRENT(kernel),zone);
	SVM_Address a = ::svm_value_pointer_get_address(svm,p);
	SVM_Size s = ::svm_value_pointer_get_size(svm,p);
	auto itp = positions.begin();
	for(SVM_Address it=a ; it<(a+s) ; ++it)
	{
		::svm_memory_write_address(svm,CURRENT(kernel),it,::svm_value_integer_new(svm,*(itp++)));
	}
	return p;
#line 509 "src/plugin.cpp"
}


/* INSTRUCTION bit.shift bit.set INT:shift -> bit.set */

SVM_Value instruction_shift(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 447 "../plugins/bit/core/bit.svm_plugin"
	auto set = ARGV_PLUGIN(0,bit,set);
	auto shift = ARGV_VALUE(1,integer);
	std::vector<size_t> yes;
	for(size_t i=0 ; i<set->_size ; ++i)
	{
		auto index = set->_size-i-1;
		unsigned char mask = 1<< (7-index%8);
		if(set->_bits[index/8] bitand mask)
		{
			if(((i+shift)>=0) and ((i+shift)<set->_size))
			{
				yes.push_back(i+shift);
			}
		}
	}
	auto result = new type_set(set->_size);
	for(const auto& y: yes)
	{
		auto index = result->_size-y-1;
		unsigned char mask = 1<< (7-index%8);
		result->_bits[index/8] |= mask;
	}
	return NEW_PLUGIN(bit,set,result);
#line 541 "src/plugin.cpp"
}


/* INSTRUCTION bit.rotate bit.set INT:shift -> bit.set */

SVM_Value instruction_rotate(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 482 "../plugins/bit/core/bit.svm_plugin"
	auto set = ARGV_PLUGIN(0,bit,set);
	auto shift = ARGV_VALUE(1,integer);
	std::vector<size_t> yes;
	for(size_t i=0 ; i<set->_size ; ++i)
	{
		auto index = set->_size-i-1;
		unsigned char mask = 1<< (7-index%8);
		if(set->_bits[index/8] bitand mask)
		{
			auto n = (i+shift)%set->_size;
			if(n<0) { n += set->_size; }
			yes.push_back(n);
		}
	}
	auto result = new type_set(set->_size);
	for(const auto& y: yes)
	{
		auto index = result->_size-y-1;
		unsigned char mask = 1<< (7-index%8);
		result->_bits[index/8] |= mask;
	}
	return NEW_PLUGIN(bit,set,result);
#line 572 "src/plugin.cpp"
}


/* INSTRUCTION bit.not bit.set -> bit.set */

SVM_Value instruction_not(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 516 "../plugins/bit/core/bit.svm_plugin"
	auto set = ARGV_PLUGIN(0,bit,set);
	auto result = new type_set(*set);
	for(size_t c=0 ; c<result->_bits.size() ; ++c)
	{
		result->_bits[c] = ~result->_bits[c];
	}
	
	return NEW_PLUGIN(bit,set,result);
#line 589 "src/plugin.cpp"
}


/* INSTRUCTION bit.all bit.set bit.set + -> bit.set */

SVM_Value instruction_all(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 532 "../plugins/bit/core/bit.svm_plugin"
	auto set = ARGV_PLUGIN(0,bit,set);
	auto result = new type_set(*set);
	for(SVM_Index i=1 ; i<argc ; ++i)
	{
		auto s = ARGV_PLUGIN(i,bit,set);
		if(s->_size!=result->_size)
		{
			ERROR_EXTERNAL(bit,size_mismatch,"Incompatible sizes between sets.");
		}
		for(size_t c=0 ; c<result->_bits.size() ; ++c)
		{
			result->_bits[c] &= s->_bits[c];
		}
	}
	return NEW_PLUGIN(bit,set,result);
#line 613 "src/plugin.cpp"
}


/* INSTRUCTION bit.any bit.set bit.set + -> bit.set */

SVM_Value instruction_any(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 558 "../plugins/bit/core/bit.svm_plugin"
	auto set = ARGV_PLUGIN(0,bit,set);
	auto result = new type_set(*set);
	for(SVM_Index i=1 ; i<argc ; ++i)
	{
		auto s = ARGV_PLUGIN(i,bit,set);
		if(s->_size!=result->_size)
		{
			ERROR_EXTERNAL(bit,size_mismatch,"Incompatible sizes between sets.");
		}
		for(size_t c=0 ; c<result->_bits.size() ; ++c)
		{
			result->_bits[c] |= s->_bits[c];
		}
	}
	return NEW_PLUGIN(bit,set,result);
#line 637 "src/plugin.cpp"
}


/* INSTRUCTION bit.operation [ = <> < <= > => ]:operation INT:threshold ( PTR | bit.set bit.set + ) -> bit.set */

SVM_Value instruction_operation(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 584 "../plugins/bit/core/bit.svm_plugin"
	auto operation = ARGV_MARKER(0);
	auto threshold = ARGV_VALUE(1,integer);
	std::vector<type_set*> sets;
	SVM_Value v = ::svm_parameter_value_get(svm,argv[2]);
	if(::svm_value_type_is_pointer(svm,v))
	{
		SVM_Address a = ::svm_value_pointer_get_address(svm,v);
		SVM_Size s = ::svm_value_pointer_get_size(svm,v);
		if(s<2)
		{
			ERROR_INTERNAL(FAILURE,"Invalid pointer size");
		}
		for(SVM_Address it=a ; it<(a+s) ; ++it)
		{
			SVM_Value_Plugin s = ::svm_memory_read_address_type_external(svm,CURRENT(kernel),it,CONST_PEP(bit,set));
			sets.push_back(reinterpret_cast<type_set*>(::svm_value_plugin_get_internal(svm,s)));
		}
	}
	else
	{
		for(SVM_Index s = 2 ; s<argc ; ++s)
		{
			sets.push_back(ARGV_PLUGIN(s,bit,set));
		}
	}
	size_t size = sets.front()->_size;
	for(const auto& s: sets)
	{
		if(s->_size!=size)
		{
			ERROR_EXTERNAL(bit,size_mismatch,"Incompatible sizes between sets.");
		}
	}
	std::vector<size_t> counters(size,0);
	for(const auto& s: sets)
	{
		for(size_t i=0 ; i<s->_size ; ++i)
		{
			auto index = s->_size-i-1;
			unsigned char mask = 1<< (7-index%8);
			counters[i] += (s->_bits[index/8] bitand mask)?1:0;
		}
	}
	auto result = new type_set(size);
	auto it=counters.begin();
	for(size_t i=0 ; i<result->_size ; ++i)
	{
		size_t value = *(it++);
		auto index = result->_size-i-1;
		unsigned char mask = 1<< (7-index%8);
		bool r = false;
		if(operation=="=")
		{
			r = value==threshold;
		}
		else if(operation=="<>")
		{
			r = value!=threshold;
		}
		else if(operation=="<=")
		{
			r = value<=threshold;
		}
		else if(operation=="<")
		{
			r = value<threshold;
		}
		else if(operation=="=>")
		{
			r = value>=threshold;
		}
		else if(operation==">")
		{
			r = value>threshold;
		}
		if(r)
		{
			result->_bits[index/8] |= mask;
		}
		else
		{
			result->_bits[index/8] &= ~mask;
		}
	}
	return NEW_PLUGIN(bit,set,result);
#line 731 "src/plugin.cpp"
}


/* INSTRUCTION bit.cmp bit.set [ = <> ] bit.set -> BLN */

SVM_Value instruction_cmp(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 707 "../plugins/bit/core/bit.svm_plugin"
	auto left = ARGV_PLUGIN(0,bit,set);
	auto op = ARGV_MARKER(1);
	auto right = ARGV_PLUGIN(2,bit,set);
	if(left->_size!=right->_size)
	{
		ERROR_EXTERNAL(bit,size_mismatch,"Incompatible sizes between sets.");
	}
	bool equal = true;
	for(size_t i=left->_size ; i>0 ; --i)
	{
		auto index = left->_size-(i-1)-1;
		unsigned char mask = 1<< (7-index%8);
		if(((left->_bits[index/8] bitand mask)>0) != ((right->_bits[index/8] bitand mask)>0))
		{
			equal = false;
			break;
		}
	}
	return NEW_VALUE(boolean,(((op=="=") and equal) or ((op=="<>") and not equal))?TRUE:FALSE);
#line 759 "src/plugin.cpp"
}


/* FUNCTION bit.map MUTABLE bit.set PEP:function . *:parameters */

SVM_Variable function_map(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 737 "../plugins/bit/core/bit.svm_plugin"
	auto set = ARGV_PLUGIN(0,bit,set);
	SVM_Value_PluginEntryPoint func = ::svm_parameter_value_get(svm,argv[1]);
	argv[0] = ::svm_parameter_value_new(svm,NEW_NULL_VALUE(integer));
	argv[1] = ::svm_parameter_value_new(svm,NEW_NULL_VALUE(boolean));
	if(not ::svm_plugin_has_function(svm,func,argc,argv,::svm_parameter_value_new(svm,NEW_VALUE(boolean,TRUE))))
	{
		ERROR_INTERNAL(FAILURE,"Invalid function");
	}
	for(size_t i=0 ; i<set->_size ; ++i)
	{
		auto index = set->_size-i-1;
		unsigned char mask = 1<< (7-index%8);
		argv[0] = ::svm_parameter_value_new(svm,NEW_VALUE(integer,i));
		argv[1] = ::svm_parameter_value_new(svm,NEW_VALUE(boolean,((set->_bits[index/8] bitand mask)?TRUE:FALSE)));
		SVM_Value_Boolean b = ::svm_function_call(svm,func,argc,argv);
		if(::svm_value_boolean_get(svm,b))
		{
			set->_bits[index/8] |= mask;
		}
		else
		{
			set->_bits[index/8] &= ~mask;
		}
	}
#line 792 "src/plugin.cpp"
	return nullptr;
}


/* INSTRUCTION bit.map MUTABLE bit.set PEP:function . *:parameters */

SVM_Value instruction_map(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 769 "../plugins/bit/core/bit.svm_plugin"
	function_map(svm,argc,argv);
#line 803 "src/plugin.cpp"
	return nullptr;
}


/* Generic handling functions */

}
