/*
 * Copyright (C) 2025-09-22  Julien BRUGUIER
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#pragma once

#include <set>
#include <map>
#include <vector>
#include <list>
#include <memory>
#include <iostream>
#include <src/plugin.h>

namespace Config
{

	struct String
	{
		static std::string unescape(const std::string& s)
		{
			std::string r;
			bool escape = false;
			for(const auto& c:s)
			{
				if(escape)
				{
					if(c=='n')
					{
						r+='\n';
					}
					else
					{
						r+=c;
					}
					escape=false;
				}
				else
				{
					if(c=='\\')
					{
						escape=true;
					}
					else
					{
						r+=c;
					}
				}
			}
			return r;
		}
		static std::string escape(const std::string& s)
		{
			std::string r;
			for(const auto& c:s)
			{
				if(c=='\n')
				{
					r+="\\n";
				}
				else if((c=='"') or (c=='\\'))
				{
					r+='\\';
					r+=c;
				}
				else
				{
					r+=c;
				}
			}
			return r;
		}
	};

	struct ConfigElement: public std::enable_shared_from_this<ConfigElement>
	{
		virtual ~ConfigElement() {}
		virtual std::shared_ptr<ConfigElement> deep_copy() const = 0;
		virtual void format(std::ostream& os, const size_t indent) const = 0;
		virtual std::shared_ptr<const ConfigElement> search(const std::shared_ptr<const ConfigElement>& root, std::list<std::string>& path, std::set<const void*>& loop_detector) const = 0;
		virtual std::vector<std::shared_ptr<const ConfigElement> > values(const std::shared_ptr<const ConfigElement>& root) const = 0;
		virtual std::shared_ptr<ConfigElement> merge(const std::shared_ptr<const ConfigElement>& element) const = 0;
		virtual void find(const std::string& end, std::list<std::list<std::string> >& result, const std::list<std::string>& path) const = 0;
	};

	struct ConfigValue : public ConfigElement
	{
		virtual std::shared_ptr<const ConfigElement> search(const std::shared_ptr<const ConfigElement>& root, std::list<std::string>& path, std::set<const void*>& loop_detector) const override
		{
			if(not path.empty()) { return std::shared_ptr<const ConfigElement>(); }
			return this->shared_from_this();
		}
		virtual std::vector<std::shared_ptr<const ConfigElement> > values(const std::shared_ptr<const ConfigElement>& root) const override
		{
			return { this->shared_from_this() };
		}
		virtual std::shared_ptr<ConfigElement> merge(const std::shared_ptr<const ConfigElement>& element) const override
		{
			return element->deep_copy();
		}
		virtual void find(const std::string& end, std::list<std::list<std::string> >& result, const std::list<std::string>& path) const override
		{
		}
		virtual SVM_Value value(const void *svm) const = 0;
	};

	struct ConfigValueInteger : public ConfigValue
	{
		ConfigValueInteger(const long long int value)
		:_value(value) {}
		virtual std::shared_ptr<ConfigElement> deep_copy() const override
		{
			return std::make_shared<ConfigValueInteger>(_value);
		}
		virtual void format(std::ostream& os, const size_t indent) const override
		{
			os << _value;
		}
		virtual SVM_Value value(const void *svm) const override
		{
			return ::svm_value_integer_new(svm,_value);
		}

		long long int _value;
	};

	struct ConfigValueString : public ConfigValue
	{
		ConfigValueString(const std::string& value)
		:_value(value) {}
		virtual std::shared_ptr<ConfigElement> deep_copy() const override
		{
			return std::make_shared<ConfigValueString>(_value);
		}
		virtual void format(std::ostream& os, const size_t indent) const override
		{
			os << "\"" << String::escape(_value) << "\"";
		}
		virtual SVM_Value value(const void *svm) const override
		{
			return ::svm_value_string_new(svm,::svm_string_new(svm,_value.c_str(),_value.size()));
		}

		std::string _value;
	};

	struct ConfigValueBoolean : public ConfigValue
	{
		ConfigValueBoolean(const bool value)
		:_value(value) {}
		virtual std::shared_ptr<ConfigElement> deep_copy() const override
		{
			return std::make_shared<ConfigValueBoolean>(_value);
		}
		virtual void format(std::ostream& os, const size_t indent) const override
		{
			if(_value) os << "TRUE"; else os << "FALSE";
		}
		virtual SVM_Value value(const void *svm) const override
		{
			return ::svm_value_boolean_new(svm,_value?TRUE:FALSE);
		}

		bool _value;
	};

	struct ConfigValuePluginEntryPoint : public ConfigValue
	{
		ConfigValuePluginEntryPoint(const std::string& plugin, const std::string& entry)
		:_plugin(plugin), _entry(entry) {}
		virtual std::shared_ptr<ConfigElement> deep_copy() const override
		{
			return std::make_shared<ConfigValuePluginEntryPoint>(_plugin,_entry);
		}
		virtual void format(std::ostream& os, const size_t indent) const override
		{
			os << _plugin << "." << _entry;
		}
		virtual SVM_Value value(const void *svm) const override
		{
			return ::svm_value_pluginentrypoint_new__raw(svm,_plugin.c_str(),_entry.c_str());
		}

		std::string _plugin;
		std::string _entry;
	};

	struct ConfigValuePlugin : public ConfigValue
	{
		ConfigValuePlugin(const std::string& plugin, const std::string& type, const std::string& value)
		:_plugin(plugin), _type(type), _value(value) {}
		virtual std::shared_ptr<ConfigElement> deep_copy() const override
		{
			return std::make_shared<ConfigValuePlugin>(_plugin,_type,_value);
		}
		virtual void format(std::ostream& os, const size_t indent) const override
		{
			os << "CONST " << _plugin << "." << _type << " \"" << String::escape(_value) << "\"";
		}
		virtual SVM_Value value(const void *svm) const override
		{
			return ::svm_value_plugin_new_const__string(svm,::svm_value_pluginentrypoint_new__raw(svm,_plugin.c_str(),_type.c_str()),::svm_string_new(svm,_value.c_str(),_value.size()));
		}

		std::string _plugin;
		std::string _type;
		std::string _value;
	};

	struct ConfigValueReference : public ConfigElement
	{
		ConfigValueReference(const std::list<std::string>& ref)
		:_ref(ref) {}
		virtual std::shared_ptr<ConfigElement> deep_copy() const override
		{
			return std::make_shared<ConfigValueReference>(_ref);
		}

		virtual void format(std::ostream& os, const size_t indent) const override
		{
			os << "$";
			bool first = true;
			for(const auto& r: _ref)
			{
				if(first) first=false; else os << ".";
				os << r;
			}
			os << "$";
		}
		virtual std::shared_ptr<const ConfigElement> search(const std::shared_ptr<const ConfigElement>& root, std::list<std::string>& path, std::set<const void*>& loop_detector) const override
		{
			if(not path.empty()) { return std::shared_ptr<const ConfigElement>(); }
			if(not loop_detector.insert(this).second) { return std::shared_ptr<const ConfigElement>(); }
			auto p = _ref;
			return root->search(root,p,loop_detector);
		}
		virtual std::vector<std::shared_ptr<const ConfigElement> > values(const std::shared_ptr<const ConfigElement>& root) const override
		{
			std::set<const void*> loop_detector;
			auto p = _ref;
			auto e = root->search(root,p,loop_detector);
			if(not e) return {};
			return e->values(root);
		}
		virtual std::shared_ptr<ConfigElement> merge(const std::shared_ptr<const ConfigElement>& element) const override
		{
			return element->deep_copy();
		}
		virtual void find(const std::string& end, std::list<std::list<std::string> >& result, const std::list<std::string>& path) const override
		{
		}

		std::list<std::string> _ref;
	};

	struct ConfigObject : public ConfigElement
	{
		virtual std::shared_ptr<ConfigElement> deep_copy() const override
		{
			auto object = std::make_shared<ConfigObject>();
			for(const auto& m:_members)
			{
				object->_members.insert(std::make_pair(m.first,m.second->deep_copy()));
			}
			object->_root = _root;
			return object;
		}
		
		virtual void format(std::ostream& os, const size_t indent) const override
		{
			size_t add = _root?0:1;
			if(not _root) os << "{" << std::endl;
			for(auto& m: _members)
			{
				os << std::string(indent+add,'\t') << m.first << ": ";
				m.second->format(os,indent+add);
				os << std::endl;
			}
			if(not _root) os << std::string(indent,'\t') << "}";
		}
		virtual std::shared_ptr<const ConfigElement> search(const std::shared_ptr<const ConfigElement>& root, std::list<std::string>& path, std::set<const void*>& loop_detector) const override
		{
			if(path.empty()) { return this->shared_from_this(); }
			auto it=_members.find(path.front());
			if(it==_members.end()) { return std::shared_ptr<const ConfigElement>(); }
			path.pop_front();
			return it->second->search(root,path,loop_detector);
		}
		
		virtual std::vector<std::shared_ptr<const ConfigElement> > values(const std::shared_ptr<const ConfigElement>& root) const override
		{
			std::vector<std::shared_ptr<const ConfigElement> > vv;
			for(const auto& m:_members)
			{
				auto v = m.second->values(root);
				vv.insert(vv.end(),v.begin(),v.end());
			}
			return vv;
		}

		std::shared_ptr<const ConfigValue> value(const std::list<std::string>& path) const
		{
			std::set<const void*> loop_detector;
			auto p = path;
			return std::dynamic_pointer_cast<const ConfigValue>(search(this->shared_from_this(),p,loop_detector));
		}

		std::vector<std::shared_ptr<const ConfigValue> > array(const std::list<std::string>& path) const
		{
			std::set<const void*> loop_detector;
			auto p = path;
			auto element = search(this->shared_from_this(),p,loop_detector);
			if(not element) { return {}; }
			auto v = element->values(this->shared_from_this());
			std::vector<std::shared_ptr<const ConfigValue> > vv;
			for(const auto& i: v)
			{
				auto ii = std::dynamic_pointer_cast<const ConfigValue>(i);
				vv.push_back(ii);
			}
			return vv;
		}

		virtual std::shared_ptr<ConfigElement> merge(const std::shared_ptr<const ConfigElement>& element) const override
		{
			auto object = std::dynamic_pointer_cast<const ConfigObject>(element);
			if(not object) { return element->deep_copy(); }
			auto result = std::make_shared<ConfigObject>();
			auto itt = _members.begin();
			auto ito = object->_members.begin();
			for(;;)
			{
				if((itt==_members.end()) and (ito==object->_members.end())) break;
				if(itt==_members.end())
				{
					result->_members.insert(std::make_pair(ito->first,ito->second->deep_copy()));
					++ito;
					continue;
				}
				if(ito==object->_members.end())
				{
					result->_members.insert(std::make_pair(itt->first,itt->second->deep_copy()));
					++itt;
					continue;
				}
				if(itt->first<ito->first)
				{
					result->_members.insert(std::make_pair(itt->first,itt->second->deep_copy()));
					++itt;
					continue;
				}
				if(itt->first>ito->first)
				{
					result->_members.insert(std::make_pair(ito->first,ito->second->deep_copy()));
					++ito;
					continue;
				}
				result->_members.insert(std::make_pair(itt->first,itt->second->merge(ito->second)));
				++itt;
				++ito;
			}
			result->_root = _root;
			return result;
		}
		virtual void find(const std::string& end, std::list<std::list<std::string> >& result, const std::list<std::string>& path) const override
		{
			std::list<std::string> p = path;
			for(const auto& m:_members)
			{
				p.push_back(m.first);
				if(m.first==end)
				{
					result.push_back(p);
				}
				m.second->find(end,result,p);
				p.pop_back();
			}
		}

		bool _root = false;
		std::map<std::string,std::shared_ptr<ConfigElement> > _members;
	};

	struct FlagSet
	{
		bool operator() (const std::string& f) const
		{
			return _flags.find(f)!=_flags.end();
		}
		std::set<std::string> _flags;
	};

	struct Error
	{
		Error()
		:_top(0), _bottom(0) {}
		explicit Error(const std::string& message)
		:_message(message), _top(0), _bottom(0) {}
		void position(const size_t top, const size_t bottom)
		{
			_top = top;
			_bottom = (bottom>top)?bottom:top;
		}
		template<typename Flux>
		friend Flux& operator<<(Flux& f, const Error& e)
		{
			f << "Error";
			if(e._top>0)
			{
				if(e._top==e._bottom)
				{
					f << " at line " << e._top;
				}
				else
				{
					f << " at lines " << e._top << "-" << e._bottom;
				}
			}
			f << ": " << e._message << std::endl;
			return f;
		}
		const std::string& message() const { return _message; }
		private:
			std::string _message;
			size_t _top;
			size_t _bottom;
	};
}
