/*
 * Copyright (C) 2024-10-27  Julien BRUGUIER
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */


#include <string>
#include <sstream>

#line 138 "../plugins/funccache/core/funccache.svm_plugin"
#include <vector>
#include <memory>
#include <atomic>
#line 27 "src/plugin.cpp"

#include <src/plugin.h>

extern "C"
{

void plugin_configure(void *plugin)
{
	::svm_plugin_configure(plugin,
		"PLUGIN funccache \n"
		"DEFINE \n"
		"	OPTION funccache.weak -w BLN \n"
		"	OPTION funccache.limit -l INT \n"
		"	OPTION funccache.policy -p INT \n"
		"	WAITING INSTRUCTION funccache.call [ STR SYM ] PTR \n"
		"	WAITING OVERRIDE INSTRUCTION funccache.store SYM PTR \n"
		"	WAITING INSTRUCTION funccache.statistics MUTABLE INT 6 \n"
		"	SYSTEM INSTRUCTION funccache.print \n"
		,SVM_API_SIGNATURE,SVM_VERSION);
}

#ifndef ARGV_VALUE
#define ARGV_VALUE(index,type) ::svm_value_##type##_get(svm,::svm_parameter_value_get(svm,argv[(index)]))
#endif
#ifndef ARGV_PLUGIN
#define ARGV_PLUGIN(index,plugin,name) reinterpret_cast<type_##name*>(::svm_value_plugin_get_internal(svm,::svm_parameter_value_get(svm,argv[(index)])))
#endif
#ifndef ARGV_MARKER
#define ARGV_MARKER(index) std::string(::svm_parameter_marker_get(svm,argv[(index)]).string)
#endif
#ifndef ARGV_KEYWORD
#define ARGV_KEYWORD(index) std::string(::svm_parameter_keyword_get(svm,argv[(index)]).string)
#endif
#ifndef ARGV_STRUCT
#define ARGV_STRUCT(index,plugin,name) reinterpret_cast<struct_##name*>(::svm_structure_get_internal(svm,::svm_value_pluginentrypoint_new__raw(svm,#plugin,#name),::svm_parameter_structure_get(svm,argv[(index)])))
#endif
#ifndef ARGV_VARIABLE
#define ARGV_VARIABLE(index) ::svm_parameter_variable_get(svm,argv[(index)]);
#endif
#ifndef NEW_VALUE
#define NEW_VALUE(type,value) ::svm_value_##type##_new(svm,(value))
#endif
#ifndef NEW_PLUGIN
#define NEW_PLUGIN(plugin,name,value) ::svm_value_plugin_new(svm,::svm_value_pluginentrypoint_new__raw(svm,#plugin,#name),(value))
#endif
#ifndef NEW_STRUCT
#define NEW_STRUCT(plugin,name,value) ::svm_structure_new(svm,::svm_value_pluginentrypoint_new__raw(svm,#plugin,#name),(value))
#endif
#ifndef NEW_STRING
#define NEW_STRING(raw_string) ::svm_string_new(svm,raw_string.c_str(),raw_string.size())
#endif
#ifndef RAW_STRING
#define RAW_STRING(svm_string) std::string(svm_string.string,svm_string.size)
#endif
#ifndef NEW_BOOLEAN
#define NEW_BOOLEAN(boolean) ((boolean)?TRUE:FALSE)
#endif
#ifndef RAW_BOOLEAN
#define RAW_BOOLEAN(boolean) ((boolean)==TRUE)
#endif
#ifndef NEW_NULL_VALUE
#define NEW_NULL_VALUE(type) ::svm_value_##type##_new_null(svm)
#endif
#ifndef NEW_NULL_PLUGIN
#define NEW_NULL_PLUGIN(plugin,name) ::svm_value_plugin_new_null(svm,::svm_value_pluginentrypoint_new__raw(svm,#plugin,#name))
#endif
#ifndef NEW_NULL_STRUCT
#define NEW_NULL_STRUCT(plugin,name) ::svm_structure_new_null(svm,::svm_value_pluginentrypoint_new__raw(svm,#plugin,#name))
#endif
#ifndef ERROR_INTERNAL
#define ERROR_INTERNAL(irq,message) ::svm_processor_current_raise_error_internal__raw(svm,irq,message)
#endif
#ifndef ERROR_EXTERNAL
#define ERROR_EXTERNAL(plugin,name,message) ::svm_processor_current_raise_error_external__raw(svm,::svm_value_pluginentrypoint_new__raw(svm,#plugin,#name),message)
#endif
#ifndef CONST_PEP
#define CONST_PEP(plugin,name) ::svm_value_pluginentrypoint_new__raw(svm,#plugin,#name)
#endif
#ifndef CURRENT
#define CURRENT(object) ::svm_##object##_get_current(svm)
#endif
#ifndef RETURN
#define RETURN return nullptr
#endif
#ifndef VARIABLE_GLOBAL
#define VARIABLE_GLOBAL(variable) ::svm_variable_scope_set_global(svm,(variable))
#endif
#ifndef VARIABLE_LOCAL
#define VARIABLE_LOCAL(variable) ::svm_variable_scope_set_local(svm,(variable))
#endif
#ifndef VARIABLE_DELETE
#define VARIABLE_DELETE(variable) ::svm_variable_delete(svm,(variable))
#endif

}

#line 145 "../plugins/funccache/core/funccache.svm_plugin"
static size_t limit = 0;
static SVM_Boolean weak = FALSE;
static size_t policy = 0;

struct CachedCall
{
	explicit CachedCall(const SVM_Value_Symbol function)
	:_function(function)
	{
	}

	void add(const SVM_Value param)
	{
		_parameters.push_back(param);
	}
	void add()
	{
		_parameters.push_back(nullptr);
	}
	void print(const void *svm, std::ostream& os) const
	{
		SVM_String s = ::svm_value_print(svm,_function);
		os << RAW_STRING(s) << " [";
		for(const auto&p: _parameters)
		{
			os << " ";
			if(not p)
			{
				os << "(void)";
			}
			else
			{
				s = ::svm_value_print(svm,p);
				os << RAW_STRING(s);
			}
		}
		os << " ] Usage=" << _usage << " Order=" << _order;

	}

	SVM_Value_Symbol _function;
	std::vector<SVM_Value> _parameters;
	size_t _usage = 0;
	size_t _order = 0;
};

struct Cache
{
	Cache()
	:_read(0), _found(0), _missed(0), _store(0), _eviction(0) {}
	std::pair<bool,size_t> find(const void *svm, const CachedCall& c) const
	{
		size_t start = 0;
		size_t end = _cache.size();
		for( ; ; )
		{
			if(start>=end)
			{
				return std::make_pair(false,start);
			}
			size_t pivot = (start+end)/2;
			SVM_Value_Plugin_Comparison comparison = Cache::compare(svm,c,*_cache[pivot]);
			switch(comparison)
			{
				case ORDER_EQUAL:
					return std::make_pair(true,pivot);
					break;
				case ORDER_INFERIOR:
					end = pivot;
					break;
				case ORDER_SUPERIOR:
					start = pivot+1;
					break;
			}
		}
	}

	void insert(const void *svm, const std::shared_ptr<CachedCall>& c)
	{
		auto it = find(svm,*c);
		if(it.first) return;
		_cache.insert(_cache.begin()+it.second,c);
	}

	size_t remove()
	{
		if(_cache.empty()) return 0;
		auto remove = _cache.begin();
		auto threashold = (policy*_cache.size())/100;
		if(threashold==0) threashold=1;
		for(auto it=_cache.begin() ; it!=_cache.end() ; ++it)
		{
			if((*it)->_order>threashold) continue;
			if((*remove)->_usage<(*it)->_usage) continue;
			if((*remove)->_order<(*it)->_order) continue;
			remove=it;
		}
		auto order = (*remove)->_order;
		_cache.erase(remove);
		++_eviction;
		return order;
	}

	void update_found(CachedCall& found)
	{
		++found._usage;
		update_order(found._order);
		found._order = _cache.size();
	}

	void update_insert(CachedCall& insert)
	{
		insert._usage = 1;
		insert._order = _cache.size();
	}

	void update_order(const size_t order)
	{
		for(auto& c: _cache)
		{
			if(c->_order>order)
			{
				--c->_order;
			}
		}
	}

	void clean(const void *svm)
	{
		for(const auto& c: _cache)
		{
			VARIABLE_LOCAL(c->_function);
			for(const auto& v: c->_parameters)
			{
				if(v)
				{
					VARIABLE_LOCAL(v);
				}
			}
		}
	}
	void print(const void *svm, std::ostream& os) const
	{
		os << "Function call cache [size=" << _cache.size() << ", read=" << _read << ", found=" << _found << ", missed=" << _missed << ", written=" << _store << ", evicted=" << _eviction << "]:" << std::endl;
		for(const auto& c:_cache)
		{
			os << "   ";
			c->print(svm,os);
			os << std::endl;
		}
	}
	std::vector<std::shared_ptr<CachedCall> > _cache;
	SVM_Lock _lock;
	std::atomic_size_t _read;
	std::atomic_size_t _found;
	std::atomic_size_t _missed;
	std::atomic_size_t _store;
	std::atomic_size_t _eviction;

	private:
		static SVM_Value_Plugin_Comparison compare(const void *svm, const CachedCall& r, const CachedCall& p)
		{
			SVM_Comparison_Result sc = ::svm_value_compare(svm,r._function,p._function);
			if(sc.inferior) { return ORDER_INFERIOR; }
			if(sc.superior) { return ORDER_SUPERIOR; }
			if(r._parameters.size()<p._parameters.size()) { return ORDER_INFERIOR; }
			if(r._parameters.size()>p._parameters.size()) { return ORDER_SUPERIOR; }
			for(auto fp=r._parameters.begin(),pp=p._parameters.begin() ; fp!=r._parameters.end() ; ++fp,++pp)
			{
				if(not *fp) continue; // ignore non initialised values in call
				if(not *pp) { return ORDER_SUPERIOR; }
				SVM_Comparison_Result c = ::svm_value_compare(svm,*fp,*pp);
				if(not (c.order and c.total))
				{
					ERROR_INTERNAL(FAILURE,"Values can not be compared with a total order");
				}
				if(not weak and c.weak)
				{
					ERROR_INTERNAL(FAILURE,"Refuse to use weak comparison between values");
				}
				if(c.inferior) { return ORDER_INFERIOR; }
				if(c.superior) { return ORDER_SUPERIOR; }
			}
			return ORDER_EQUAL;
		}
};

static Cache cache;

#line 314 "src/plugin.cpp"

extern "C"
{

/* OPTION funccache.weak -w BLN */


/* OPTION funccache.limit -l INT */


/* OPTION funccache.policy -p INT */


/* WAITING INSTRUCTION funccache.call [ STR SYM ] PTR */

SVM_Value instruction_call(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 384 "../plugins/funccache/core/funccache.svm_plugin"
	SVM_Value function = ::svm_parameter_value_get(svm,argv[0]);
	if(::svm_value_type_is_string(svm,function))
	{
		SVM_Code code = ::svm_processor_get_currentcode(svm,CURRENT(kernel));
		SVM_Address address = ::svm_code_label_get_address(svm,code,function);
		function = ::svm_value_symbol_new(svm,code,address);
	}
	CachedCall trial(function);
	SVM_Value_Pointer params = ::svm_parameter_value_get(svm,argv[1]);
	SVM_Address address = ::svm_value_pointer_get_address(svm,params);
	SVM_Size size = ::svm_value_pointer_get_size(svm,params);
	for(SVM_Address i=address ; i<(address+size) ; ++i)
	{
		if(::svm_memory_address_is_initialised(svm,CURRENT(kernel),i))
		{
			trial.add(::svm_memory_read_address(svm,CURRENT(kernel),i));
		}
		else
		{
			trial.add();
		}
	}
	SVM_LockGuard_Read guard = ::svm_lock_readguard_new(svm,cache._lock,TRUE);
	auto found = cache.find(svm,trial);
	++cache._read;
	if(found.first)
	{
		++cache._found;
		CachedCall& c = *(cache._cache[found.second]);
		cache.update_found(c);
		auto it = c._parameters.begin();
		for(SVM_Address i=address ; i<(address+size) ; ++i,++it)
		{
			if(static_cast<bool>(*it) and not ::svm_memory_address_is_initialised(svm,CURRENT(kernel),i))
			{
				SVM_Value v = ::svm_value_copy(svm,*it);
				::svm_value_state_set_movable(svm,v);
				::svm_memory_write_address(svm,CURRENT(kernel),i,v);
			}
		}
		RETURN;
	}
	++cache._missed;
	SVM_Value_Symbol current = ::svm_processor_get_currentinstruction(svm,CURRENT(kernel));
	::svm_processor_jump_global(svm,CURRENT(kernel),current);
	SVM_Parameter *p = ::svm_parameter_array_new(svm,2);
	p[0] = ::svm_parameter_value_new(svm,function);
	p[1] = argv[1];
	::svm_processor_instructionoverride_set_global(svm,CURRENT(kernel),current,CONST_PEP(funccache,store),2,p,LOCAL);
	::svm_processor_call_global(svm,CURRENT(kernel),function,params);
#line 383 "src/plugin.cpp"
	return nullptr;
}


/* WAITING OVERRIDE INSTRUCTION funccache.store SYM PTR */

SVM_Value instruction_store(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 451 "../plugins/funccache/core/funccache.svm_plugin"
	SVM_Value_Symbol function = ::svm_parameter_value_get(svm,argv[0]);
	VARIABLE_GLOBAL(function);
	SVM_Value_Pointer params = ::svm_parameter_value_get(svm,argv[1]);
	auto value = std::make_shared<CachedCall>(function);
	SVM_Address address = ::svm_value_pointer_get_address(svm,params);
	SVM_Size size = ::svm_value_pointer_get_size(svm,params);
	for(SVM_Address i=address ; i<(address+size) ; ++i)
	{
		if(::svm_memory_address_is_initialised(svm,CURRENT(kernel),i))
		{
			SVM_Value v = ::svm_memory_read_address(svm,CURRENT(kernel),i);
			v = ::svm_value_copy(svm,v);
			VARIABLE_GLOBAL(v);
			value->add(v);
		}
		else
		{
			value->add();
		}
	}
	SVM_LockGuard_Write guard = ::svm_lock_writeguard_new(svm,cache._lock,TRUE);
	if((limit>0) and (cache._cache.size()>=limit))
	{
		cache.update_order(cache.remove());
	}
	cache.insert(svm,value);
	cache.update_insert(*value);
	++cache._store;
	SVM_Value_Symbol current = ::svm_processor_get_currentinstruction(svm,CURRENT(kernel));
	::svm_processor_instructionoverride_reset_global(svm,CURRENT(kernel),current,LOCAL);
#line 423 "src/plugin.cpp"
	return nullptr;
}


/* WAITING INSTRUCTION funccache.statistics MUTABLE INT 6 */

SVM_Value instruction_statistics(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 490 "../plugins/funccache/core/funccache.svm_plugin"
	SVM_Value_Integer c = ::svm_parameter_value_get(svm,argv[0]);
	SVM_Value_Integer r = ::svm_parameter_value_get(svm,argv[1]);
	SVM_Value_Integer f = ::svm_parameter_value_get(svm,argv[2]);
	SVM_Value_Integer m = ::svm_parameter_value_get(svm,argv[3]);
	SVM_Value_Integer s = ::svm_parameter_value_get(svm,argv[4]);
	SVM_Value_Integer e = ::svm_parameter_value_get(svm,argv[5]);
	SVM_LockGuard_Read guard = ::svm_lock_readguard_new(svm,cache._lock,TRUE);
	::svm_value_integer_set(svm,c,cache._cache.size());
	::svm_value_integer_set(svm,r,cache._read);
	::svm_value_integer_set(svm,e,cache._found);
	::svm_value_integer_set(svm,m,cache._missed);
	::svm_value_integer_set(svm,s,cache._store);
	::svm_value_integer_set(svm,e,cache._eviction);
#line 446 "src/plugin.cpp"
	return nullptr;
}


/* SYSTEM INSTRUCTION funccache.print */

SVM_Value instruction_print(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 523 "../plugins/funccache/core/funccache.svm_plugin"
	std::ostringstream oss;
	cache.print(svm,oss);
	::svm_machine_trace__string(svm,NEW_STRING(oss.str()));
#line 459 "src/plugin.cpp"
	return nullptr;
}


/* Generic handling functions */

void plugin_initialisation(const void *svm)
{
#line 338 "../plugins/funccache/core/funccache.svm_plugin"
	cache._lock = ::svm_lock_new(svm);
	auto l = ::svm_plugin_get_option(svm,CONST_PEP(funccache,limit));
	if(not ::svm_value_state_is_null(svm,l))
	{
		limit = ::svm_value_integer_get(svm,l);
	}
	weak = ::svm_value_boolean_get(svm,::svm_plugin_get_option(svm,CONST_PEP(funccache,weak)));
	policy = 70;
	auto p = ::svm_plugin_get_option(svm,CONST_PEP(funccache,policy));
	if(not ::svm_value_state_is_null(svm,p))
	{
		auto rp = ::svm_value_integer_get(svm,p);
		if(rp<0) rp=0;
		if(rp>100) rp=100;
		policy = rp;
	}
#line 485 "src/plugin.cpp"
}

void plugin_finalisation(const void *svm)
{
#line 358 "../plugins/funccache/core/funccache.svm_plugin"
	cache.clean(svm);
	VARIABLE_DELETE(cache._lock);
#line 493 "src/plugin.cpp"
}

}
