/*
 * Copyright (C) 2022-03-20  Julien BRUGUIER
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#pragma once

#include <iostream>
#include <sstream>
#include <memory>
#include <vector>
#include <set>
#include <map>

namespace Html
{

template<typename E>
struct Compare
{
	bool operator() (const std::weak_ptr<E>& l, const std::weak_ptr<E>& r)
	{
		return l.lock()<r.lock();
	}
};

struct Element : public std::enable_shared_from_this<Element>
{
	typedef std::shared_ptr<Element> SP;
	typedef std::weak_ptr<Element> WP;
	virtual ~Element() {}
	virtual Element::SP clone() const = 0;

	virtual Element::SP id(const std::string& id)
	{
		return Element::SP();
	}
	virtual std::vector<Element::SP> in_class(const std::string& class_name)
	{
		std::vector<Element::SP> elements;
		return elements;
	}
	template<typename oStream>
	friend oStream& operator<<(oStream& os, const Element::SP& e)
	{
		e->print(os);
		return os;
	}
	virtual void print(std::ostream& os) const = 0;
	std::set<Element::WP,Compare<Element> > _parents;
	bool verification(const Element::SP& child) const
	{
		if(child==this->shared_from_this())
		{
			return false;
		}
		for(const auto& p:_parents)
		{
			auto pp = p.lock();
			if(not pp) continue;
			if(not pp->verification(child)) return false;
		}
		return true;
	}
};

struct Text : public Element
{
	Text(const std::string& text)
	:_text(text) {}
	virtual Element::SP clone() const override
	{
		return std::make_shared<Text>(_text);
	}

	virtual void print(std::ostream& os) const override
	{
		os << _text;
	}

	std::string _text;
};

struct Comment : public Element
{
	Comment(const std::string& text)
	:_text(text) {}
	virtual Element::SP clone() const override
	{
		return std::make_shared<Comment>(_text);
	}

	virtual void print(std::ostream& os) const override
	{
		os << "<!--" << _text << "-->";
	}
	std::string _text;
};

struct Meta : public Element
{
	Meta(const std::string& text)
	:_text(text) {}
	virtual Element::SP clone() const override
	{
		return std::make_shared<Meta>(_text);
	}

	virtual void print(std::ostream& os) const override
	{
		os << "<?" << _text << "?>";
	}
	std::string _text;
};

struct Node : public Element
{
	Node(const std::string& type)
	:_type(type) {}
	virtual Element::SP clone() const override
	{
		auto e = std::make_shared<Node>(_type);
		e->_id = _id;
		e->_class = _class;
		e->_attributes = _attributes;
		for(const auto& c: _children)
		{
			e->_children.push_back(c->clone());
		}
		return e;
	}
	void id_and_class()
	{
		auto it = _attributes.find("id");
		if(it!=_attributes.end())
		{
			_id = it->second;
		}
		it = _attributes.find("class");
		if(it!=_attributes.end())
		{
			_class.clear();
			auto c = it->second;
			for( ; ; )
			{
				auto itt=c.find(" ");
				if(itt==std::string::npos)
				{
					_class.insert(c);
					break;
				}
				if(itt>0)
				{
					_class.insert(c.substr(0,itt));
				}
				c = c.substr(itt+1);
			}
		}
	}

	virtual Element::SP id(const std::string& id)
	{
		if(_id==id) return this->shared_from_this();
		for(auto& c:_children)
		{
			auto e = c->id(id);
			if(static_cast<bool>(e)) return e;
		}
		return Element::SP();
	}
	virtual std::vector<Element::SP> in_class(const std::string& class_name)
	{
		std::vector<Element::SP> elements;
		auto it = _class.find(class_name);
		if(it!=_class.end())
		{
			elements.push_back(this->shared_from_this());
		}
		for(auto& c:_children)
		{
			auto e = c->in_class(class_name);
			elements.insert(elements.end(),e.begin(),e.end());
		}
		return elements;
	}
	virtual void print(std::ostream& os) const override
	{
		os << "<" << _type;
		for(const auto& a:_attributes)
		{
			os << " " << a.first << "=\"" << a.second << "\"";
		}
		if(_children.empty())
		{
			os << "/>";
			return;
		}
		os << ">";
		for(const auto& c:_children)
		{
			os << c;
		}
		os << "</" << _type << ">";
	}

	std::string _type;
	std::string _id;
	std::set<std::string> _class;
	std::map<std::string,std::string> _attributes;
	std::vector<Element::SP> _children;
};

struct Dom
{
	using SP = std::shared_ptr<Dom>;
	Dom() = default;
	Dom(const Dom& dom)
	:_error(dom._error)
	{
		for(const auto& r:dom._root)
		{
			_root.push_back(r->clone());
		}
	}
	std::vector<Element::SP> _root;
	std::string _error;

	Element::SP id(const std::string& id) const
	{
		for(auto& r:_root)
		{
			auto e = r->id(id);
			if(static_cast<bool>(e)) return e;
		}
		return Element::SP();
	}

	std::vector<Element::SP> in_class(const std::string& class_name) const
	{
		std::vector<Element::SP> elements;
		for(auto& r:_root)
		{
			auto e = r->in_class(class_name);
			elements.insert(elements.end(),e.begin(),e.end());
		}
		return elements;
	}

	template<typename oStream>
	friend oStream& operator<<(oStream& os, const Dom& d)
	{
		if(not d._error.empty())
		{
			os << "Error: " << d._error;
		}
		else
		{
			for(auto& e:d._root)
			{
				os << e;
			}
		}
		return os;
	}
};
}
