/*
 * Copyright (C) 2022-01-30  Julien BRUGUIER
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */


#include <string>
#include <sstream>

#line 53 "../plugins/http/core/http.svm_plugin"
#include <string>
#include <map>
#include <set>
#include <vector>
#include <sstream>
#include <cstring>
#line 30 "src/plugin.cpp"

#include <src/plugin.h>

extern "C"
{

void plugin_configure(void *plugin)
{
	::svm_plugin_configure(plugin,
		"PLUGIN http \n"
		"USE \n"
		"	TYPE com.device \n"
		"DEFINE \n"
		"	FUNCTION http.protocol_mesg_1_1 STR BLN -> INT ? \n"
		"	TYPE http.mesg_1_1 \n"
		"	INTERRUPTION http.bad_mesg_1_1 \n"
		"	INSTRUCTION http.decode STR -> http.mesg_1_1 \n"
		"	INSTRUCTION http.encode http.mesg_1_1 -> STR \n"
		"	INTERRUPTION http.bad_returncode \n"
		"	INSTRUCTION http.new ( 'QUERY' [ 'GET' 'HEAD' 'POST' 'PUT' 'DELETE' 'CONNECT' 'OPTIONS' 'TRACE' 'PATCH' ]:method STR:uri STR:hostname | 'REPLY' INT:return_code ) -> http.mesg_1_1 \n"
		"	INSTRUCTION http.query http.mesg_1_1 -> BLN \n"
		"	INSTRUCTION http.reply http.mesg_1_1 -> BLN \n"
		"	INTERRUPTION http.bad_mesgtype \n"
		"	INSTRUCTION http.get_method http.mesg_1_1 -> STR \n"
		"	INSTRUCTION http.set_method MUTABLE http.mesg_1_1 STR \n"
		"	INSTRUCTION http.get_uri http.mesg_1_1 -> STR \n"
		"	INSTRUCTION http.set_uri MUTABLE http.mesg_1_1 STR \n"
		"	INSTRUCTION http.get_returncode http.mesg_1_1 -> INT \n"
		"	INSTRUCTION http.set_returncode MUTABLE http.mesg_1_1 INT \n"
		"	INSTRUCTION http.get_header http.mesg_1_1 STR:key -> STR ?:value \n"
		"	INSTRUCTION http.set_header MUTABLE http.mesg_1_1 STR:key STR ?:value \n"
		"	INSTRUCTION http.get_headers http.mesg_1_1 -> PTR \n"
		"	INSTRUCTION http.set_headers MUTABLE http.mesg_1_1 PTR \n"
		"	INSTRUCTION http.get_payload http.mesg_1_1 -> STR \n"
		"	INSTRUCTION http.set_payload MUTABLE http.mesg_1_1 STR 'LENGTH' ? \n"
		,SVM_API_SIGNATURE,SVM_VERSION);
}

#ifndef ARGV_VALUE
#define ARGV_VALUE(index,type) ::svm_value_##type##_get(svm,::svm_parameter_value_get(svm,argv[(index)]))
#endif
#ifndef ARGV_PLUGIN
#define ARGV_PLUGIN(index,plugin,name) reinterpret_cast<type_##name*>(::svm_value_plugin_get_internal(svm,::svm_parameter_value_get(svm,argv[(index)])))
#endif
#ifndef ARGV_MARKER
#define ARGV_MARKER(index) std::string(::svm_parameter_marker_get(svm,argv[(index)]).string)
#endif
#ifndef ARGV_KEYWORD
#define ARGV_KEYWORD(index) std::string(::svm_parameter_keyword_get(svm,argv[(index)]).string)
#endif
#ifndef ARGV_STRUCT
#define ARGV_STRUCT(index,plugin,name) reinterpret_cast<struct_##name*>(::svm_structure_get_internal(svm,::svm_value_pluginentrypoint_new__raw(svm,#plugin,#name),::svm_parameter_structure_get(svm,argv[(index)])))
#endif
#ifndef ARGV_VARIABLE
#define ARGV_VARIABLE(index) ::svm_parameter_variable_get(svm,argv[(index)]);
#endif
#ifndef NEW_VALUE
#define NEW_VALUE(type,value) ::svm_value_##type##_new(svm,(value))
#endif
#ifndef NEW_PLUGIN
#define NEW_PLUGIN(plugin,name,value) ::svm_value_plugin_new(svm,::svm_value_pluginentrypoint_new__raw(svm,#plugin,#name),(value))
#endif
#ifndef NEW_STRUCT
#define NEW_STRUCT(plugin,name,value) ::svm_structure_new(svm,::svm_value_pluginentrypoint_new__raw(svm,#plugin,#name),(value))
#endif
#ifndef NEW_STRING
#define NEW_STRING(raw_string) ::svm_string_new(svm,raw_string.c_str(),raw_string.size())
#endif
#ifndef RAW_STRING
#define RAW_STRING(svm_string) std::string(svm_string.string,svm_string.size)
#endif
#ifndef NEW_BOOLEAN
#define NEW_BOOLEAN(boolean) ((boolean)?TRUE:FALSE)
#endif
#ifndef RAW_BOOLEAN
#define RAW_BOOLEAN(boolean) ((boolean)==TRUE)
#endif
#ifndef NEW_NULL_VALUE
#define NEW_NULL_VALUE(type) ::svm_value_##type##_new_null(svm)
#endif
#ifndef NEW_NULL_PLUGIN
#define NEW_NULL_PLUGIN(plugin,name) ::svm_value_plugin_new_null(svm,::svm_value_pluginentrypoint_new__raw(svm,#plugin,#name))
#endif
#ifndef NEW_NULL_STRUCT
#define NEW_NULL_STRUCT(plugin,name) ::svm_structure_new_null(svm,::svm_value_pluginentrypoint_new__raw(svm,#plugin,#name))
#endif
#ifndef ERROR_INTERNAL
#define ERROR_INTERNAL(irq,message) ::svm_processor_current_raise_error_internal__raw(svm,irq,message)
#endif
#ifndef ERROR_EXTERNAL
#define ERROR_EXTERNAL(plugin,name,message) ::svm_processor_current_raise_error_external__raw(svm,::svm_value_pluginentrypoint_new__raw(svm,#plugin,#name),message)
#endif
#ifndef CONST_PEP
#define CONST_PEP(plugin,name) ::svm_value_pluginentrypoint_new__raw(svm,#plugin,#name)
#endif
#ifndef CURRENT
#define CURRENT(object) ::svm_##object##_get_current(svm)
#endif
#ifndef RETURN
#define RETURN return nullptr
#endif
#ifndef VARIABLE_GLOBAL
#define VARIABLE_GLOBAL(variable) ::svm_variable_scope_set_global(svm,(variable))
#endif
#ifndef VARIABLE_LOCAL
#define VARIABLE_LOCAL(variable) ::svm_variable_scope_set_local(svm,(variable))
#endif
#ifndef VARIABLE_DELETE
#define VARIABLE_DELETE(variable) ::svm_variable_delete(svm,(variable))
#endif

}

#line 63 "../plugins/http/core/http.svm_plugin"
struct Outils
{
	static std::string conversion(const SVM_String& s)
	{
		return std::string(s.string,s.size);
	}
	static std::vector<std::string> decoupage(const std::string& chaine, const std::string& separateur)
	{
		std::vector<std::string> v;
		std::string c = chaine;
		size_t p;
		while((p=c.find(separateur))!=std::string::npos)
		{
			v.push_back(c.substr(0,p));
			c = c.substr(p+separateur.size());
		}
		v.push_back(c);
		return v;
	}
};

struct Http
{
	Http() = default;
	virtual ~Http() {};
	virtual bool requete() const { return false; };
	virtual bool erreur() const { return false; };
	virtual void premiere_ligne(std::ostream& os) const {};
	virtual Http* clone() const { return new Http(*this); };
	friend std::ostream& operator<<(std::ostream& os, const Http& message)
	{
		message.premiere_ligne(os);
		os << "\r\n";
		for(auto e: message._entetes)
		{
			os << e.first << ": " << e.second << "\r\n";
		}
		os << "\r\n" << message._message;
		return os;
	}
	std::map<std::string,std::string> _entetes;
	std::string _message;
	bool _erreur;
	protected:
		Http(const std::string& message)
		:_message(message) {};
};

struct Http_requete : public Http
{
	Http_requete(const std::string& methode, const std::string uri)
	:_methode(methode),_uri(uri) {};
	virtual bool requete() const { return true; }
	virtual void premiere_ligne(std::ostream& os) const
	{
		os << _methode << " " << _uri << " HTTP/1.1";
	};
	virtual Http* clone() const
	{
		return new Http_requete(*this);
	};
	static void verifie(const void *svm, const std::string& methode)
	{
		static std::set<std::string> methodes_valides = {  "GET", "HEAD", "POST", "PUT", "DELETE", "CONNECT", "OPTIONS", "TRACE", "PATCH" };
		auto it = methodes_valides.find(methode);
		if(it==methodes_valides.end())
		{
			std::ostringstream oss;
			oss << "Invalid method " << methode;
			ERROR_EXTERNAL(http,bad_method,oss.str().c_str());
		}
	}

	std::string _methode;
	std::string _uri;
};

struct Http_reponse : public Http
{
	Http_reponse(const unsigned int code_retour)
	:_code_retour(code_retour) {};
	virtual void premiere_ligne(std::ostream& os) const
	{
		os << "HTTP/1.1 " << _code_retour << " " << Http_reponse::nom_code(_code_retour);
	};
	virtual Http* clone() const
	{
		return new Http_reponse(*this);
	};
	static std::string nom_code(unsigned int code_retour)
	{
		switch(code_retour)
		{
			case 100: return "Continue"; break; 
			case 101: return "Switching Protocols"; break; 
			case 200: return "OK"; break; 
			case 201: return "Created"; break; 
			case 202: return "Accepted"; break; 
			case 203: return "Non-Authoritative Information"; break; 
			case 204: return "No Content"; break; 
			case 205: return "Reset Content"; break; 
			case 206: return "Partial Content"; break; 
			case 300: return "Multiple Choices"; break; 
			case 301: return "Moved Permanently"; break; 
			case 302: return "Found"; break; 
			case 303: return "See Other"; break; 
			case 304: return "Not Modified"; break; 
			case 305: return "Use Proxy"; break; 
			case 306: return ""; break; 
			case 307: return "Temporary Redirect"; break; 
			case 400: return "Bad Request"; break; 
			case 401: return "Unauthorized"; break; 
			case 402: return "Payment Required"; break; 
			case 403: return "Forbidden"; break; 
			case 404: return "Not Found"; break; 
			case 405: return "Method Not Allowed"; break; 
			case 406: return "Not Acceptable"; break; 
			case 407: return "Proxy Authentication Required"; break; 
			case 408: return "Request Timeout"; break; 
			case 409: return "Conflict"; break; 
			case 410: return "Gone"; break; 
			case 411: return "Length Required"; break; 
			case 412: return "Precondition Failed"; break; 
			case 413: return "Request Entity Too Large"; break; 
			case 414: return "Request-URI Too Long"; break; 
			case 415: return "Unsupported Media Type"; break; 
			case 416: return "Requested Range Not Satisfiable"; break; 
			case 417: return "Expectation Failed"; break; 
			case 418: return "I am a teapot"; break; 
			case 500: return "Internal Server Error"; break; 
			case 501: return "Not Implemented"; break; 
			case 502: return "Bad Gateway"; break; 
			case 503: return "Service Unavailable"; break; 
			case 504: return "Gateway Timeout"; break; 
			case 505: return "HTTP Version Not Supported"; break; 
		default:
			break;
		}
		return "";
	}
	static void verifie(const void *svm, unsigned int code_retour)
	{
		static std::set<unsigned int> codes_valides = { 100, 101, 200, 201, 202, 203, 204, 205, 206, 300, 301, 302, 303, 304, 305, 306, 307, 400, 401, 402, 403, 404, 405, 406, 407, 408, 409, 410, 411, 412, 413, 414, 415, 416, 417, 418, 500, 501, 502, 503, 504, 505};
		auto it = codes_valides.find(code_retour);
		if(it==codes_valides.end())
		{
			std::ostringstream oss;
			oss << "Invalid return code " << code_retour;
			ERROR_EXTERNAL(http,bad_returncode,oss.str().c_str());
		}
	}
	unsigned int _code_retour;
};

struct Http_erreur : public Http
{
	Http_erreur(const size_t ligne, const std::string& mesg)
	:Http(mesg), _ligne(ligne) {};
	virtual bool erreur() const { return true; };
	virtual Http* clone() const
	{
		return new Http_erreur(*this);
	};
	size_t _ligne;
};
#line 310 "src/plugin.cpp"

extern "C"
{

/* FUNCTION http.protocol_mesg_1_1 STR BLN -> INT ? */

SVM_Variable function_protocol_mesg_1_1(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 242 "../plugins/http/core/http.svm_plugin"
	SVM_String s_tampon = ARGV_VALUE(0,string);
	SVM_Boolean s_eof = ARGV_VALUE(1,boolean);
	std::string tampon = Outils::conversion(s_tampon);
	size_t debut = std::string::npos;
	size_t indice = tampon.find("GET"); if(indice<debut) debut=indice;
	indice = tampon.find("HEAD"); if(indice<debut) debut=indice;
	indice = tampon.find("POST"); if(indice<debut) debut=indice;
	indice = tampon.find("PUT"); if(indice<debut) debut=indice;
	indice = tampon.find("DELETE"); if(indice<debut) debut=indice;
	indice = tampon.find("CONNECT"); if(indice<debut) debut=indice;
	indice = tampon.find("OPTIONS"); if(indice<debut) debut=indice;
	indice = tampon.find("TRACE"); if(indice<debut) debut=indice;
	indice = tampon.find("PATCH"); if(indice<debut) debut=indice;
	indice = tampon.find("HTTP/"); if(indice<debut) debut=indice;
	if(debut==std::string::npos) return NEW_NULL_VALUE(integer);
	if(debut>0) return NEW_VALUE(integer,-debut);
	if(s_eof) return NEW_VALUE(integer,tampon.size());
	size_t t = 4;
	indice = tampon.find("\r\n\r\n");
	if(indice==std::string::npos) 
	{
		indice = tampon.find("\n\n");
		t=2;
	}
	if(indice!=std::string::npos)
	{
		size_t indice_taille = tampon.substr(0,indice).find("Content-Length: ");
		if(indice_taille==std::string::npos)
			return NEW_VALUE(integer,indice+t);
		size_t tt = tampon.find("\n",indice_taille+16);
		long int taille = ::atoi(tampon.substr(indice_taille+16,tt-(indice_taille-16)).c_str());
		if(taille>0)
		{
			return NEW_VALUE(integer,indice+t+taille);
		}
	}
	return NEW_NULL_VALUE(integer);
#line 357 "src/plugin.cpp"
}


/* TYPE http.mesg_1_1 */

struct type_mesg_1_1
{
#line 287 "../plugins/http/core/http.svm_plugin"
	type_mesg_1_1(Http *p)
	:_p(p) {};
	type_mesg_1_1(const type_mesg_1_1& p)
	:_p(p._p->clone())
	{ };
	~type_mesg_1_1() 
	{
		delete _p;
	}
	Http *_p;
#line 376 "src/plugin.cpp"
};

void type_mesg_1_1_delete(const void *svm, void *handler)
{
	type_mesg_1_1 * const object = reinterpret_cast<type_mesg_1_1*>(handler);
	{
#line 299 "../plugins/http/core/http.svm_plugin"

#line 385 "src/plugin.cpp"
	}
	delete object;
}

void* type_mesg_1_1_copy(const void *svm, const void *handler)
{
	const type_mesg_1_1 *object = reinterpret_cast<const type_mesg_1_1*>(handler);
	type_mesg_1_1 *copy = new type_mesg_1_1(*object);
	{
#line 300 "../plugins/http/core/http.svm_plugin"

#line 397 "src/plugin.cpp"
	}
	return copy;
}

void* type_mesg_1_1_constant(const void *svm, const SVM_String value)
{
#line 302 "../plugins/http/core/http.svm_plugin"
	auto lignes = Outils::decoupage(std::string(value.string,value.size),"\r\n");
	if(lignes.size()==0)
	{
		ERROR_EXTERNAL(http,bad_mesg_1_1,"Message incomplete line 1");
	}
	auto premiere_ligne = Outils::decoupage(lignes[0]," ");
	if(premiere_ligne.size()==0)
	{
		ERROR_EXTERNAL(http,bad_mesg_1_1,"Line 1 empty");
	}
	type_mesg_1_1 *message = nullptr;
	if(premiere_ligne[0]=="HTTP/1.1")
	{
		if(premiere_ligne.size()<2)
		{
			ERROR_EXTERNAL(http,bad_mesg_1_1,"Line 1 incomplete for response");
		}
		long int code = ::atoi(premiere_ligne[1].c_str());
		Http_reponse::verifie(svm,code);
		message = new type_mesg_1_1(new Http_reponse(code));
	}
	else
	{
		if(premiere_ligne.size()<3)
		{
			ERROR_EXTERNAL(http,bad_mesg_1_1,"Line 1 incomplete for query");
		}
		Http_requete::verifie(svm,premiere_ligne[0]);
		message = new type_mesg_1_1(new Http_requete(premiere_ligne[0],premiere_ligne[1]));
	}
	
	size_t e = 1;
	for( ; ; ++e)
	{
		if(lignes.size()<=e)
		{
			delete message;
			ERROR_EXTERNAL(http,bad_mesg_1_1,"No header end");
		}
		if(lignes[e]=="")
			break;
		auto entete = Outils::decoupage(lignes[e],": ");
		if(entete.size()!=2)
		{
			delete message;
			ERROR_EXTERNAL(http,bad_mesg_1_1,"Invalid header");
		}
		message->_p->_entetes.insert(std::make_pair(entete[0],entete[1]));
	}
	++e;
	for( ; e<lignes.size() ; ++e)
	{
		message->_p->_message += lignes[e];
		if(e!=lignes.size()-1)
		{
			message->_p->_message += "\r\n";
		}
	}
	return message;
#line 464 "src/plugin.cpp"
}

SVM_String type_mesg_1_1_print(const void *svm, const void *handler)
{
	const type_mesg_1_1 *object = reinterpret_cast<const type_mesg_1_1*>(handler);
	{
#line 364 "../plugins/http/core/http.svm_plugin"
	std::ostringstream oss;
	oss << (*object->_p); 
	return ::svm_string_new(svm,oss.str().c_str(),oss.str().size());
#line 475 "src/plugin.cpp"
	}
}


/* INTERRUPTION http.bad_mesg_1_1 */


/* INSTRUCTION http.decode STR -> http.mesg_1_1 */

SVM_Value instruction_decode(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 380 "../plugins/http/core/http.svm_plugin"
	SVM_String chaine = ARGV_VALUE(0,string);
	void *message = ::type_mesg_1_1_constant(svm,chaine);
	return NEW_PLUGIN(http,mesg_1_1,message);
#line 491 "src/plugin.cpp"
}


/* INSTRUCTION http.encode http.mesg_1_1 -> STR */

SVM_Value instruction_encode(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 394 "../plugins/http/core/http.svm_plugin"
	void *message = ARGV_PLUGIN(0,http,mesg_1_1);
	SVM_String chaine = type_mesg_1_1_print(svm,message);
	return NEW_VALUE(string,chaine);
#line 503 "src/plugin.cpp"
}


/* INTERRUPTION http.bad_returncode */


/* INSTRUCTION http.new ( 'QUERY' [ 'GET' 'HEAD' 'POST' 'PUT' 'DELETE' 'CONNECT' 'OPTIONS' 'TRACE' 'PATCH' ]:method STR:uri STR:hostname | 'REPLY' INT:return_code ) -> http.mesg_1_1 */

SVM_Value instruction_new(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 409 "../plugins/http/core/http.svm_plugin"
	auto type = ARGV_KEYWORD(0);
	type_mesg_1_1 *message = nullptr;
	if(type=="QUERY")
	{
		auto methode = ARGV_KEYWORD(1);
		SVM_String uri = ARGV_VALUE(2,string);
		SVM_String hote = ARGV_VALUE(3,string);
		message=new type_mesg_1_1(new Http_requete(methode,Outils::conversion(uri)));
		message->_p->_entetes.insert(std::make_pair("Host",Outils::conversion(hote)));
	}
	else
	{
		auto code_retour = ARGV_VALUE(1,integer);
		Http_reponse::verifie(svm,code_retour);
		message=new type_mesg_1_1(new Http_reponse(code_retour));
	}
	return NEW_PLUGIN(http,mesg_1_1,message);
#line 532 "src/plugin.cpp"
}


/* INSTRUCTION http.query http.mesg_1_1 -> BLN */

SVM_Value instruction_query(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 436 "../plugins/http/core/http.svm_plugin"
	type_mesg_1_1 *message = ARGV_PLUGIN(0,http,mesg_1_1);
	return ::svm_value_boolean_new__raw(svm,message->_p->requete());
#line 543 "src/plugin.cpp"
}


/* INSTRUCTION http.reply http.mesg_1_1 -> BLN */

SVM_Value instruction_reply(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 443 "../plugins/http/core/http.svm_plugin"
	type_mesg_1_1 *message = ARGV_PLUGIN(0,http,mesg_1_1);
	return ::svm_value_boolean_new__raw(svm,not message->_p->requete());
#line 554 "src/plugin.cpp"
}


/* INTERRUPTION http.bad_mesgtype */


/* INSTRUCTION http.get_method http.mesg_1_1 -> STR */

SVM_Value instruction_get_method(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 453 "../plugins/http/core/http.svm_plugin"
	type_mesg_1_1 *message = ARGV_PLUGIN(0,http,mesg_1_1);
	Http_requete *requete = dynamic_cast<Http_requete*>(message->_p);
	if(not requete)
	{
		ERROR_EXTERNAL(http,bad_mesgtype,"Response has no method");
	}
	return ::svm_value_string_new__buffer(svm,requete->_methode.c_str(),requete->_methode.size());
#line 573 "src/plugin.cpp"
}


/* INSTRUCTION http.set_method MUTABLE http.mesg_1_1 STR */

SVM_Value instruction_set_method(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 465 "../plugins/http/core/http.svm_plugin"
	type_mesg_1_1 *message = ARGV_PLUGIN(0,http,mesg_1_1);
	Http_requete *requete = dynamic_cast<Http_requete*>(message->_p);
	SVM_String methode = ARGV_VALUE(1,string);
	if(not requete)
	{
		ERROR_EXTERNAL(http,bad_mesgtype,"Response has no method");
	}
	Http_requete::verifie(svm,Outils::conversion(methode));
	requete->_methode = Outils::conversion(methode);
#line 591 "src/plugin.cpp"
	return nullptr;
}


/* INSTRUCTION http.get_uri http.mesg_1_1 -> STR */

SVM_Value instruction_get_uri(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 479 "../plugins/http/core/http.svm_plugin"
	type_mesg_1_1 *message = ARGV_PLUGIN(0,http,mesg_1_1);
	Http_requete *requete = dynamic_cast<Http_requete*>(message->_p);
	if(not requete)
	{
		ERROR_EXTERNAL(http,bad_mesgtype,"Response has no URI");
	}
	return ::svm_value_string_new__buffer(svm,requete->_uri.c_str(),requete->_uri.size());
#line 608 "src/plugin.cpp"
}


/* INSTRUCTION http.set_uri MUTABLE http.mesg_1_1 STR */

SVM_Value instruction_set_uri(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 491 "../plugins/http/core/http.svm_plugin"
	type_mesg_1_1 *message = ARGV_PLUGIN(0,http,mesg_1_1);
	Http_requete *requete = dynamic_cast<Http_requete*>(message->_p);
	SVM_String uri = ARGV_VALUE(1,string);
	if(not requete)
	{
		ERROR_EXTERNAL(http,bad_mesgtype,"Response has no URI");
	}
	requete->_uri = Outils::conversion(uri);
#line 625 "src/plugin.cpp"
	return nullptr;
}


/* INSTRUCTION http.get_returncode http.mesg_1_1 -> INT */

SVM_Value instruction_get_returncode(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 504 "../plugins/http/core/http.svm_plugin"
	type_mesg_1_1 *message = ARGV_PLUGIN(0,http,mesg_1_1);
	Http_reponse *reponse = dynamic_cast<Http_reponse*>(message->_p);
	if(not reponse)
	{
		ERROR_EXTERNAL(http,bad_mesgtype,"Request has no return code");
	}
	return NEW_VALUE(integer,reponse->_code_retour);
#line 642 "src/plugin.cpp"
}


/* INSTRUCTION http.set_returncode MUTABLE http.mesg_1_1 INT */

SVM_Value instruction_set_returncode(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 516 "../plugins/http/core/http.svm_plugin"
	type_mesg_1_1 *message = ARGV_PLUGIN(0,http,mesg_1_1);
	Http_reponse *reponse = dynamic_cast<Http_reponse*>(message->_p);
	auto code_retour = ARGV_VALUE(1,integer);
	if(not reponse)
	{
		ERROR_EXTERNAL(http,bad_mesgtype,"Request has no return code");
	}
	Http_reponse::verifie(svm,code_retour);
	reponse->_code_retour = code_retour;
#line 660 "src/plugin.cpp"
	return nullptr;
}


/* INSTRUCTION http.get_header http.mesg_1_1 STR:key -> STR ?:value */

SVM_Value instruction_get_header(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 530 "../plugins/http/core/http.svm_plugin"
	type_mesg_1_1 *message = ARGV_PLUGIN(0,http,mesg_1_1);
	SVM_String clef = ::svm_value_string_get(svm,::svm_parameter_value_get(svm,argv[1]));
	auto it = message->_p->_entetes.find(Outils::conversion(clef));
	if(it==message->_p->_entetes.end())
	{
		return NEW_NULL_VALUE(string);
	}
	return ::svm_value_string_new__buffer(svm,it->second.c_str(),it->second.size());
#line 678 "src/plugin.cpp"
}


/* INSTRUCTION http.set_header MUTABLE http.mesg_1_1 STR:key STR ?:value */

SVM_Value instruction_set_header(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 549 "../plugins/http/core/http.svm_plugin"
	type_mesg_1_1 *message = ARGV_PLUGIN(0,http,mesg_1_1);
	SVM_String clef = ARGV_VALUE(1,string);
	if(argc==2)
	{
		auto it = message->_p->_entetes.find(Outils::conversion(clef));
		if(it!=message->_p->_entetes.end())
		{
			message->_p->_entetes.erase(it);
		}
	}
	else
	{
		SVM_String valeur = ARGV_VALUE(2,string);
		message->_p->_entetes[Outils::conversion(clef)] = Outils::conversion(valeur);
	}
#line 702 "src/plugin.cpp"
	return nullptr;
}


/* INSTRUCTION http.get_headers http.mesg_1_1 -> PTR */

SVM_Value instruction_get_headers(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 575 "../plugins/http/core/http.svm_plugin"
	type_mesg_1_1 *message = ARGV_PLUGIN(0,http,mesg_1_1);
	SVM_Memory_Zone zone_tableau_principal = ::svm_memory_zone_new(svm);
	::svm_memory_zone_append_internal__raw(svm,zone_tableau_principal,POINTER,message->_p->_entetes.size());
	SVM_Value_Pointer tableau_principal = ::svm_memory_allocate(svm,CURRENT(kernel),zone_tableau_principal);
	SVM_Address adresse = ::svm_value_pointer_get_address(svm,tableau_principal);
	for(auto e: message->_p->_entetes)
	{
		SVM_Memory_Zone zone_tableau_secondaire = ::svm_memory_zone_new(svm);
		::svm_memory_zone_append_internal__raw(svm,zone_tableau_secondaire,STRING,2);
		SVM_Value_Pointer tableau_secondaire = ::svm_memory_allocate(svm,CURRENT(kernel),zone_tableau_secondaire);
		SVM_Address a = ::svm_value_pointer_get_address(svm,tableau_secondaire);
		::svm_memory_write_address(svm,CURRENT(kernel),a,::svm_value_string_new__buffer(svm,e.first.c_str(),e.first.size()));
		::svm_memory_write_address(svm,CURRENT(kernel),a+1,::svm_value_string_new__buffer(svm,e.second.c_str(),e.second.size()));
		::svm_memory_write_address(svm,CURRENT(kernel),adresse,tableau_secondaire);
		++adresse;
	}
	return tableau_principal;
#line 729 "src/plugin.cpp"
}


/* INSTRUCTION http.set_headers MUTABLE http.mesg_1_1 PTR */

SVM_Value instruction_set_headers(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 605 "../plugins/http/core/http.svm_plugin"
	type_mesg_1_1 *message = ARGV_PLUGIN(0,http,mesg_1_1);
	std::map<std::string,std::string> entetes;
	SVM_Value_Pointer tableau_principal = ::svm_parameter_value_get(svm,argv[1]);
	SVM_Address adresse = ::svm_value_pointer_get_address(svm,tableau_principal);
	SVM_Size taille = ::svm_value_pointer_get_size(svm,tableau_principal);
	for(SVM_Address a = adresse ; a<(adresse+taille) ; ++a)
	{
		SVM_Value_Pointer tableau_secondaire = ::svm_memory_read_address_type_internal(svm,::svm_kernel_get_current(svm),a,POINTER);
		SVM_Address aa = ::svm_value_pointer_get_address(svm,tableau_secondaire);
		SVM_Value_String clef = ::svm_memory_read_address_type_internal(svm,::svm_kernel_get_current(svm),aa,STRING);
		SVM_Value_String valeur = ::svm_memory_read_address_type_internal(svm,::svm_kernel_get_current(svm),aa+1,STRING);
		entetes[Outils::conversion(::svm_value_string_get(svm,clef))] = Outils::conversion(::svm_value_string_get(svm,valeur));
	}
	message->_p->_entetes.swap(entetes);
#line 752 "src/plugin.cpp"
	return nullptr;
}


/* INSTRUCTION http.get_payload http.mesg_1_1 -> STR */

SVM_Value instruction_get_payload(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 629 "../plugins/http/core/http.svm_plugin"
	type_mesg_1_1 *message = ARGV_PLUGIN(0,http,mesg_1_1);
	return ::svm_value_string_new__buffer(svm,message->_p->_message.c_str(),message->_p->_message.size());
#line 764 "src/plugin.cpp"
}


/* INSTRUCTION http.set_payload MUTABLE http.mesg_1_1 STR 'LENGTH' ? */

SVM_Value instruction_set_payload(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 636 "../plugins/http/core/http.svm_plugin"
	type_mesg_1_1 *message = ARGV_PLUGIN(0,http,mesg_1_1);
	SVM_String mesg = ARGV_VALUE(1,string);
	message->_p->_message = Outils::conversion(mesg);
	if(argc>2)
	{
		std::ostringstream oss;
		oss << mesg.size;
		message->_p->_entetes["Content-Length"] = oss.str();
	}
#line 782 "src/plugin.cpp"
	return nullptr;
}


/* Generic handling functions */

}
