/*
 * Copyright (C) 2022-05-24  Julien BRUGUIER
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#pragma once

#include <iostream>
#include <sstream>
#include <memory>
#include <vector>
#include <set>
#include <map>

namespace JSON
{

template<typename E>
struct Compare
{
	bool operator() (const std::weak_ptr<E>& l, const std::weak_ptr<E>& r)
	{
		return l.lock()<r.lock();
	}
};

struct Value : public std::enable_shared_from_this<Value>
{
	typedef std::shared_ptr<Value> SP;
	virtual ~Value() {}
	virtual Value::SP clone() const = 0;

	template<typename oStream>
	friend oStream& operator<<(oStream& os, const Value::SP& e)
	{
		e->print(os);
		return os;
	}
	virtual void print(std::ostream& os) const = 0;
};

struct Null : public Value
{
	typedef std::shared_ptr<Null> SP;
	Null() = default;
	virtual ~Null() {}
	virtual Value::SP clone() const override
	{
		return strict_clone();
	}
	Null::SP strict_clone() const
	{
		return std::make_shared<Null>();
	}
	virtual void print(std::ostream& os) const override
	{
		os << "null";
	}
};

struct Integer : public Value
{
	typedef std::shared_ptr<Integer> SP;
	Integer(const long long int i)
	:_i(i) {}
	virtual ~Integer() {}
	virtual Value::SP clone() const override
	{
		return strict_clone();
	}
	Integer::SP strict_clone() const
	{
		return std::make_shared<Integer>(_i);
	}
	virtual void print(std::ostream& os) const override
	{
		os << _i;
	}

	long long int _i;
};

struct String : public Value
{
	typedef std::shared_ptr<String> SP;
	String(const std::string& s)
	:_s(s) {}
	virtual ~String() {}
	virtual Value::SP clone() const override
	{
		return strict_clone();
	}
	String::SP strict_clone() const
	{
		return std::make_shared<String>(_s);
	}
	virtual void print(std::ostream& os) const override
	{
		os << "\"" << _s << "\"";
	}

	std::string _s;
};

struct Boolean : public Value
{
	typedef std::shared_ptr<Boolean> SP;
	Boolean(const bool b)
	:_b(b) {}
	virtual ~Boolean() {}
	virtual Value::SP clone() const override
	{
		return strict_clone();
	}
	Boolean::SP strict_clone() const
	{
		return std::make_shared<Boolean>(_b);
	}
	virtual void print(std::ostream& os) const override
	{
		os << (_b?"true":"false");
	}

	bool _b;
};

struct Object : public Value
{
	typedef std::shared_ptr<Object> SP;
	virtual ~Object() {}
	virtual Value::SP clone() const override
	{
		return strict_clone();
	}
	Object::SP strict_clone() const
	{
		auto o = std::make_shared<Object>();
		for(const auto& m:_members)
		{
			o->_members.insert(std::make_pair(m.first,m.second->clone()));
		}
		return o;
	}
	virtual void print(std::ostream& os) const override
	{
		os << "{";
		bool cont=false;
		for(const auto& m:_members)
		{
			if(cont)
			{
				os << ",";
			}
			cont=true;
			os << " \"" << m.first << "\": ";
			m.second->print(os);
		}
		os << " }";
	}

	std::map<std::string, Value::SP> _members;
};

struct Array : public Value
{
	typedef std::shared_ptr<Array> SP;
	virtual ~Array() {}
	virtual Value::SP clone() const override
	{
		return strict_clone();
	}
	Array::SP strict_clone() const
	{
		auto o = std::make_shared<Array>();
		for(const auto& e:_elements)
		{
			o->_elements.push_back(e->clone());
		}
		return o;
	}
	virtual void print(std::ostream& os) const override
	{
		os << "[";
		bool cont=false;
		for(const auto& e:_elements)
		{
			if(cont)
			{
				os << ",";
			}
			cont=true;
			os << " ";
			e->print(os);
		}
		os << " ]";
	}

	std::vector<Value::SP> _elements;
};

}
