/*
 * Copyright (C) 2022-01-31  Julien BRUGUIER
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */


#include <string>
#include <sstream>

#line 49 "../plugins/shell/core/shell.svm_plugin"
#include <sys/types.h>
#include <unistd.h>
#include <sstream>
#include <string.h>
#include <vector>
#include <string>
#line 30 "src/plugin.cpp"

#include <src/plugin.h>

extern "C"
{

void plugin_configure(void *plugin)
{
	::svm_plugin_configure(plugin,
		"PLUGIN shell \n"
		"USE \n"
		"	TYPE com.device \n"
		"	INTERRUPTION com.interrupted \n"
		"DEFINE \n"
		"	STRUCT shell.command \n"
		"	FUNCTION shell.device_command_open [ < > <> ] [ > >> ] ? 'FREE' ? ( STR + | PTR ) -> $shell.command \n"
		"	FUNCTION shell.device_command_print $shell.command -> STR \n"
		"	FUNCTION shell.device_command_read $shell.command -> STR ? \n"
		"	FUNCTION shell.device_command_write $shell.command STR \n"
		"	FUNCTION shell.device_command_idle $shell.command MUTABLE INT 3 \n"
		"	FUNCTION shell.device_command_command $shell.command . * -> VALUE ? \n"
		"	FUNCTION shell.device_command_close $shell.command -> BLN \n"
		,SVM_API_SIGNATURE,SVM_VERSION);
}

#ifndef ARGV_VALUE
#define ARGV_VALUE(index,type) ::svm_value_##type##_get(svm,::svm_parameter_value_get(svm,argv[(index)]))
#endif
#ifndef ARGV_PLUGIN
#define ARGV_PLUGIN(index,plugin,name) reinterpret_cast<type_##name*>(::svm_value_plugin_get_internal(svm,::svm_parameter_value_get(svm,argv[(index)])))
#endif
#ifndef ARGV_MARKER
#define ARGV_MARKER(index) std::string(::svm_parameter_marker_get(svm,argv[(index)]).string)
#endif
#ifndef ARGV_KEYWORD
#define ARGV_KEYWORD(index) std::string(::svm_parameter_keyword_get(svm,argv[(index)]).string)
#endif
#ifndef ARGV_STRUCT
#define ARGV_STRUCT(index,plugin,name) reinterpret_cast<struct_##name*>(::svm_structure_get_internal(svm,::svm_value_pluginentrypoint_new__raw(svm,#plugin,#name),::svm_parameter_structure_get(svm,argv[(index)])))
#endif
#ifndef ARGV_VARIABLE
#define ARGV_VARIABLE(index) ::svm_parameter_variable_get(svm,argv[(index)]);
#endif
#ifndef NEW_VALUE
#define NEW_VALUE(type,value) ::svm_value_##type##_new(svm,(value))
#endif
#ifndef NEW_PLUGIN
#define NEW_PLUGIN(plugin,name,value) ::svm_value_plugin_new(svm,::svm_value_pluginentrypoint_new__raw(svm,#plugin,#name),(value))
#endif
#ifndef NEW_STRUCT
#define NEW_STRUCT(plugin,name,value) ::svm_structure_new(svm,::svm_value_pluginentrypoint_new__raw(svm,#plugin,#name),(value))
#endif
#ifndef NEW_STRING
#define NEW_STRING(raw_string) ::svm_string_new(svm,raw_string.c_str(),raw_string.size())
#endif
#ifndef RAW_STRING
#define RAW_STRING(svm_string) std::string(svm_string.string,svm_string.size)
#endif
#ifndef NEW_BOOLEAN
#define NEW_BOOLEAN(boolean) ((boolean)?TRUE:FALSE)
#endif
#ifndef RAW_BOOLEAN
#define RAW_BOOLEAN(boolean) ((boolean)==TRUE)
#endif
#ifndef NEW_NULL_VALUE
#define NEW_NULL_VALUE(type) ::svm_value_##type##_new_null(svm)
#endif
#ifndef NEW_NULL_PLUGIN
#define NEW_NULL_PLUGIN(plugin,name) ::svm_value_plugin_new_null(svm,::svm_value_pluginentrypoint_new__raw(svm,#plugin,#name))
#endif
#ifndef NEW_NULL_STRUCT
#define NEW_NULL_STRUCT(plugin,name) ::svm_structure_new_null(svm,::svm_value_pluginentrypoint_new__raw(svm,#plugin,#name))
#endif
#ifndef ERROR_INTERNAL
#define ERROR_INTERNAL(irq,message) ::svm_processor_current_raise_error_internal__raw(svm,irq,message)
#endif
#ifndef ERROR_EXTERNAL
#define ERROR_EXTERNAL(plugin,name,message) ::svm_processor_current_raise_error_external__raw(svm,::svm_value_pluginentrypoint_new__raw(svm,#plugin,#name),message)
#endif
#ifndef CONST_PEP
#define CONST_PEP(plugin,name) ::svm_value_pluginentrypoint_new__raw(svm,#plugin,#name)
#endif
#ifndef CURRENT
#define CURRENT(object) ::svm_##object##_get_current(svm)
#endif
#ifndef RETURN
#define RETURN return nullptr
#endif
#ifndef VARIABLE_GLOBAL
#define VARIABLE_GLOBAL(variable) ::svm_variable_scope_set_global(svm,(variable))
#endif
#ifndef VARIABLE_LOCAL
#define VARIABLE_LOCAL(variable) ::svm_variable_scope_set_local(svm,(variable))
#endif
#ifndef VARIABLE_DELETE
#define VARIABLE_DELETE(variable) ::svm_variable_delete(svm,(variable))
#endif

}

#line 59 "../plugins/shell/core/shell.svm_plugin"
	struct RAII
	{
		RAII()
		{
			_in[0]=-1;
			_in[1]=-1;
			_out[0]=-1;
			_out[1]=-1;
			_err[0]=-1;
			_err[1]=-1;
		}
		void commit()
		{
			_in[0]=-1;
			_in[1]=-1;
			_out[0]=-1;
			_out[1]=-1;
			_err[0]=-1;
			_err[1]=-1;
		}
		int _in[2];
		int _out[2];
		int _err[2];
		~RAII()
		{
			if(_in[0]>=0) { ::close(_in[0]); }
			if(_in[1]>=0) { ::close(_in[1]); }
			if(_out[0]>=0) { ::close(_out[0]); }
			if(_out[1]>=0) { ::close(_out[1]); }
			if(_err[0]>=0) { ::close(_err[0]); }
			if(_err[1]>=0) { ::close(_err[1]); }
		}
	};
#line 165 "src/plugin.cpp"

extern "C"
{

/* STRUCT shell.command */

struct struct_command
{
#line 112 "../plugins/shell/core/shell.svm_plugin"
	struct_command(const int in, const int out, const int error, const pid_t child, const std::vector<std::string>& command)
	:_in(in),_out(out),_err(error),_child(child), _command(command) {}
	int _in;
	int _out;
	int _err;
	pid_t _child;
	std::vector<std::string> _command;
#line 182 "src/plugin.cpp"
};

void struct_command_delete(const void *svm, void *handler)
{
	struct_command * const object = reinterpret_cast<struct_command*>(handler);
	{
#line 121 "../plugins/shell/core/shell.svm_plugin"

#line 191 "src/plugin.cpp"
	}
	delete object;
}


/* FUNCTION shell.device_command_open [ < > <> ] [ > >> ] ? 'FREE' ? ( STR + | PTR ) -> $shell.command */

SVM_Variable function_device_command_open(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 125 "../plugins/shell/core/shell.svm_plugin"
	int begin = 1;
	std::string s = ARGV_MARKER(0);
	bool enable_in = ((s==">") or (s=="<>"));
	bool enable_out = ((s=="<") or (s=="<>"));
	bool enable_err = false;
	bool enable_lock = true;
	bool merge_err = false;
	if(::svm_parameter_type_is_marker(svm,argv[begin]))
	{
		enable_err = true;
		merge_err = (ARGV_MARKER(begin)==">>");
		++begin;
	}
	if(::svm_parameter_type_is_keyword(svm,argv[begin]))
	{
		enable_lock = false;
		++begin;
	}
	std::vector<std::string> cmd;
	SVM_Value pp = ::svm_parameter_value_get(svm,argv[begin]);
	if(::svm_value_type_is_pointer(svm,pp))
	{
		SVM_Address b = ::svm_value_pointer_get_address(svm,pp);
		SVM_Address e = b+::svm_value_pointer_get_size(svm,pp);
		for(SVM_Address a=b ; a<e ; ++a)
		{
			SVM_String ss = ::svm_value_string_get(svm,::svm_memory_read_address_type_internal(svm,CURRENT(kernel),a,STRING));
			cmd.push_back(std::string(ss.string,ss.size));
		}
	}
	else
	{
		for(size_t i=begin ; i<argc ; ++i)
		{
			SVM_String ss = ARGV_VALUE(i,string);
			cmd.push_back(std::string(ss.string,ss.size));
		}
	}
	RAII raii;
	::pipe(raii._in);
	::pipe(raii._out);
	::pipe(raii._err);

	SVM_Process_Lock lock = nullptr;
	
	if(enable_lock)
	{
		lock = ::svm_process_ownership_lock_critical(svm);

		if(not lock)
		{
			ERROR_INTERNAL(FAILURE,"Unable to acquire lock");
		}
	}
	
	pid_t p = ::fork();
	if(p<0)
	{
		ERROR_INTERNAL(FAILURE,"Unable to fork");
	}
	if(p==0)
	{
		::close(raii._in[1]);
		::close(raii._out[0]);
		::close(raii._err[0]);
		::close(0);
		::close(1);
		::close(2);
		if(enable_in)
		{
			::dup2(raii._in[0],0);
		}
		if(enable_out)
		{
			::dup2(raii._out[1],1);
		}
		if(enable_err)
		{
			if(merge_err)
			{
				::dup2(raii._out[1],2);
			}
			else
			{
				::dup2(raii._err[1],2);
			}
		}
		::close(raii._in[0]);
		::close(raii._out[1]);
		::close(raii._err[1]);
		char * *params = new char *[cmd.size()+1];
		size_t j=0;
		for(const auto& ss: cmd)
		{
			params[j++] = ::strdup(ss.c_str());
		}
		params[j]=nullptr;
		raii.commit();
		::execvp(params[0],params);
		exit(1);
	}
	else
	{
		::close(raii._in[0]);
		::close(raii._out[1]);
		::close(raii._err[1]);
		if(not enable_in)
		{
			::close(raii._in[1]);
			raii._in[1] = -1;
		}
		if(not enable_out)
		{
			::close(raii._out[0]);
			raii._out[0] = -1;
		}
		if(not enable_err)
		{
			::close(raii._err[0]);
			raii._err[0] = -1;
		}
		struct_command *command = new struct_command(raii._in[1],raii._out[0],raii._err[0],p,cmd);
		raii.commit();
		return NEW_STRUCT(shell,command,command);
	}
#line 327 "src/plugin.cpp"
}


/* FUNCTION shell.device_command_print $shell.command -> STR */

SVM_Variable function_device_command_print(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 273 "../plugins/shell/core/shell.svm_plugin"
	const struct_command *command = ARGV_STRUCT(0,shell,command);
	std::ostringstream oss;
	oss << "$" ;
	for(const auto& c: command->_command)
	{
		oss << " " << c;
	}
	return ::svm_value_string_new__buffer(svm,oss.str().c_str(),oss.str().size());
#line 344 "src/plugin.cpp"
}


/* FUNCTION shell.device_command_read $shell.command -> STR ? */

SVM_Variable function_device_command_read(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 286 "../plugins/shell/core/shell.svm_plugin"
	struct_command *command = ARGV_STRUCT(0,shell,command);
	if(command->_out<0)
	{
		ERROR_INTERNAL(DEVICE,"Unable to read from shell command.");
	}
	char buffer[1025];
	::svm_process_pause(svm);
	::svm_process_interruptionnotification_enable(svm,CURRENT(process));
	int r = ::read(command->_out,buffer,1024);
	::svm_process_interruptionnotification_disable(svm,CURRENT(process));
	::svm_process_resume(svm);
	if(r<0)
	{
		if(errno==EINTR)
		{
			ERROR_EXTERNAL(com,interrupted,"Read interrupted.");
		}
		ERROR_INTERNAL(DEVICE,"Failed to read from shell command.");
	}
	if(r==0)
	{
		return ::svm_value_string_new_null(svm);
	}
	return ::svm_value_string_new__buffer(svm,buffer,r);
#line 377 "src/plugin.cpp"
}


/* FUNCTION shell.device_command_write $shell.command STR */

SVM_Variable function_device_command_write(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 315 "../plugins/shell/core/shell.svm_plugin"
	struct_command *command = ARGV_STRUCT(0,shell,command);
	if(command->_in<0)
	{
		ERROR_INTERNAL(DEVICE,"Unable to write to shell command.");
	}
	SVM_String buffer = ARGV_VALUE(1,string);
	ssize_t w = ::write(command->_in,buffer.string,buffer.size);
	if(w<0)
	{
		ERROR_INTERNAL(DEVICE,"Failed to write to shell command.");
	}
	if(static_cast<size_t>(w)!=buffer.size)
	{
		ERROR_INTERNAL(DEVICE,"Failed to write everything to shell command.");
	}
#line 401 "src/plugin.cpp"
	return nullptr;
}


/* FUNCTION shell.device_command_idle $shell.command MUTABLE INT 3 */

SVM_Variable function_device_command_idle(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 335 "../plugins/shell/core/shell.svm_plugin"
	struct_command *command = ARGV_STRUCT(0,shell,command);
	if(command->_out>=0)
	{
		SVM_Value_Integer i = ::svm_parameter_value_get(svm,argv[1]);
		::svm_value_integer_set(svm,i,command->_out);
	}
	if(command->_in>=0)
	{
		SVM_Value_Integer i = ::svm_parameter_value_get(svm,argv[2]);
		::svm_value_integer_set(svm,i,command->_in);
	}
#line 422 "src/plugin.cpp"
	return nullptr;
}


/* FUNCTION shell.device_command_command $shell.command . * -> VALUE ? */

SVM_Variable function_device_command_command(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 351 "../plugins/shell/core/shell.svm_plugin"
	struct_command *command = ARGV_STRUCT(0,shell,command);
	if(argc==1)
	{
		ERROR_INTERNAL(DEVICE,"Missing command");
	}
	if(not ::svm_parameter_type_is_keyword(svm,argv[1]))
	{
		ERROR_INTERNAL(DEVICE,"Invalid command");
	}
	std::string k = ARGV_KEYWORD(1);
	if(k=="ERROR")
	{
		std::ostringstream oss;
		char buffer[1025];
		ssize_t r=0;
		while((r=::read(command->_err,buffer,1024))>0)
		{
			oss << std::string(buffer,r);
		}
		if(r<0)
		{
			ERROR_INTERNAL(DEVICE,"Failed to read error from shell command.");
		}
		return ::svm_value_string_new__buffer(svm,oss.str().c_str(),oss.str().size());
	}
	if(k=="CLOSE")
	{
		if(command->_in<0)
		{
			ERROR_INTERNAL(DEVICE,"stdin already closed.");
		}
		::close(command->_in);
		command->_in = -1;
		return ::svm_value_string_new_null(svm);
	}
	ERROR_INTERNAL(DEVICE,"Invalid command");
	return ::svm_value_string_new_null(svm);
#line 469 "src/plugin.cpp"
}


/* FUNCTION shell.device_command_close $shell.command -> BLN */

SVM_Variable function_device_command_close(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 402 "../plugins/shell/core/shell.svm_plugin"
	return NEW_VALUE(boolean,TRUE);
#line 479 "src/plugin.cpp"
}


/* Generic handling functions */

}
