/*
 * Copyright (C) 2022-08-11  Julien BRUGUIER, Julien TALLON, Francois GOASMAT
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */


#include <string>
#include <sstream>

#line 67 "../plugins/sqlite/core/sqlite.svm_plugin"
#include <iostream>
#include <vector>
#include <map>
#include <memory>
#include <mutex>
#include <sqlite3.h>
#line 30 "src/plugin.cpp"

#include <src/plugin.h>

extern "C"
{

void plugin_configure(void *plugin)
{
	::svm_plugin_configure(plugin,
		"PLUGIN sqlite \n"
		"DEFINE \n"
		"	INTERRUPTION sqlite.error \n"
		"	INTERRUPTION sqlite.busy \n"
		"	TYPE sqlite.database \n"
		"	TYPE sqlite.query \n"
		"	SYSTEM INSTRUCTION sqlite.database STR:file [ 'RO' 'RW' ]:mode -> sqlite.database \n"
		"	INSTRUCTION sqlite.query sqlite.database STR:query -> sqlite.query \n"
		"	INSTRUCTION sqlite.init MUTABLE sqlite.query ( VALUE | KEYWORD = VALUE ) *:parameters \n"
		"	INSTRUCTION sqlite.exec MUTABLE sqlite.query PTR ? -> PTR ? \n"
		"	INSTRUCTION sqlite.var STR:name VALUE ?:value \n"
		,SVM_API_SIGNATURE,SVM_VERSION);
}

#ifndef ARGV_VALUE
#define ARGV_VALUE(index,type) ::svm_value_##type##_get(svm,::svm_parameter_value_get(svm,argv[(index)]))
#endif
#ifndef ARGV_PLUGIN
#define ARGV_PLUGIN(index,plugin,name) reinterpret_cast<type_##name*>(::svm_value_plugin_get_internal(svm,::svm_parameter_value_get(svm,argv[(index)])))
#endif
#ifndef ARGV_MARKER
#define ARGV_MARKER(index) std::string(::svm_parameter_marker_get(svm,argv[(index)]).string)
#endif
#ifndef ARGV_KEYWORD
#define ARGV_KEYWORD(index) std::string(::svm_parameter_keyword_get(svm,argv[(index)]).string)
#endif
#ifndef ARGV_STRUCT
#define ARGV_STRUCT(index,plugin,name) reinterpret_cast<struct_##name*>(::svm_structure_get_internal(svm,::svm_value_pluginentrypoint_new__raw(svm,#plugin,#name),::svm_parameter_structure_get(svm,argv[(index)])))
#endif
#ifndef ARGV_VARIABLE
#define ARGV_VARIABLE(index) ::svm_parameter_variable_get(svm,argv[(index)]);
#endif
#ifndef NEW_VALUE
#define NEW_VALUE(type,value) ::svm_value_##type##_new(svm,(value))
#endif
#ifndef NEW_PLUGIN
#define NEW_PLUGIN(plugin,name,value) ::svm_value_plugin_new(svm,::svm_value_pluginentrypoint_new__raw(svm,#plugin,#name),(value))
#endif
#ifndef NEW_STRUCT
#define NEW_STRUCT(plugin,name,value) ::svm_structure_new(svm,::svm_value_pluginentrypoint_new__raw(svm,#plugin,#name),(value))
#endif
#ifndef NEW_STRING
#define NEW_STRING(raw_string) ::svm_string_new(svm,raw_string.c_str(),raw_string.size())
#endif
#ifndef RAW_STRING
#define RAW_STRING(svm_string) std::string(svm_string.string,svm_string.size)
#endif
#ifndef NEW_BOOLEAN
#define NEW_BOOLEAN(boolean) ((boolean)?TRUE:FALSE)
#endif
#ifndef RAW_BOOLEAN
#define RAW_BOOLEAN(boolean) ((boolean)==TRUE)
#endif
#ifndef NEW_NULL_VALUE
#define NEW_NULL_VALUE(type) ::svm_value_##type##_new_null(svm)
#endif
#ifndef NEW_NULL_PLUGIN
#define NEW_NULL_PLUGIN(plugin,name) ::svm_value_plugin_new_null(svm,::svm_value_pluginentrypoint_new__raw(svm,#plugin,#name))
#endif
#ifndef NEW_NULL_STRUCT
#define NEW_NULL_STRUCT(plugin,name) ::svm_structure_new_null(svm,::svm_value_pluginentrypoint_new__raw(svm,#plugin,#name))
#endif
#ifndef ERROR_INTERNAL
#define ERROR_INTERNAL(irq,message) ::svm_processor_current_raise_error_internal__raw(svm,irq,message)
#endif
#ifndef ERROR_EXTERNAL
#define ERROR_EXTERNAL(plugin,name,message) ::svm_processor_current_raise_error_external__raw(svm,::svm_value_pluginentrypoint_new__raw(svm,#plugin,#name),message)
#endif
#ifndef CONST_PEP
#define CONST_PEP(plugin,name) ::svm_value_pluginentrypoint_new__raw(svm,#plugin,#name)
#endif
#ifndef CURRENT
#define CURRENT(object) ::svm_##object##_get_current(svm)
#endif
#ifndef RETURN
#define RETURN return nullptr
#endif
#ifndef VARIABLE_GLOBAL
#define VARIABLE_GLOBAL(variable) ::svm_variable_scope_set_global(svm,(variable))
#endif
#ifndef VARIABLE_LOCAL
#define VARIABLE_LOCAL(variable) ::svm_variable_scope_set_local(svm,(variable))
#endif
#ifndef VARIABLE_DELETE
#define VARIABLE_DELETE(variable) ::svm_variable_delete(svm,(variable))
#endif

}

#line 90 "../plugins/sqlite/core/sqlite.svm_plugin"
struct Value
{
	typedef std::shared_ptr<Value> SP;
	virtual int bind(sqlite3_stmt* q, int i) const = 0;
};

struct Integer : public Value
{
	Integer(const long long int i)
	:_i(i) {}
	virtual int bind(sqlite3_stmt* q, int i) const override
	{
		return ::sqlite3_bind_int64(q,i,_i);
	}
	long long int _i;
};

struct String : public Value
{
	String(const std::string& s)
	:_s(s) {}
	virtual int bind(sqlite3_stmt* q, int i) const override
	{
		return ::sqlite3_bind_text(q,i,_s.c_str(),_s.size(),SQLITE_TRANSIENT);
	}
	std::string _s;
};

struct Variables
{
	static std::shared_ptr<Variables>& instance()
	{
		static auto v = std::make_shared<Variables>();
		return v;
	}
	mutable std::mutex _lock;
	std::map<std::string,Value::SP> _variables;
};
#line 168 "src/plugin.cpp"

extern "C"
{

/* INTERRUPTION sqlite.error */


/* INTERRUPTION sqlite.busy */


/* TYPE sqlite.database */

struct type_database
{
#line 252 "../plugins/sqlite/core/sqlite.svm_plugin"
	type_database(const std::string& file, const bool ro)
	:_file(file), _ro(ro),_db(nullptr) {}
	~type_database()
	{
		if(_db)
		{
			::sqlite3_close_v2(_db);
		}
	}
	operator std::string () const
	{
		return _file+" "+(_ro?"RO":"RW")+" ("+(_db?"open":"clos")+"ed)";
	}
	std::string _file;
	bool _ro;
	sqlite3* _db;
#line 200 "src/plugin.cpp"
};

void type_database_delete(const void *svm, void *handler)
{
	type_database * const object = reinterpret_cast<type_database*>(handler);
	{
#line 270 "../plugins/sqlite/core/sqlite.svm_plugin"

#line 209 "src/plugin.cpp"
	}
	delete object;
}

SVM_String type_database_print(const void *svm, const void *handler)
{
	const type_database *object = reinterpret_cast<const type_database*>(handler);
	std::string string = static_cast<std::string>(*object);
	{
#line 271 "../plugins/sqlite/core/sqlite.svm_plugin"

#line 221 "src/plugin.cpp"
	}
	return ::svm_string_new(svm,string.c_str(),string.size());
}


/* TYPE sqlite.query */

struct type_query
{
#line 278 "../plugins/sqlite/core/sqlite.svm_plugin"
	type_query()
	:_q(nullptr) {}
	~type_query()
	{
		if(_q)
		{
			::sqlite3_finalize(_q);
		}
	}
	operator std::string () const
	{
		if(_q)
		{
			return ::sqlite3_sql(_q);
		}
		return "<no query>";
	}
	sqlite3_stmt *_q;
#line 250 "src/plugin.cpp"
};

void type_query_delete(const void *svm, void *handler)
{
	type_query * const object = reinterpret_cast<type_query*>(handler);
	{
#line 298 "../plugins/sqlite/core/sqlite.svm_plugin"

#line 259 "src/plugin.cpp"
	}
	delete object;
}

SVM_String type_query_print(const void *svm, const void *handler)
{
	const type_query *object = reinterpret_cast<const type_query*>(handler);
	std::string string = static_cast<std::string>(*object);
	{
#line 299 "../plugins/sqlite/core/sqlite.svm_plugin"

#line 271 "src/plugin.cpp"
	}
	return ::svm_string_new(svm,string.c_str(),string.size());
}


/* SYSTEM INSTRUCTION sqlite.database STR:file [ 'RO' 'RW' ]:mode -> sqlite.database */

SVM_Value instruction_database(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 306 "../plugins/sqlite/core/sqlite.svm_plugin"
	SVM_String raw_file = ARGV_VALUE(0,string);
	auto mode = ARGV_KEYWORD(1);
	std::string file(raw_file.string,raw_file.size);
	bool ro = mode =="RO";
	auto db = new type_database(file,ro);
	int rc = ::sqlite3_open_v2(raw_file.string,&db->_db,ro?SQLITE_OPEN_READONLY:(SQLITE_OPEN_READWRITE|SQLITE_OPEN_CREATE),nullptr);
	if(rc == SQLITE_OK)
	{
		return NEW_PLUGIN(sqlite,database,db);
	}
	delete db;
	std::string err = ::sqlite3_errstr(rc);
	ERROR_EXTERNAL(sqlite,error,err.c_str());
	return nullptr;
#line 296 "src/plugin.cpp"
}


/* INSTRUCTION sqlite.query sqlite.database STR:query -> sqlite.query */

SVM_Value instruction_query(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 335 "../plugins/sqlite/core/sqlite.svm_plugin"
	auto db = ARGV_PLUGIN(0,sqlite,database);
	auto query = ARGV_VALUE(1,string);
	auto q = new type_query();
	int rc = ::sqlite3_prepare_v2(db->_db,query.string,query.size,&q->_q,nullptr);
	if(rc==SQLITE_OK)
	{
		return NEW_PLUGIN(sqlite,query,q);
	}
	delete q;
	std::string err = ::sqlite3_errstr(rc);
	ERROR_EXTERNAL(sqlite,error,err.c_str());
	return nullptr;
#line 317 "src/plugin.cpp"
}


/* INSTRUCTION sqlite.init MUTABLE sqlite.query ( VALUE | KEYWORD = VALUE ) *:parameters */

SVM_Value instruction_init(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 359 "../plugins/sqlite/core/sqlite.svm_plugin"
	auto q = ARGV_PLUGIN(0,sqlite,query);
	::sqlite3_reset(q->_q);
	::sqlite3_clear_bindings(q->_q);
	SVM_Index global_index = 1;
	for(SVM_Index i = 1 ; i<argc ; ++i)
	{
		SVM_Index index = global_index;
		if(::svm_parameter_type_is_keyword(svm,argv[i]))
		{
			auto n = ::svm_parameter_keyword_get(svm,argv[i]);
			std::string nn = ":";
			for(size_t ii=0 ; ii<n.size ; ++ii)
			{
				if((n.string[ii]>='A') and (n.string[ii]<='Z'))
				{
					nn += static_cast<char>(n.string[ii]-'A'+'a');
				}
				else
				{
					nn += n.string[ii];
				}
			}
			index = ::sqlite3_bind_parameter_index(q->_q,nn.c_str());
			if(index==0)
			{
				ERROR_EXTERNAL(sqlite,error,"Invalid binding name.");
			}
			i += 2;
		}
		else
		{
			++global_index;
		}
		SVM_Value v = ::svm_parameter_value_get(svm,argv[i]);
		int rc = 0;
		if(::svm_value_state_is_null(svm,v))
		{
			rc = ::sqlite3_bind_null(q->_q,index);
		}
		else if(::svm_value_type_is_integer(svm,v))
		{
			auto vv = ::svm_value_integer_get(svm,v);
			rc = ::sqlite3_bind_int64(q->_q,index,vv);
		}
		else if(::svm_value_type_is_string(svm,v))
		{
			auto vv = ::svm_value_string_get(svm,v);
			rc = ::sqlite3_bind_text(q->_q,index,vv.string,vv.size,SQLITE_TRANSIENT);
		}
		else
		{
			SVM_String vv = ::svm_value_print(svm,v);
			rc = ::sqlite3_bind_text(q->_q,index,vv.string,vv.size,SQLITE_TRANSIENT);
		}
		if(rc!=SQLITE_OK)
		{
			std::string err = ::sqlite3_errstr(rc);
			ERROR_EXTERNAL(sqlite,error,err.c_str());
		}
	}
	auto vars = Variables::instance();
	std::lock_guard<std::mutex> protect(vars->_lock);
	for(const auto& v: vars->_variables)
	{
		std::string name("$");
		name += v.first;
		int index = ::sqlite3_bind_parameter_index(q->_q,name.c_str());
		if(index==0) continue;
		v.second->bind(q->_q,index);
	}
#line 396 "src/plugin.cpp"
	return nullptr;
}


/* INSTRUCTION sqlite.exec MUTABLE sqlite.query PTR ? -> PTR ? */

SVM_Value instruction_exec(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 446 "../plugins/sqlite/core/sqlite.svm_plugin"
	auto q = ARGV_PLUGIN(0,sqlite,query);
	int rc = ::sqlite3_step(q->_q);
	switch(rc)
	{
		case SQLITE_ROW:
			{
				int count = ::sqlite3_column_count(q->_q);
				std::vector<SVM_Value> values;
				SVM_Memory_Zone zone = ::svm_memory_zone_new(svm);
				for(size_t i=0 ; i<count ; ++i)
				{
					int t = ::sqlite3_column_type(q->_q,i);
					if(t==SQLITE_INTEGER)
					{
						::svm_memory_zone_append_internal__raw(svm,zone,INTEGER,1);
						auto v = ::sqlite3_column_int64(q->_q,i);
						values.push_back(::svm_value_integer_new(svm,v));
					}
					else if(t==SQLITE_NULL)
					{
						::svm_memory_zone_append_internal__raw(svm,zone,AUTOMATIC,1);
						values.push_back(nullptr);
					}
					else
					{
						::svm_memory_zone_append_internal__raw(svm,zone,STRING,1);
						auto v = ::sqlite3_column_text(q->_q,i);
						values.push_back(::svm_value_string_new__raw(svm,reinterpret_cast<const char*>(v)));
					}
				}
				SVM_Value_Pointer pointer;
				if(argc==1)
				{
					pointer = ::svm_memory_allocate(svm,CURRENT(kernel),zone);
				}
				else
				{
					pointer = ::svm_parameter_value_get(svm,argv[1]);
					if(values.size()!=::svm_value_pointer_get_size(svm,pointer))
					{
						ERROR_INTERNAL(MEMORY,"Pointer size mismatch");
					}
				}
				SVM_Address a = ::svm_value_pointer_get_address(svm,pointer);
				for(const auto& v: values)
				{
					if(v)
					{
						::svm_memory_write_address(svm,CURRENT(kernel),a,v);
					}
					else
					{
						SVM_Type t = ::svm_memory_address_get_type(svm,CURRENT(kernel),a);
						if(::svm_type_is_external(svm,t))
						{
							ERROR_INTERNAL(MEMORY,"Invalid type for null value: Only INT, STR and AUTO allowed.");
						}
						switch(::svm_type_get_internal(svm,t))
						{
							case INTEGER:
								::svm_memory_write_address(svm,CURRENT(kernel),a,::svm_value_integer_new_null(svm));
								break;
							case STRING:
								::svm_memory_write_address(svm,CURRENT(kernel),a,::svm_value_string_new_null(svm));
								break;
							case AUTOMATIC:
								break;
							default:
								ERROR_INTERNAL(MEMORY,"Invalid type for null value: Only INT, STR and AUTO allowed.");
								break;
						}
					}
					++a;
				}
				return pointer;
			}
			break;
		case SQLITE_DONE:
			{
				return NEW_NULL_VALUE(pointer);
			}
			break;
		case SQLITE_BUSY:
		case SQLITE_LOCKED:
			{
				std::string err = ::sqlite3_errstr(rc);
				ERROR_EXTERNAL(sqlite,busy,err.c_str());
			}
			break;
		default:
			{
				std::string err = ::sqlite3_errstr(rc);
				ERROR_EXTERNAL(sqlite,error,err.c_str());
			}
			break;
	}
	return nullptr;
#line 503 "src/plugin.cpp"
}


/* INSTRUCTION sqlite.var STR:name VALUE ?:value */

SVM_Value instruction_var(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 566 "../plugins/sqlite/core/sqlite.svm_plugin"
	auto raw_name = ARGV_VALUE(0,string);
	std::string name(raw_name.string,raw_name.size);
	auto vars = Variables::instance();
	std::lock_guard<std::mutex> protect(vars->_lock);
	auto it = vars->_variables.find(name);
	if(argc==1)
	{
		if(it==vars->_variables.end()) return nullptr;
		vars->_variables.erase(it);
		return nullptr;
	}
	Value::SP value;
	SVM_Value v = ::svm_parameter_value_get(svm,argv[1]);
	if(::svm_value_type_is_integer(svm,v))
	{
		value = std::make_shared<Integer>(::svm_value_integer_get(svm,v));
	} 
	else if(::svm_value_type_is_string(svm,v))
	{
		SVM_String s = ::svm_value_string_get(svm,v);
		value = std::make_shared<String>(std::string(s.string,s.size));
	}
	else
	{
		SVM_String s = ::svm_value_print(svm,v);
		value = std::make_shared<String>(std::string(s.string,s.size));
	}
	if(it==vars->_variables.end())
	{
		vars->_variables.insert(std::make_pair(name,value));
	}
	else
	{
		it->second = value;
	}
#line 547 "src/plugin.cpp"
	return nullptr;
}


/* Generic handling functions */

void plugin_initialisation(const void *svm)
{
#line 77 "../plugins/sqlite/core/sqlite.svm_plugin"
	if(::sqlite3_config(SQLITE_CONFIG_SERIALIZED)==SQLITE_ERROR)
	{
		std::cerr << "SQLite library not compiled with multi-threading support." << std::endl
			<< "Your application may crash!" << std::endl;
	}
	::sqlite3_initialize();
#line 563 "src/plugin.cpp"
}

void plugin_finalisation(const void *svm)
{
#line 86 "../plugins/sqlite/core/sqlite.svm_plugin"
	::sqlite3_shutdown();
#line 570 "src/plugin.cpp"
}

}
