/*
 * Copyright (C) 2024-05-01  Julien BRUGUIER
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */


#include <string>
#include <sstream>

#line 90 "../plugins/tcp/core/tcp.svm_plugin"
#include <sstream>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/tcp.h>
#include <list>
#include <algorithm>
#include <atomic>
#line 34 "src/plugin.cpp"

#include <src/plugin.h>

extern "C"
{

void plugin_configure(void *plugin)
{
	::svm_plugin_configure(plugin,
		"PLUGIN tcp \n"
		"USE \n"
		"	TYPE com.device \n"
		"	FUNCTION com.device $tcp.client -> com.device ? \n"
		"	INTERRUPTION com.interrupted \n"
		"DEFINE \n"
		"	STRUCT tcp.client \n"
		"	FUNCTION tcp.device_client_open STR:host STR:service -> $tcp.client \n"
		"	FUNCTION tcp.device_client_print $tcp.client -> STR \n"
		"	FUNCTION tcp.device_client_read $tcp.client -> STR ? \n"
		"	FUNCTION tcp.device_client_write $tcp.client STR \n"
		"	FUNCTION tcp.device_client_idle $tcp.client MUTABLE INT 3 \n"
		"	FUNCTION tcp.device_client_command $tcp.client . * -> VALUE ? \n"
		"	FUNCTION tcp.device_client_close $tcp.client -> BLN \n"
		"	WAITING INSTRUCTION tcp.server STR:host STR:service INT:queue > STR:process_name SYM:code PTR:memory PEP:scheduler PEP:sequencer ? \n"
		"	OPTION tcp.scheduler_run_limit -l INT \n"
		"	OPTION tcp.scheduler_run_time -t INT \n"
		"	SYSTEM INSTRUCTION tcp.scheduler_run_limit INT \n"
		"	SCHEDULER tcp.scheduler \n"
		,SVM_API_SIGNATURE,SVM_VERSION);
}

#ifndef ARGV_VALUE
#define ARGV_VALUE(index,type) ::svm_value_##type##_get(svm,::svm_parameter_value_get(svm,argv[(index)]))
#endif
#ifndef ARGV_PLUGIN
#define ARGV_PLUGIN(index,plugin,name) reinterpret_cast<type_##name*>(::svm_value_plugin_get_internal(svm,::svm_parameter_value_get(svm,argv[(index)])))
#endif
#ifndef ARGV_MARKER
#define ARGV_MARKER(index) std::string(::svm_parameter_marker_get(svm,argv[(index)]).string)
#endif
#ifndef ARGV_KEYWORD
#define ARGV_KEYWORD(index) std::string(::svm_parameter_keyword_get(svm,argv[(index)]).string)
#endif
#ifndef ARGV_STRUCT
#define ARGV_STRUCT(index,plugin,name) reinterpret_cast<struct_##name*>(::svm_structure_get_internal(svm,::svm_value_pluginentrypoint_new__raw(svm,#plugin,#name),::svm_parameter_structure_get(svm,argv[(index)])))
#endif
#ifndef ARGV_VARIABLE
#define ARGV_VARIABLE(index) ::svm_parameter_variable_get(svm,argv[(index)]);
#endif
#ifndef NEW_VALUE
#define NEW_VALUE(type,value) ::svm_value_##type##_new(svm,(value))
#endif
#ifndef NEW_PLUGIN
#define NEW_PLUGIN(plugin,name,value) ::svm_value_plugin_new(svm,::svm_value_pluginentrypoint_new__raw(svm,#plugin,#name),(value))
#endif
#ifndef NEW_STRUCT
#define NEW_STRUCT(plugin,name,value) ::svm_structure_new(svm,::svm_value_pluginentrypoint_new__raw(svm,#plugin,#name),(value))
#endif
#ifndef NEW_STRING
#define NEW_STRING(raw_string) ::svm_string_new(svm,raw_string.c_str(),raw_string.size())
#endif
#ifndef RAW_STRING
#define RAW_STRING(svm_string) std::string(svm_string.string,svm_string.size)
#endif
#ifndef NEW_BOOLEAN
#define NEW_BOOLEAN(boolean) ((boolean)?TRUE:FALSE)
#endif
#ifndef RAW_BOOLEAN
#define RAW_BOOLEAN(boolean) ((boolean)==TRUE)
#endif
#ifndef NEW_NULL_VALUE
#define NEW_NULL_VALUE(type) ::svm_value_##type##_new_null(svm)
#endif
#ifndef NEW_NULL_PLUGIN
#define NEW_NULL_PLUGIN(plugin,name) ::svm_value_plugin_new_null(svm,::svm_value_pluginentrypoint_new__raw(svm,#plugin,#name))
#endif
#ifndef NEW_NULL_STRUCT
#define NEW_NULL_STRUCT(plugin,name) ::svm_structure_new_null(svm,::svm_value_pluginentrypoint_new__raw(svm,#plugin,#name))
#endif
#ifndef ERROR_INTERNAL
#define ERROR_INTERNAL(irq,message) ::svm_processor_current_raise_error_internal__raw(svm,irq,message)
#endif
#ifndef ERROR_EXTERNAL
#define ERROR_EXTERNAL(plugin,name,message) ::svm_processor_current_raise_error_external__raw(svm,::svm_value_pluginentrypoint_new__raw(svm,#plugin,#name),message)
#endif
#ifndef CONST_PEP
#define CONST_PEP(plugin,name) ::svm_value_pluginentrypoint_new__raw(svm,#plugin,#name)
#endif
#ifndef CURRENT
#define CURRENT(object) ::svm_##object##_get_current(svm)
#endif
#ifndef RETURN
#define RETURN return nullptr
#endif
#ifndef VARIABLE_GLOBAL
#define VARIABLE_GLOBAL(variable) ::svm_variable_scope_set_global(svm,(variable))
#endif
#ifndef VARIABLE_LOCAL
#define VARIABLE_LOCAL(variable) ::svm_variable_scope_set_local(svm,(variable))
#endif
#ifndef VARIABLE_DELETE
#define VARIABLE_DELETE(variable) ::svm_variable_delete(svm,(variable))
#endif

}

#line 104 "../plugins/tcp/core/tcp.svm_plugin"
	#define BUFFER_SIZE 4096

	std::string strerror_local(int e)
	{
		return ::strerror(e);
	}

	std::string gai_strerror_local(int e)
	{
		return ::gai_strerror(e);
	}

	struct Memoire
	{
		Memoire()
		{
			res0 = nullptr;
		}

		~Memoire()
		{
			if(res0)
				::freeaddrinfo(res0);
		}
		struct addrinfo *res0;
	};

	bool resolution_ip_port(const struct sockaddr* sock, const socklen_t longueur, std::string& ip, std::string& port)
	{
		char host[BUFFER_SIZE+1];
		char serv[BUFFER_SIZE+1];
		bool resultat=::getnameinfo(sock,longueur,host,BUFFER_SIZE,serv,BUFFER_SIZE,NI_NUMERICHOST)==0;
		if(resultat)
		{
			ip = host;
			port = serv;
		}
		return resultat;
	}

	void log_erreur(struct addrinfo *res, std::ostringstream& os, const std::string& fonction, int erreur)
	{
		std::string ip;
		std::string port;
		os << std::endl << fonction << " ";
		if(resolution_ip_port(res->ai_addr,res->ai_addrlen,ip,port))
		{
			os << "(" << ip << " : " << port << ")";
		}
		os << ": " << ::strerror_local(erreur);
	}

	template<bool appelle_bind, bool appelle_listen, bool appelle_connect>
		static int ouvrir(const void *svm, const int type, const std::string& ip, const std::string& port, const int tcp_file)
		{
			int fd;
			struct addrinfo hints, *res;
			int error;
			Memoire memoire;
			std::ostringstream details;
			details << "Unable to open TCP socket.";

			::memset(&hints, 0, sizeof(hints));
			hints.ai_family = PF_UNSPEC;
			hints.ai_socktype = type;
			error = ::getaddrinfo(ip.c_str(), port.c_str(), &hints, &memoire.res0);
			if (error)
			{
				details << std::endl << "getaddrinfo: " << ::gai_strerror_local(error);
				::svm_processor_current_raise_error_internal__raw(svm,DEVICE,details.str().c_str());
			}
			fd = -1;
			for (res = memoire.res0 ; res ; res = res->ai_next)
			{
				fd = ::socket(res->ai_family, res->ai_socktype, res->ai_protocol);
				if (fd < 0)
				{
					log_erreur(res,details,"socket",errno);
					continue;
				}

				if(appelle_bind)
				{
					if (::bind(fd, res->ai_addr, res->ai_addrlen) < 0)
					{
						log_erreur(res,details,"bind",errno);
						::close(fd);
						fd = -1;
						continue;
					}
				}

				if(appelle_listen)
				{
					if(::listen(fd,tcp_file)<0)
					{
						log_erreur(res,details,"listen",errno);
						::close(fd);
						fd = -1;
						continue;
					}
				}

				if(appelle_connect)
				{
					if (::connect(fd, res->ai_addr, res->ai_addrlen) < 0)
					{
						log_erreur(res,details,"connect",errno);
						::close(fd);
						fd = -1;
						continue;
					}
				}
				break;
			}
			if (fd < 0)
			{
				details << std::endl;
				::svm_processor_current_raise_error_internal__raw(svm,DEVICE,details.str().c_str());
			}
			return fd;
		}
#line 264 "src/plugin.cpp"

extern "C"
{

/* STRUCT tcp.client */

struct struct_client
{
#line 314 "../plugins/tcp/core/tcp.svm_plugin"
	struct_client(const std::string& host, const std::string& port)
	:_host(host),_port(port) {}
	std::string _host;
	std::string _port;
	int _socket {0};
	bool _blocking{true};
	bool _read_half_closed{false};
	bool _write_half_closed{false};
#line 282 "src/plugin.cpp"
};

void struct_client_delete(const void *svm, void *handler)
{
	struct_client * const object = reinterpret_cast<struct_client*>(handler);
	{
#line 324 "../plugins/tcp/core/tcp.svm_plugin"

#line 291 "src/plugin.cpp"
	}
	delete object;
}


/* FUNCTION tcp.device_client_open STR:host STR:service -> $tcp.client */

SVM_Variable function_device_client_open(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 331 "../plugins/tcp/core/tcp.svm_plugin"
	SVM_String shost = ARGV_VALUE(0,string);
	SVM_String sservice = ARGV_VALUE(1,string);
	int fd_server = ouvrir<false,false,true>(svm,SOCK_STREAM,RAW_STRING(shost),RAW_STRING(sservice),0);
	struct_client *client = new struct_client(RAW_STRING(shost),RAW_STRING(sservice));
	client->_socket = fd_server;
	return NEW_STRUCT(tcp,client,client);
#line 308 "src/plugin.cpp"
}


/* FUNCTION tcp.device_client_print $tcp.client -> STR */

SVM_Variable function_device_client_print(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 347 "../plugins/tcp/core/tcp.svm_plugin"
	struct_client *client = ARGV_STRUCT(0,tcp,client);
	std::ostringstream oss;
	oss << "TCP client " << client->_host << ":" << client->_port << " (" << client->_socket << ")" << (client->_blocking?"":"non blocking");
	return NEW_VALUE(string,NEW_STRING(oss.str()));
#line 321 "src/plugin.cpp"
}


/* FUNCTION tcp.device_client_read $tcp.client -> STR ? */

SVM_Variable function_device_client_read(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 359 "../plugins/tcp/core/tcp.svm_plugin"
	struct_client *client = ARGV_STRUCT(0,tcp,client);
	if(client->_read_half_closed)
	{
		ERROR_INTERNAL(DEVICE,"Can not read from closed device");
	}
	char buffer[BUFFER_SIZE+1];
	::svm_process_pause(svm);
	::svm_process_interruptionnotification_enable(svm,CURRENT(process));
	int read = ::recv(client->_socket,buffer,BUFFER_SIZE,0);
	::svm_process_interruptionnotification_disable(svm,CURRENT(process));
	::svm_process_resume(svm);
	if(read<0)
	{
		if(errno==EINTR)
		{
			ERROR_EXTERNAL(com,interrupted,"Read interrupted.");
		}
		if(((errno==EAGAIN) or (errno==EWOULDBLOCK)) and not client->_blocking)
		{
			return ::svm_value_string_new__raw(svm,"");
		}
		std::ostringstream oss;
		oss << "Read error on TCP socket: " << ::strerror_local(errno);
		ERROR_INTERNAL(DEVICE,oss.str().c_str());
	}
	if(read==0)
	{
		return NEW_NULL_VALUE(string);
	}
	return ::svm_value_string_new__buffer(svm,buffer,read);
#line 360 "src/plugin.cpp"
}


/* FUNCTION tcp.device_client_write $tcp.client STR */

SVM_Variable function_device_client_write(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 399 "../plugins/tcp/core/tcp.svm_plugin"
	struct_client *client = ARGV_STRUCT(0,tcp,client);
	if(client->_write_half_closed)
	{
		ERROR_INTERNAL(DEVICE,"Can not write to closed device");
	}
	SVM_String text = ARGV_VALUE(1,string);
	ssize_t written = ::send(client->_socket,text.string,text.size,MSG_NOSIGNAL);
	if(written<0)
	{
		std::ostringstream oss;
		oss << "Write error on TCP socket: " << ::strerror_local(errno);
		ERROR_INTERNAL(DEVICE,oss.str().c_str());
	}
	if(static_cast<size_t>(written)!=text.size)
	{
		std::ostringstream oss;
		oss << "Write error on TCP socket: Incomplete write to device.";
		ERROR_INTERNAL(DEVICE,oss.str().c_str());
	}
#line 388 "src/plugin.cpp"
	return nullptr;
}


/* FUNCTION tcp.device_client_idle $tcp.client MUTABLE INT 3 */

SVM_Variable function_device_client_idle(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 426 "../plugins/tcp/core/tcp.svm_plugin"
	struct_client *client = ARGV_STRUCT(0,tcp,client);
	for(size_t i=1 ; i<4 ; ++i)
	{
		SVM_Value_Integer c = ::svm_parameter_value_get(svm,argv[i]);
		::svm_value_integer_set(svm,c,client->_socket);
	}
#line 404 "src/plugin.cpp"
	return nullptr;
}


/* FUNCTION tcp.device_client_command $tcp.client . * -> VALUE ? */

SVM_Variable function_device_client_command(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 440 "../plugins/tcp/core/tcp.svm_plugin"
	struct_client *client = ARGV_STRUCT(0,tcp,client);
	for(SVM_Index index=1 ; index<argc ; ++index)
	{
		if(::svm_parameter_type_is_keyword(svm,argv[index])==FALSE)
		{
			ERROR_INTERNAL(FAILURE,"Invalid command name");
		}
		SVM_String rcommand = ::svm_parameter_keyword_get(svm,argv[index]);
		auto command = RAW_STRING(rcommand);
		if(command=="CLOSE")
		{
			++index;
			if((index>=argc) or (::svm_parameter_type_is_marker(svm,argv[index])==FALSE))
			{
				ERROR_INTERNAL(FAILURE,"Invalid close command");
			}
			SVM_String rdirection = ::svm_parameter_marker_get(svm,argv[index]);
			auto direction = RAW_STRING(rdirection);
			if(direction==">")
			{
				::shutdown(client->_socket,SHUT_WR);
				client->_write_half_closed = true;
			}
			else if(direction=="<")
			{
				::shutdown(client->_socket,SHUT_RD);
				client->_read_half_closed = true;
			}
			else
			{
				ERROR_INTERNAL(FAILURE,"Invalid close command");
			}
		}
		else if(command=="LINGER")
		{
			++index;
			if(index>=argc)
			{
				ERROR_INTERNAL(FAILURE,"Invalid linger command");
			}
			linger linger;
			linger.l_onoff = 0;
			linger.l_linger = 0;
			if(::svm_parameter_type_is_value(svm,argv[index])==TRUE)
			{
				SVM_Value rvalue = ::svm_parameter_value_get(svm,argv[index]);
				if(::svm_value_type_is_integer(svm,rvalue)==FALSE)
				{
					ERROR_INTERNAL(FAILURE,"Invalid linger command");
				}
				auto l = ::svm_value_integer_get(svm,rvalue);
				if(l<0)
				{
					ERROR_INTERNAL(FAILURE,"Invalid linger command");
				}
				linger.l_onoff = 1;
				linger.l_linger = l;
			}
			else if(::svm_parameter_type_is_keyword(svm,argv[index])==TRUE)
			{
				SVM_String roff = ::svm_parameter_keyword_get(svm,argv[index]);
				if(RAW_STRING(roff)!="OFF")
				{
					ERROR_INTERNAL(FAILURE,"Invalid linger command");
				}
			}
			else
			{
				ERROR_INTERNAL(FAILURE,"Invalid linger command");
			}
			if(::setsockopt(client->_socket,SOL_SOCKET,SO_LINGER,&linger,sizeof(linger)))
			{
				std::ostringstream oss;
				oss	<< "Unable to set linger mode on TCP socket." << std::endl
					<< ::strerror_local(errno) << std::endl;
				ERROR_INTERNAL(DEVICE,oss.str().c_str());
			}
				
		}
		else if(command=="BLOCKING")
		{
			++index;
			if(index>=argc)
			{
				ERROR_INTERNAL(FAILURE,"Invalid blocking command");
			}
			if(::svm_parameter_type_is_value(svm,argv[index])==FALSE)
			{
				ERROR_INTERNAL(FAILURE,"Invalid blocking command");
			}
			SVM_Value rtrigger = ::svm_parameter_value_get(svm,argv[index]);
			if(::svm_value_type_is_boolean(svm,rtrigger)==FALSE)
			{
				ERROR_INTERNAL(FAILURE,"Invalid blocking command");
			}
			SVM_Boolean trigger = ::svm_value_boolean_get(svm,rtrigger);
			client->_blocking=trigger==TRUE;
			if(client->_blocking)
			{
				int flags = ::fcntl(client->_socket, F_GETFL, 0);
				::fcntl(client->_socket, F_SETFL, flags bitand (compl O_NONBLOCK));
			}
			else
			{
				int flags = ::fcntl(client->_socket, F_GETFL, 0);
				::fcntl(client->_socket, F_SETFL, flags bitor O_NONBLOCK);
			}
		}
		else if(command=="NODELAY")
		{
			++index;
			if(index>=argc)
			{
				ERROR_INTERNAL(FAILURE,"Invalid nodelay command");
			}
			if(::svm_parameter_type_is_value(svm,argv[index])==FALSE)
			{
				ERROR_INTERNAL(FAILURE,"Invalid nodelay command");
			}
			SVM_Value rtrigger = ::svm_parameter_value_get(svm,argv[index]);
			if(::svm_value_type_is_boolean(svm,rtrigger)==FALSE)
			{
				ERROR_INTERNAL(FAILURE,"Invalid nodelay command");
			}
			SVM_Boolean trigger = ::svm_value_boolean_get(svm,rtrigger);
			int itrigger = trigger?1:0;
			::setsockopt( client->_socket, IPPROTO_TCP, TCP_NODELAY, (void *)&itrigger, sizeof(itrigger));
		}
		else if(command=="REMOTE")
		{
			++index;
			if(index>=argc)
			{
				ERROR_INTERNAL(FAILURE,"Invalid remote command");
			}
			if(::svm_parameter_type_is_keyword(svm,argv[index])==FALSE)
			{
				ERROR_INTERNAL(FAILURE,"Invalid remote command");
			}
			SVM_String rattr = ::svm_parameter_keyword_get(svm,argv[index]);
			auto attr = RAW_STRING(rattr);
			if(attr=="IP")
			{
				return NEW_VALUE(string,NEW_STRING(client->_host));
			}
			else if(attr=="PORT")
			{
				return NEW_VALUE(string,NEW_STRING(client->_port));
			}
			else
			{
				ERROR_INTERNAL(FAILURE,"Invalid remote command");
			}
		}
		else
		{
			ERROR_INTERNAL(FAILURE,"Invalid command name");
		}
	}
#line 573 "src/plugin.cpp"
}


/* FUNCTION tcp.device_client_close $tcp.client -> BLN */

SVM_Variable function_device_client_close(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 614 "../plugins/tcp/core/tcp.svm_plugin"
	struct_client *client = ARGV_STRUCT(0,tcp,client);
	::shutdown(client->_socket,SHUT_RDWR);
	::close(client->_socket);
	return NEW_VALUE(boolean,TRUE);
#line 586 "src/plugin.cpp"
}


/* WAITING INSTRUCTION tcp.server STR:host STR:service INT:queue > STR:process_name SYM:code PTR:memory PEP:scheduler PEP:sequencer ? */

SVM_Value instruction_server(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 626 "../plugins/tcp/core/tcp.svm_plugin"
	SVM_String shost = ARGV_VALUE(0,string);
	SVM_String sservice = ARGV_VALUE(1,string);
	long long int client_queue_size = ARGV_VALUE(2,integer);
	if(client_queue_size<1)
	{
		ERROR_INTERNAL(FAILURE,"Invalid client queue size");
	}
	SVM_String sclient = ARGV_VALUE(4,string);
	SVM_Value_Symbol code = ::svm_parameter_value_get(svm,argv[5]);
	SVM_Value_Pointer memory = ::svm_parameter_value_get(svm,argv[6]);
	SVM_Value_PluginEntryPoint scheduler = ::svm_parameter_value_get(svm,argv[7]);
	SVM_Value_PluginEntryPoint sequencer = nullptr;
	if(argc>8)
	{
		sequencer = ::svm_parameter_value_get(svm,argv[8]);
	}
	int fd_server = ouvrir<true,true,false>(svm,SOCK_STREAM,RAW_STRING(shost),RAW_STRING(sservice),client_queue_size);
	for(;;)
	{
		errno=0;
		::svm_process_interruptionnotification_enable(svm,CURRENT(process));
		struct sockaddr sockclient;
		socklen_t l = sizeof(sockclient);
		int fd_client = ::accept(fd_server,&sockclient,&l);
		::svm_process_interruptionnotification_disable(svm,CURRENT(process));
		if(fd_client<0)
		{
			if(errno==EINTR)
			{
				::shutdown(fd_server,SHUT_RDWR);
				::close(fd_server);
				RETURN;
			}
			else
			{
				std::ostringstream oss;
				oss << "Error raised while waiting for client:" << ::strerror_local(errno);
				ERROR_INTERNAL(DEVICE,oss.str().c_str());
			}
		}
		std::string remote_host;
		std::string remote_port;
		resolution_ip_port(&sockclient,l,remote_host,remote_port);
		struct_client *rclient = new struct_client(remote_host,remote_port);
		rclient->_socket = fd_client;
		SVM_Structure client = NEW_STRUCT(tcp,client,rclient);
		SVM_Parameter *create_params = ::svm_parameter_array_new(svm,1);
		create_params[0] = ::svm_parameter_structure_new(svm,client);
		SVM_Value_Plugin device = ::svm_function_call(svm,CONST_PEP(com,device),1,create_params);
		VARIABLE_DELETE(client);
		::svm_value_state_set_movable(svm,device);
		VARIABLE_DELETE(create_params);
		SVM_Kernel kernel = ::svm_kernel_new_symbol(svm,TRUE,FALSE,FALSE,nullptr,code);
		SVM_Value_Pointer *all_memory = ::svm_memory_pointer_list_accessible(svm,CURRENT(kernel),memory);
		for(SVM_Value_Pointer *it_memory=all_memory ; *it_memory ; ++it_memory)
		{
			SVM_Address address = ::svm_value_pointer_get_address(svm,*it_memory);
			SVM_Size size = ::svm_value_pointer_get_size(svm,*it_memory);
			SVM_Memory_Zone zone = ::svm_memory_zone_new(svm);
			::svm_memory_zone_append_internal__raw(svm,zone,AUTOMATIC,size);
			::svm_memory_allocate_address(svm,kernel,zone,address);
			VARIABLE_DELETE(zone);
			::svm_memory_share(svm,CURRENT(kernel),*it_memory,kernel,*it_memory);
		}
		SVM_Memory_Zone zone = ::svm_memory_zone_new(svm);
		::svm_memory_zone_append_external__raw(svm,zone,CONST_PEP(com,device),1);
		::svm_memory_zone_append_internal__raw(svm,zone,POINTER,1);
		SVM_Value_Pointer pointer = ::svm_memory_allocate(svm,kernel,zone);
		::svm_value_state_set_movable(svm,pointer);
		SVM_Address address = ::svm_value_pointer_get_address(svm,pointer);
		::svm_memory_write_address(svm,kernel,address,device);
		::svm_memory_write_address(svm,kernel,address+1,all_memory[0]);
		::svm_processor_set_currentpointer(svm,kernel,pointer);
		VARIABLE_DELETE(zone);
		VARIABLE_DELETE(pointer);
		VARIABLE_DELETE(device);
		VARIABLE_DELETE(all_memory);
		SVM_Process process = ::svm_process_new__string(svm,sclient,sequencer,FALSE,kernel);
		VARIABLE_DELETE(kernel);
		SVM_Process_Lock lock = ::svm_process_ownership_lock(svm,process);
		SVM_Scheduler rscheduler = ::svm_scheduler_get(svm,scheduler);
		::svm_scheduler_process_attach__raw(svm,rscheduler,process,0);
		VARIABLE_DELETE(lock);
		VARIABLE_DELETE(process);
	}

#line 681 "src/plugin.cpp"
	return nullptr;
}


/* OPTION tcp.scheduler_run_limit -l INT */


/* OPTION tcp.scheduler_run_time -t INT */


/* SYSTEM INSTRUCTION tcp.scheduler_run_limit INT */

SVM_Value instruction_scheduler_run_limit(const void *svm, SVM_Size argc, SVM_Parameter argv[])
{
#line 741 "../plugins/tcp/core/tcp.svm_plugin"
	auto l = ARGV_VALUE(0,integer);
	if(l<1)
	{
		ERROR_INTERNAL(FAILURE,"Invalid tcp.scheduler run limit");
	}
	::svm_scheduler_notify__raw(svm,::svm_scheduler_get(svm,CONST_PEP(tcp,scheduler)),l);
#line 703 "src/plugin.cpp"
	return nullptr;
}


/* SCHEDULER tcp.scheduler */

struct scheduler_scheduler
{
#line 757 "../plugins/tcp/core/tcp.svm_plugin"
	std::list<SVM_Process> _run;
	std::list<SVM_Process> _ready;
	std::list<SVM_Process> _waiting;
	std::list<SVM_Process> _other;
	std::atomic_size_t _scheduler_run_limit{1};
	std::atomic_size_t _scheduler_run_time{1000};

	static void list_clean(const void *svm, std::list<SVM_Process>& liste)
	{
		for(auto& p:liste)
		{
			VARIABLE_DELETE(p);
		}
	}

	static bool process_removal(const void *svm, std::list<SVM_Process>& liste, const SVM_Process processus)
	{
		auto it = std::find(liste.begin(),liste.end(),processus);
		if(it==liste.end()) return false;
		liste.erase(it);
		return true;
	}

	static void list_display(const void *svm,std::ostringstream& oss, const std::list<SVM_Process>& liste)
	{
		for(auto& p: liste)
		{
			SVM_String s = ::svm_process_print(svm,p);
			oss << "  " << RAW_STRING(s) << std::endl;
		}
	}
#line 744 "src/plugin.cpp"
};

void* scheduler_scheduler_create(const void *svm)
{
	scheduler_scheduler *object = new scheduler_scheduler();
	{
#line 790 "../plugins/tcp/core/tcp.svm_plugin"

#line 753 "src/plugin.cpp"
	}
	return object;
}

void scheduler_scheduler_delete(const void *svm, void *handler)
{
	scheduler_scheduler * const object = reinterpret_cast<scheduler_scheduler*>(handler);
	{
#line 792 "../plugins/tcp/core/tcp.svm_plugin"
	scheduler_scheduler::list_clean(svm,object->_run);
	scheduler_scheduler::list_clean(svm,object->_ready);
	scheduler_scheduler::list_clean(svm,object->_waiting);
	scheduler_scheduler::list_clean(svm,object->_other);
#line 767 "src/plugin.cpp"
	}
	delete object;
}

unsigned long int scheduler_scheduler_schedule(const void *svm, void *handler, const SVM_Process process, const SVM_Process_State state)
{
	scheduler_scheduler *object = reinterpret_cast<scheduler_scheduler*>(handler);
	{
#line 799 "../plugins/tcp/core/tcp.svm_plugin"
	bool deleted = false;
	deleted = scheduler_scheduler::process_removal(svm,object->_run,process);
	if(not deleted) deleted = scheduler_scheduler::process_removal(svm,object->_ready,process);
	if(not deleted) deleted = scheduler_scheduler::process_removal(svm,object->_waiting,process);
	if(not deleted) deleted = scheduler_scheduler::process_removal(svm,object->_other,process);
	switch(state)
	{
		case RUNNING:
			object->_run.push_back(process);	
			break;
		case SUSPENDED:
		case CONTINUE:
			object->_ready.push_back(process);	
			break;
		case WAITING:
			object->_waiting.push_back(process);
			break;
		default:
			object->_other.push_back(process);
			break;
	}
	while(object->_run.size()<object->_scheduler_run_limit)
	{
		if(object->_ready.empty()) break;
		SVM_Process p = object->_ready.front();
		object->_run.push_back(p);
		object->_ready.pop_front();
		::svm_process_run__raw(svm,p,0);
	}
	return object->_scheduler_run_time.load();
#line 807 "src/plugin.cpp"
	}
}

unsigned long int scheduler_scheduler_notification(const void *svm, void *handler, const SVM_Notification_Type type, unsigned long int parameter)
{
	scheduler_scheduler *object = reinterpret_cast<scheduler_scheduler*>(handler);
	{
#line 832 "../plugins/tcp/core/tcp.svm_plugin"
	if(type == NOTIFICATION)
	{
		if(parameter>0)
		{
			if(parameter<object->_scheduler_run_limit)
			{
				size_t suspend = object->_scheduler_run_limit-parameter;
				for(auto& p:object->_run)
				{
					::svm_process_suspend(svm,p);
					if(--suspend==0) break;
				}
			}
			if(parameter>object->_scheduler_run_limit)
			{
				size_t run = parameter-object->_scheduler_run_limit;
				for(auto& p:object->_ready)
				{
					::svm_process_run__raw(svm,p,0);
					if(--run==0) break;
				}
			}
			object->_scheduler_run_limit = parameter;
		}
	}
	if(type == TIMER and not object->_run.empty() and not object->_ready.empty())
	{
		::svm_process_suspend(svm,object->_run.front());
	}
	return object->_scheduler_run_time.load();
#line 846 "src/plugin.cpp"
	}
}

SVM_Boolean scheduler_scheduler_attach(const void *svm, void *handler, const SVM_Process process, unsigned long int parameter)
{
	scheduler_scheduler *object = reinterpret_cast<scheduler_scheduler*>(handler);
	{
#line 865 "../plugins/tcp/core/tcp.svm_plugin"
	object->_other.push_back(process);
	VARIABLE_GLOBAL(process);
	return TRUE;
#line 858 "src/plugin.cpp"
	}
}

SVM_Boolean scheduler_scheduler_detach(const void *svm, void *handler, const SVM_Process process, unsigned long int parameter)
{
	scheduler_scheduler *object = reinterpret_cast<scheduler_scheduler*>(handler);
	{
#line 871 "../plugins/tcp/core/tcp.svm_plugin"
	bool deleted = false;
	deleted = scheduler_scheduler::process_removal(svm,object->_run,process);
	if(not deleted) deleted = scheduler_scheduler::process_removal(svm,object->_ready,process);
	if(not deleted) deleted = scheduler_scheduler::process_removal(svm,object->_waiting,process);
	if(not deleted) deleted = scheduler_scheduler::process_removal(svm,object->_other,process);
	if(not deleted) return FALSE;
	VARIABLE_LOCAL(process);
	return TRUE;
#line 875 "src/plugin.cpp"
	}
}

SVM_String scheduler_scheduler_print(const void *svm, const void *handler)
{
	const scheduler_scheduler *object = reinterpret_cast<const scheduler_scheduler*>(handler);
	{
#line 882 "../plugins/tcp/core/tcp.svm_plugin"
	std::ostringstream oss;
	oss << "Executed processes:" << std::endl;
	scheduler_scheduler::list_display(svm,oss,object->_run);
	oss << "Ready processes:" << std::endl;
	scheduler_scheduler::list_display(svm,oss,object->_ready);
	oss << "Waiting processes:" << std::endl;
	scheduler_scheduler::list_display(svm,oss,object->_waiting);
	oss << "Other processes:" << std::endl;
	scheduler_scheduler::list_display(svm,oss,object->_other);
	return NEW_STRING(oss.str());
#line 894 "src/plugin.cpp"
	}
}


/* Generic handling functions */

}
