/*----------------------------------------------------------------------------/
  Lovyan GFX - Graphics library for embedded devices.

Original Source:
 https://github.com/lovyan03/LovyanGFX/

Licence:
 [FreeBSD](https://github.com/lovyan03/LovyanGFX/blob/master/license.txt)

Author:
 [lovyan03](https://twitter.com/lovyan03)

Contributors:
 [ciniml](https://github.com/ciniml)
 [mongonta0716](https://github.com/mongonta0716)
 [tobozo](https://github.com/tobozo)
/----------------------------------------------------------------------------*/
#pragma once

#include "../../misc/DataWrapper.hpp"
#include "../../misc/enum.hpp"
#include "../../../utility/result.hpp"

#include <malloc.h>

#if defined ( ARDUINO )
 #include <Arduino.h>
#endif

namespace lgfx
{
 inline namespace v1
 {
//----------------------------------------------------------------------------

#if defined ( ARDUINO )
  __attribute__ ((unused))
  static inline unsigned long millis(void)
  {
    return ::millis();
  }
  __attribute__ ((unused))
  static inline unsigned long micros(void)
  {
    return ::micros();
  }
  __attribute__ ((unused))
  static inline void delay(unsigned long milliseconds)
  {
    ::delay(milliseconds);
  }

#else

  static inline unsigned long micros(void) { return system_get_time(); }
  unsigned long millis(void);
  void delay(unsigned long ms);

#endif

  __attribute__ ((unused))
  static void delayMicroseconds(unsigned int us)
  {
    ets_delay_us(us);
  }

  static inline void* heap_alloc(      size_t length) { return malloc(length); }
  static inline void* heap_alloc_psram(size_t length) { return malloc(length); }
  static inline void* heap_alloc_dma(  size_t length) { return malloc(length); } // aligned_alloc(16, length);
  static inline void heap_free(void* buf) { free(buf); }
  static inline bool heap_capable_dma(const void* ptr) { return false; }

  static inline void gpio_hi(int_fast8_t pin) { if (pin & 16) { if (pin == 16) *(volatile uint32_t*)(0x60000768) |=  1; } else { *(volatile uint32_t*)(0x60000304) = 1 << (pin & 15); } }
  static inline void gpio_lo(int_fast8_t pin) { if (pin & 16) { if (pin == 16) *(volatile uint32_t*)(0x60000768) &= ~1; } else { *(volatile uint32_t*)(0x60000308) = 1 << (pin & 15); } }
  static inline bool gpio_in(int_fast8_t pin)
  {
    return *(volatile uint32_t*)((pin & 16)
         ? 0x6000078C // GP16I
         : 0x60000318 // GPI
         ) & (1 << (pin & 15));
  }

  enum pin_mode_t
  { output
  , input
  , input_pullup
  , input_pulldown
  };

  void pinMode(int_fast16_t pin, pin_mode_t mode);
  inline void lgfxPinMode(int_fast16_t pin, pin_mode_t mode)
  {
    pinMode(pin, mode);
  }

//----------------------------------------------------------------------------

#if defined (ARDUINO)
 #if defined (SDFS_H)
 sdfs::
 #endif
 #if defined (_SD_H_)
   #define LGFX_FILESYSTEM_SD_INSTANCE SD
   #define LGFX_FILESYSTEM_SD_TYPENAME fs::SDFS
 #elif defined (SDFS_H) \
    || defined (__SD_H__) // Seeed_SD.h
   #define LGFX_FILESYSTEM_SD_INSTANCE SDFS
   #define LGFX_FILESYSTEM_SD_TYPENAME fs::SDFS
 #endif
 #if defined (_LITTLEFS_H_) || defined (__LITTLEFS_H) || defined (_LiffleFS_H_)
   #define LGFX_FILESYSTEM_LITTLEFS_INSTANCE LittleFS
   #define LGFX_FILESYSTEM_LITTLEFS_TYPENAME fs::LittleFSFS
 #endif
 #if defined (_SPIFFS_H_)
   #define LGFX_FILESYSTEM_SPIFFS_INSTANCE SPIFFS
   #define LGFX_FILESYSTEM_SPIFFS_TYPENAME fs::SPIFFSFS
 #endif
 #if defined (LGFX_FILESYSTEM_SD_INSTANCE)
   #define LGFX_DEFAULT_FILESYSTEM LGFX_FILESYSTEM_SD_INSTANCE
 #elif defined (LGFX_FILESYSTEM_LITTLEFS_INSTANCE)
   #define LGFX_DEFAULT_FILESYSTEM LGFX_FILESYSTEM_LITTLEFS_INSTANCE
 #elif defined (LGFX_FILESYSTEM_SPIFFS_INSTANCE)
   #define LGFX_DEFAULT_FILESYSTEM LGFX_FILESYSTEM_SPIFFS_INSTANCE
 #endif

 #if defined (FS_H) || defined (__SD_H__) \
  || defined (LGFX_FILESYSTEM_SD_INSTANCE) \
  || defined (LGFX_FILESYSTEM_LITTLEFS_INSTANCE) \
  || defined (LGFX_FILESYSTEM_SPIFFS_INSTANCE)

  template <>
  struct DataWrapperT<fs::File> : public DataWrapper {
    DataWrapperT(fs::File* fp = nullptr) : DataWrapper{}, _fp { fp } {
      need_transaction = true;
    }
    int read(uint8_t *buf, uint32_t len) override { return _fp->read(buf, len); }
    void skip(int32_t offset) override { _fp->seek(offset, SeekCur); }
    bool seek(uint32_t offset) override { return _fp->seek(offset, SeekSet); }
    bool seek(uint32_t offset, SeekMode mode) { return _fp->seek(offset, mode); }
    void close(void) override { if (_fp) _fp->close(); }
    int32_t tell(void) override { return _fp->position(); }
protected:
    fs::File *_fp;
  };

  template <>
  struct DataWrapperT<fs::FS> : public DataWrapperT<fs::File> {
    DataWrapperT(fs::FS* fs, fs::File* fp = nullptr) : DataWrapperT<fs::File> { fp }, _fs { fs } {
#if defined (LGFX_FILESYSTEM_SD_INSTANCE)
      need_transaction = (fs == &LGFX_FILESYSTEM_SD_INSTANCE);
#endif
    }
    bool open(const char* path) override
    {
      _file = _fs->open(path, "r");
      DataWrapperT<fs::File>::_fp = &_file;
      return _file;
    }

protected:
    fs::FS* _fs;
    fs::File _file;
  };

  #if defined (LGFX_FILESYSTEM_SD_INSTANCE)
  template <>
  struct DataWrapperT<fs::SDFS> : public DataWrapperT<fs::FS> {
    DataWrapperT(fs::FS* fs, fs::File* fp = nullptr) : DataWrapperT<fs::FS>(fs, fp) {}
  };
  #endif
 #endif
#endif
/*
#if defined (ARDUINO)
 #if defined (FS_H)

  struct FileWrapper : public DataWrapper
  {
private:
#if defined (_SD_H_)
    bool _check_need_transaction(void) const { return _fs == &SD; }
#elif defined (_SPIFFS_H_)
    bool _check_need_transaction(void) const { return _fs != &SPIFFS; }
#else
    bool _check_need_transaction(void) const { return false; }
#endif

public:
    FileWrapper() : DataWrapper()
    {
#if defined (_SD_H_)
      _fs = &SD;
#elif defined (_SPIFFS_H_)
      _fs = &SPIFFS;
#else
      _fs = nullptr;
#endif
      need_transaction = _check_need_transaction();
      _fp = nullptr;
    }

    fs::FS* _fs;
    fs::File *_fp;
    fs::File _file;

    FileWrapper(fs::FS& fs, fs::File* fp = nullptr) : DataWrapper(), _fs(&fs), _fp(fp) { need_transaction = _check_need_transaction(); }
    void setFS(fs::FS& fs) {
      _fs = &fs;
      need_transaction = _check_need_transaction();
    }

    bool open(fs::FS& fs, const char* path)
    {
      setFS(fs);
      return open(path);
    }
    bool open(const char* path) override
    {
      _file = _fs->open(path, "r");
      _fp = &_file;
      return _file;
    }
    int read(uint8_t *buf, uint32_t len) override { return _fp->read(buf, len); }
    void skip(int32_t offset) override { seek(offset, SeekCur); }
    bool seek(uint32_t offset) override { return seek(offset, SeekSet); }
    bool seek(uint32_t offset, SeekMode mode) { return _fp->seek(offset, mode); }
    void close(void) override { if (_fp) _fp->close(); }
    int32_t tell(void) override { return _fp->position(); }
  };
 #else
  // dummy
  struct FileWrapper : public DataWrapper
  {
    FileWrapper() : DataWrapper()
    {
      need_transaction = true;
    }
    void* _fp;

    template <typename T>
    void setFS(T& fs) {}

    bool open(const char*, const char*) { return false; }
    int read(uint8_t*, uint32_t) override { return false; }
    void skip(int32_t) override { }
    bool seek(uint32_t) override { return false; }
    bool seek(uint32_t, int) { return false; }
    void close() override { }
    int32_t tell(void) override { return 0; }
  };

 #endif
#else // ESP-IDF

  struct FileWrapper : public DataWrapper
  {
    FileWrapper() : DataWrapper()
    {
      need_transaction = true;
    }
    FILE* _fp;
    bool open(const char* path) override { return (_fp = fopen(path, "r")); }
    int read(uint8_t *buf, uint32_t len) override { return fread((char*)buf, 1, len, _fp); }
    void skip(int32_t offset) override { seek(offset, SEEK_CUR); }
    bool seek(uint32_t offset) override { return seek(offset, SEEK_SET); }
    bool seek(uint32_t offset, int origin) { return fseek(_fp, offset, origin); }
    void close() override { if (_fp) fclose(_fp); }
    int32_t tell(void) override { return ftell(_fp); }
  };

#endif
*/

//----------------------------------------------------------------------------
 }
}
