/*****************************************************************************
 *
 * Copyright (C) 2009 - 2025  Florian Pose <fp@igh.de>
 *
 * This file is part of the QtPdCom library.
 *
 * The QtPdCom library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 *
 * The QtPdCom library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with the QtPdCom Library. If not, see <http://www.gnu.org/licenses/>.
 *
 ****************************************************************************/

#include "MessageModelUnion.h"
using QtPdCom::MessageModelUnion;

#include "MessageModel.h"

#include <QAbstractProxyModel>

#if PD_DEBUG_MESSAGE_MODEL
#include <QDebug>
#endif

/*****************************************************************************
 * MessageModelUnion::Impl
 ****************************************************************************/

struct MessageModelUnion::Impl
{
        Impl() = delete;
        Impl(MessageModelUnion *parent):
            parentModel {parent}
        {}

        MessageModelUnion *const parentModel;

        struct SourceModel
        {
                ~SourceModel() { QObject::disconnect(connection); }

                QAbstractItemModel *model;
                QString name;
                const QtPdCom::Message *announcedMessage;
                QMetaObject::Connection connection;
        };
        QList<SourceModel> sourceModels;

        struct SourceEntry
        {
                SourceModel *sourceModel;
                int row;  // original row in source model
        };
        QVector<SourceEntry> mergedRows;

        enum { SortColumn = 1 };

        void rebuildMergedRows()
        {
            parentModel->beginResetModel();
            mergedRows.clear();

            for (auto &sourceModel : sourceModels) {
                int rows = sourceModel.model->rowCount();
                for (int row = 0; row < rows; ++row) {
                    mergedRows.append({&sourceModel, row});
                }
            }

            std::sort(
                    mergedRows.begin(),
                    mergedRows.end(),
                    [this](const SourceEntry &a, const SourceEntry &b) {
                        QVariant va = a.sourceModel->model->data(
                                a.sourceModel->model->index(
                                        a.row,
                                        SortColumn));
                        QVariant vb = b.sourceModel->model->data(
                                b.sourceModel->model->index(
                                        b.row,
                                        SortColumn));
                        return vb.toString() < va.toString();
                    });

#if PD_DEBUG_MESSAGE_MODEL
            qDebug() << __func__ << "(): Now " << mergedRows.size()
                     << " rows.";
#endif
            parentModel->endResetModel();
        }

        void currentMessage(
                SourceModel *sourceModel,
                const QtPdCom::Message *message)
        {
#if PD_DEBUG_MESSAGE_MODEL
            qDebug() << __func__ << sourceModel << message;
#endif
            sourceModel->announcedMessage = message;
            updateCurrentMessage();
        }

        void updateCurrentMessage()
        {
            const QtPdCom::Message *candidate {nullptr};
            for (auto &sourceModel : sourceModels) {
                if (sourceModel.announcedMessage) {
                    if (not candidate
                        or sourceModel.announcedMessage->getType()
                                > candidate->getType()) {
                        candidate = sourceModel.announcedMessage;
                    }
                }
            }
#if PD_DEBUG_MESSAGE_MODEL
            qDebug() << __func__ << candidate;
#endif
            emit parentModel->currentMessage(candidate);
        }
};

/*****************************************************************************
 * MessageModelUnion
 ****************************************************************************/

/** Constructor.
 */
MessageModelUnion::MessageModelUnion(QObject *parent):
    QAbstractTableModel(parent),
    impl(std::unique_ptr<Impl>(new MessageModelUnion::Impl(this)))
{}

/****************************************************************************/

/** Destructor.
 */
MessageModelUnion::~MessageModelUnion()
{
    clearSourceModels();
}

/****************************************************************************/

void MessageModelUnion::addSourceModel(
        QAbstractItemModel *model,
        QString sourceName)
{
    impl->sourceModels.push_back(
            {model,
             sourceName,
             nullptr,  // announced message
             QMetaObject::Connection()});
    auto *sourceModel = &impl->sourceModels.last();

    connect(model, &QAbstractTableModel::rowsInserted, [this]() {
        impl->rebuildMergedRows();
    });
    connect(model, &QAbstractTableModel::rowsRemoved, [this]() {
        impl->rebuildMergedRows();
    });
    connect(model, &QAbstractTableModel::dataChanged, [this]() {
        impl->rebuildMergedRows();
    });
    connect(model, &QAbstractTableModel::layoutChanged, [this]() {
        impl->rebuildMergedRows();
    });
    connect(model, &QAbstractTableModel::modelReset, [this]() {
        impl->rebuildMergedRows();
    });

    // find a QtPdCom::MessageModel after a chain of QAbstractProxyModels
    QtPdCom::MessageModel *messageModel {nullptr};
    while (true) {
        auto *proxy {qobject_cast<QAbstractProxyModel *>(model)};
        if (not proxy) {
            messageModel = qobject_cast<QtPdCom::MessageModel *>(model);
            break;
        }
        model = proxy->sourceModel();
    }

    if (messageModel) {
        sourceModel->connection =
                connect(messageModel,
                        &MessageModel::currentMessage,
                        [this, sourceModel](const QtPdCom::Message *message) {
                            impl->currentMessage(sourceModel, message);
                        });
    }

    impl->rebuildMergedRows();
}

/****************************************************************************/

void MessageModelUnion::removeSourceModel(QAbstractItemModel *model)
{
    auto it = std::remove_if(
            impl->sourceModels.begin(),
            impl->sourceModels.end(),
            [model](const Impl::SourceModel &s) { return s.model == model; });
    impl->sourceModels.erase(it, impl->sourceModels.end());

    impl->rebuildMergedRows();
}

/****************************************************************************/

void MessageModelUnion::clearSourceModels()
{
    beginResetModel();
    impl->sourceModels.clear();
    endResetModel();
}

/****************************************************************************/

int MessageModelUnion::rowCount(const QModelIndex &) const
{
    return impl->mergedRows.count();
}

/****************************************************************************/

int MessageModelUnion::columnCount(const QModelIndex &) const
{
    return 4;
}

/****************************************************************************/

QVariant MessageModelUnion::data(const QModelIndex &index, int role) const
{
    QVariant ret;

    if (index.isValid() and index.row() < impl->mergedRows.size()) {
        auto entry {impl->mergedRows[index.row()]};
        QModelIndex sourceIndex =
                entry.sourceModel->model->index(entry.row, index.column());
        if (index.column() >= TextColumn and index.column() < SourceColumn) {
            ret = entry.sourceModel->model->data(sourceIndex, role);
        }
        else if (index.column() == SourceColumn and role == Qt::DisplayRole) {
            ret = entry.sourceModel->name;
        }
    }

#if PD_DEBUG_MESSAGE_MODEL
    qDebug() << __func__ << index << role << ret;
#endif
    return ret;
}

/****************************************************************************/

QVariant
MessageModelUnion::headerData(int section, Qt::Orientation o, int role) const
{
    if (role == Qt::DisplayRole && o == Qt::Horizontal) {
        switch (section) {
            case TextColumn:
                return tr("Message");

            case TimeOccurredColumn:
                return tr("Time");

            case TimeResetColumn:
                return tr("Reset");

            case SourceColumn:
                return tr("Source");

            default:
                return QVariant();
        }
    }
    else {
        return QVariant();
    }
}

/****************************************************************************/

Qt::ItemFlags MessageModelUnion::flags(const QModelIndex &index) const
{
    Qt::ItemFlags ret;

    if (index.isValid() and index.row() < impl->mergedRows.size()) {
        auto entry {impl->mergedRows[index.row()]};
        QModelIndex sourceIndex =
                entry.sourceModel->model->index(entry.row, index.column());
        if (index.column() >= TextColumn and index.column() < SourceColumn) {
            ret = entry.sourceModel->model->flags(sourceIndex);
        }
        else if (index.column() == SourceColumn) {
            QModelIndex textIndex =
                    entry.sourceModel->model->index(entry.row, TextColumn);
            ret = entry.sourceModel->model->flags(textIndex);
        }
    }

#if PD_DEBUG_MESSAGE_MODEL
    qDebug() << __func__ << index << ret;
#endif
    return ret;
}

/****************************************************************************/

bool MessageModelUnion::canFetchMore(const QModelIndex &index) const
{
#if PD_DEBUG_MESSAGE_MODEL
    qDebug() << __func__ << index;
#endif
    if (not index.isValid()) {
        // root layer - return true if *any* model can fetch more rows
        for (auto &sourceModel : impl->sourceModels) {
            if (sourceModel.model->canFetchMore(index)) {
#if PD_DEBUG_MESSAGE_MODEL
                qDebug() << sourceModel.model << sourceModel.name << "yes";
#endif
                return true;
            }
        }
#if PD_DEBUG_MESSAGE_MODEL
        qDebug() << "no";
#endif
        return false;
    }
    else if (index.row() < impl->mergedRows.size()) {
        auto entry {impl->mergedRows[index.row()]};
        auto sourceIndex {
                entry.sourceModel->model->index(entry.row, index.column())};
        auto canFetch {entry.sourceModel->model->canFetchMore(sourceIndex)};
#if PD_DEBUG_MESSAGE_MODEL
        qDebug() << "specific" << canFetch;
#endif
        return canFetch;
    }
    else {
#if PD_DEBUG_MESSAGE_MODEL
        qDebug() << "invalid";
#endif
        return false;
    }
}

/****************************************************************************/

void MessageModelUnion::fetchMore(const QModelIndex &index)
{
#if PD_DEBUG_MESSAGE_MODEL
    qDebug() << __func__ << index;
#endif
    if (not index.isValid()) {
        // root layer - pass on to *any* model that can fetch more rows
        for (auto &sourceModel : impl->sourceModels) {
            if (sourceModel.model->canFetchMore(index)) {
#if PD_DEBUG_MESSAGE_MODEL
                qDebug() << __func__ << sourceModel.model << sourceModel.name;
#endif
                sourceModel.model->fetchMore(index);
            }
        }
    }
    else if (index.row() < impl->mergedRows.size()) {
        auto entry {impl->mergedRows[index.row()]};
        auto sourceIndex {
                entry.sourceModel->model->index(entry.row, index.column())};
        entry.sourceModel->model->fetchMore(sourceIndex);
    }
}

/****************************************************************************/
