/*****************************************************************************
 *
 * Testmanager - Graphical Automation and Visualisation Tool
 *
 * Copyright (C) 2025  Florian Pose <fp@igh.de>
 *
 * This file is part of Testmanager.
 *
 * Testmanager is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option)
 * any later version.
 *
 * Testmanager is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with Testmanager. If not, see <http://www.gnu.org/licenses/>.
 *
 ****************************************************************************/

#include "RotorPlugin.h"

#include "DataSlot.h"
#include "DataModel.h"
#include "DataNode.h"
#include "SlotModel.h"
#include "SlotModelCollection.h"

#include <QtPdWidgets2/Rotor.h>
#include <QTimer>

#include <QDebug>
#include <QJsonObject>
#include <QJsonArray>

/****************************************************************************/

class RotorSlotModel;

/****************************************************************************/

class RotorSlot: public DataSlot
{
    public:
        explicit RotorSlot(SlotNode *parent):
            DataSlot(parent)
        {}

        bool supportsRemovingByUser() const override { return true; }

        bool nodeSetData(int, const QVariant &);
};

/****************************************************************************/

class RotorSlotModel: public SlotModel
{
        Q_OBJECT

    public:
        RotorSlotModel(
                SlotModelCollection &parent,
                WidgetContainer &container,
                Pd::Rotor *rotor):
            SlotModel(parent, container),
            rotor(rotor)
        {
            SlotModelCollection *root = &SlotNode::getRoot();
            connect(root,
                    &QAbstractItemModel::rowsRemoved,
                    this,
                    [this](const QModelIndex &, int, int) {
                        reconnectVariables();
                    });
        }

        virtual void fromJson(const QJsonValue &val) override
        {
            foreach (QJsonValue arrayValue, val.toArray()) {
                if (arrayValue.isObject()) {
                    auto node = new RotorSlot(this);
                    node->fromJson(arrayValue.toObject());
                }
            }
        }

        virtual AcceptedDataConnectionModes canInsertDataNodes(
                QList<DataNode *> const &nodes,
                int,
                QString &err,
                bool replace) const override
        {
            if (!replace && getChildCount() > 0) {
                err = QObject::tr(
                        "Already connected. Press Shift to replace.");
                return SlotModel::AcceptedDataConnectionModes::None;
            }
            if (nodes.size() != 1) {
                err = QObject::tr("Exactly one connection possible.");
                return SlotModel::AcceptedDataConnectionModes::None;
            }
            const auto var = nodes[0]->getVariable();
            if (var.empty()) {
                err = QObject::tr("Please select a Variable");
                return SlotModel::AcceptedDataConnectionModes::None;
            }
            return SlotModel::AcceptedDataConnectionModes::Scalar;
        }

        virtual bool insertDataNodes(
                QList<DataNode *> const &nodes,
                const Selector &selector,
                int row,
                QString &err,
                bool replace) override
        {
            if (canInsertDataNodes(nodes, row, err, replace)
                != SlotModel::AcceptedDataConnectionModes::Scalar) {
                return false;
            }

            if (replace) {
                clear();
            }

            const auto var = nodes[0]->getVariable();
            QtPdCom::Transmission transmission {
                    std::chrono::milliseconds(100)};
            if (var.isWriteable()) {
                // assuming variable is parameter which does not support
                // period
                transmission = QtPdCom::event_mode;
            }

            auto child = new RotorSlot(this);
            child->period = transmission;
            child->selector = selector;
            child->url = nodes[0]->nodeUrl();
            try {
                rotor->setSpeedVariable(
                        var,
                        selector.toPdComSelector(),
                        transmission,
                        child->scale,
                        child->offset,
                        child->tau);
            }
            catch (const PdCom::Exception &e) {
                SlotModelCollection &root = SlotNode::getRoot();
                root.remove(root.index(0, 0, SlotNode::getIndex()));
                err = QObject::tr("Error occured during subscribing"
                                  " to a variable: ")
                        + e.what();
                return false;
            }
            return true;
        }

        virtual void appendDataSources(DataModel *dataModel) override
        {
            SlotModel::appendDataSources(dataModel);
            reconnectVariables();
        }

        void reconnectVariables()
        {
            const DataModel *dataModel = getRoot().dataModel;
            rotor->clearSpeedVariable();
            if (not getChildCount()) {
                return;
            }
            const auto &item = *static_cast<DataSlot *>(getChildNode(0));
            auto var = dataModel->findDataNode(item.url);
            if (!var || var->getVariable().empty()) {
                return;
            }
            rotor->setSpeedVariable(
                    var->getVariable(),
                    item.selector.toPdComSelector(),
                    item.period,
                    item.scale,
                    item.offset,
                    item.tau);
        }

        virtual void clear() override
        {
            rotor->clearSpeedVariable();
            getRoot().removeRows(0, getChildCount(), getIndex());
        }

    private:
        Pd::Rotor *rotor;
};

/****************************************************************************/

bool RotorSlot::nodeSetData(int col, const QVariant &data)
{
    if (not DataSlot::nodeSetData(col, data)) {
        return false;
    }

    auto model = dynamic_cast<RotorSlotModel *>(getParentNode());
    if (model) {
        model->reconnectVariables();
    }
    return true;
}

/*****************************************************************************
 * RotorPlugin
 ****************************************************************************/

QString RotorPlugin::name() const
{
    return tr("Rotor");
}

/****************************************************************************/

QIcon RotorPlugin::icon() const
{
    return QIcon(":/images/plugin-rotor.svg");
}

/****************************************************************************/

SlotModel *RotorPlugin::createSlotModel(
        QWidget *widget,
        SlotModelCollection &parent,
        WidgetContainer &container) const
{
    auto rotor = qobject_cast<Pd::Rotor *>(widget);
    if (!rotor) {
        return nullptr;
    }
    return new RotorSlotModel(parent, container, rotor);
}

/****************************************************************************/

QWidget *RotorPlugin::createWidget(QWidget *parent) const
{
    return new Pd::Rotor(parent);
}

/****************************************************************************/

#include "RotorPlugin.moc"
