/*****************************************************************************
 *
 * Testmanager - Graphical Automation and Visualisation Tool
 *
 * Copyright (C) 2018  Florian Pose <fp@igh.de>
 *
 * This file is part of Testmanager.
 *
 * Testmanager is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option)
 * any later version.
 *
 * Testmanager is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with Testmanager. If not, see <http://www.gnu.org/licenses/>.
 *
 ****************************************************************************/

#include "DetachableTabWidget.h"

#include "DetachedDockWidget.h"
#include "MainWindow.h"
#include "TabDialog.h"
#include "TabPage.h"

#include <QEvent>
#include <QMimeData>
#include <QMouseEvent>
#include <QScrollArea>
#include <QTabBar>
#include <QVBoxLayout>

/****************************************************************************/

DetachableTabWidget::DetachableTabWidget(MainWindow *parent):
    QTabWidget(parent),
    mainWindow(parent)
{
    tabBar()->setContextMenuPolicy(Qt::CustomContextMenu);
    connect(this,
            SIGNAL(tabBarDoubleClicked(int)),
            this,
            SLOT(tabBarDoubleClicked(int)));
    connect(tabBar(),
            &QTabBar::customContextMenuRequested,
            this,
            &DetachableTabWidget::showTabContextMenu);
}

/****************************************************************************/

void DetachableTabWidget::attachTab(DetachedDockWidget *dockWidget)
{
    auto scrollAreas = dockWidget->findChildren<QScrollArea *>();
    if (scrollAreas.size() != 1) {
        return;
    }
    auto scrollArea {scrollAreas[0]};

    auto tabPages = scrollArea->findChildren<TabPage *>();
    if (tabPages.size() != 1) {
        return;
    }
    auto tabPage {tabPages[0]};

    scrollArea->setParent(this);
    addTab(scrollArea, tabPage->getName());
    scrollArea->show();

    mainWindow->removeDetachedTab(dockWidget);

    dockWidget->close();
    dockWidget->deleteLater();
}

/****************************************************************************/

void DetachableTabWidget::renameTab(int index)
{
    if (index < 0) {
        return;
    }

    auto *dialog = new TabDialog(this, index);
    connect(dialog, &QDialog::accepted, this, [this]() {
        mainWindow->setLayoutChanged();
    });
    connect(dialog, &QDialog::finished, dialog, &QObject::deleteLater);
    dialog->open();
}

/****************************************************************************/

void DetachableTabWidget::detachTab(int index)
{
    auto scrollArea = dynamic_cast<QWidget *>(widget(index));
    if (not scrollArea) {
        return;
    }
    auto tabPages = scrollArea->findChildren<TabPage *>();
    if (tabPages.size() != 1) {
        return;
    }
    auto tabPage = tabPages[0];

    removeTab(index);

    DetachedDockWidget *dockWidget = new DetachedDockWidget(mainWindow);
    dockWidget->setWindowTitle(tabPage->getName());
    dockWidget->resize(500, 500);
    dockWidget->show();

    scrollArea->setParent(dockWidget);
    dockWidget->setWidget(scrollArea);
    scrollArea->show();

    dockWidget->setFloating(true);

    mainWindow->appendDetachedTab(dockWidget, Qt::RightDockWidgetArea);
}

/****************************************************************************/

void DetachableTabWidget::showTabContextMenu(const QPoint &pos)
{
    if (!mainWindow->getEditMode()) {
        return;
    }

    int index = tabBar()->tabAt(pos);
    if (index < 0) {
        return;
    }

    QMenu menu;

    QAction *renameAction = menu.addAction(tr("Rename Tab"));
    menu.setDefaultAction(renameAction);
    connect(renameAction, &QAction::triggered, this, [=]() {
        this->renameTab(index);
    });

    menu.addSeparator();

    menu.addAction(tr("Detach Tab"), [=]() { this->detachTab(index); });
    menu.addAction(tr("Remove Tab"), [=]() { this->removeTab(index); });

    menu.exec(tabBar()->mapToGlobal(pos));
}

/****************************************************************************/

void DetachableTabWidget::tabBarDoubleClicked(int index)
{
    if (!mainWindow->getEditMode()) {
        return;
    }

    renameTab(index);
}

/****************************************************************************/
