/*
    Данный пример демонстрирует процедуру вывода значений портов LTR43.
    По умолчанию идет работа с первым слотом первого крейта, но это можно изменить
        через параметры командной строки:
    ltr43_sec_pulse  slot crate_serial srvip
    где:
        slot         - номер слота (от 1 до 16)
        crate_serial - серийный номер крейта (если крейтов больше одного)
        srvip        - ip-адрес программы ltrd/LtrServer (если он запущен на другой машине)
    Параметры опциональны: можно указать только слот, слот и серийный номер крейта или все

    Пример выводит сперва на нечетные выходы 1, а на четные 0, а затем раз в пол
    секунды (интервал выдерживается засыпанием, не является очень точным и
    может несколько изменяться) сменяет значения на противоположные.

    Вывод идет до нажатия любой клавиши на Windows или  CTRL+C на Linux.

    Сборка в VisualStudio:
    Для того чтобы собрать проект в Visual Studio, измените путь к заголовочным файлам
    (Проект (Project) -> Свойства (Properties) -> Свойства конфигурации (Configuration Properties)
    -> С/С++ -> Общие (General) -> Дополнительные каталоги включения (Additional Include Directories))
    на нужный в зависимаости от пути установки библиотек (ltr43api.h  и остальные заголовочные
    файлы должны находится в поддиректории ltr/include относительно указанного пути)
    и измените путь к .lib файлам на <путь установки библиотек>/lib/msvc
    (Проект (Project) -> Свойства (Properties) -> Свойства конфигурации (Configuration Properties) ->
    Компоновщик (Linker) -> Общие (General) -> Дополнительные каталоги библиотек (Additional Library Directories)).

    Внимание!: Если Вы собираете проект под Visual Studio и у Вас некорректно
    отображается вывод русских букв, то нужно изменить кодировку:
    выберите Файл (File) -> Дополнительные параметры сохранения (Advanced Save Options)...
    и в поле Кодировка (Encoding) выберите Юникод (UTF8, с сигнатурой)/Unicode (UTF-8 with signature)
    и сохраните изменения в файле. А также следует убедится, что в настройках
    консоли стоит шрифт с поддержкой русского языка (например Consolas).
*/



#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef _WIN32
#include <locale.h>
#include <conio.h>
#else
#include <signal.h>
#include <unistd.h>
#endif

#include "ltr/include/ltr43api.h"

#define RECV_BLOCK_SIZE 10000


/* признак необходимости завершить сбор данных */
static int f_out = 0;

#ifndef _WIN32
/* Обработчик сигнала завершения для Linux */
static void f_abort_handler(int sig) {
    f_out = 1;
}
#endif



typedef struct {
    int slot;
    const char *serial;
    DWORD addr;
} t_open_param;

/* Разбор параметров командной строки. Если указано меньше, то используются
 * значения по умолчанию:
 * 1 параметр - номер слота (от 1 до 16)
 * 2 параметр - серийный номер крейта
 * 3 параметр - ip-адрес сервера */
static int f_get_params(int argc, char** argv, t_open_param* par) {
    int err = 0;
    par->slot = LTR_CC_CHNUM_MODULE1;
    par->serial = "";
    par->addr = LTRD_ADDR_DEFAULT;


    if (argc > 1)
        par->slot = atoi(argv[1]);
    if (argc > 2)
        par->serial = argv[2];
    if (argc > 3) {
        int a[4],i;
        if (sscanf(argv[3], "%d.%d.%d.%d", &a[0], &a[1], &a[2], &a[3])!=4) {
            fprintf(stderr, "Неверный формат IP-адреса!!\n");
            err = -1;
        }

        for (i=0; (i < 4) && !err; i++) {
            if ((a[i]<0) || (a[i] > 255)) {
                fprintf(stderr, "Недействительный IP-адрес!!\n");
                err = -1;
            }
        }

        if (!err) {
            par->addr = (a[0] << 24) | (a[1]<<16) | (a[2]<<8) | a[3];
        }
    }
    return err;
}
/*------------------------------------------------------------------------------------------------*/
int main(int argc, char *argv[]) {
    INT err = LTR_OK;
    TLTR43 hltr43;
    t_open_param par;

#ifndef _WIN32
    struct sigaction sa;
    memset(&sa, 0, sizeof(sa));
    /* В ОС Linux устанавливаем свой обработчик на сигнал закрытия,
       чтобы завершить сбор корректно */
    sa.sa_handler = f_abort_handler;
    sigaction(SIGTERM, &sa, NULL);
    sigaction(SIGINT, &sa, NULL);
    sigaction(SIGABRT, &sa, NULL);
#endif
#ifdef _WIN32
    /* для вывода русских букв в консоль для ОС Windows*/
    setlocale(LC_ALL, "");
#endif

    err = f_get_params(argc, argv, &par);

    if (err==LTR_OK) {
        LTR43_Init(&hltr43);
        err = LTR43_Open(&hltr43, par.addr, LTRD_PORT_DEFAULT, par.serial, par.slot);
        if (err != LTR_OK) {
            fprintf(stderr, "Не удалось установить связь с модулем. Ошибка %d (%s)\n",
                    err, LTR43_GetErrorString(err));
        } else {
            DWORD wr_val = 0x55555555;

            printf("Модуль открыт успешно:\n");
            printf("  Название:        %s\n", hltr43.ModuleInfo.Name);
            printf("  Серийный номер:  %s\n", hltr43.ModuleInfo.Serial);
            printf("  Версия прошивки: %s\n", hltr43.ModuleInfo.FirmwareVersion);
            printf("  Дата прошивки:   %s\n", hltr43.ModuleInfo.FirmwareDate);


            hltr43.IO_Ports.Port1 = LTR43_PORT_DIR_OUT;
            hltr43.IO_Ports.Port2 = LTR43_PORT_DIR_OUT;
            hltr43.IO_Ports.Port3 = LTR43_PORT_DIR_OUT;
            hltr43.IO_Ports.Port4 = LTR43_PORT_DIR_OUT;

            err = LTR43_Config(&hltr43);
            if (err != LTR_OK) {
                fprintf(stderr, "Невозможно сконфигурировать модуль. Ошибка %d: %s\n",
                        err, LTR43_GetErrorString(err));
            }
            printf("Запущена генерация импульсов. Для останова нажмите %s\n",
#ifdef _WIN32
                   "любую клавишу"
#else
                   "CTRL+C"
#endif
                   );

            while ((err == LTR_OK) && !f_out) {

                err = LTR43_WritePort(&hltr43, wr_val);
                if (err!=LTR_OK) {
                    fprintf(stderr, "Невозможно записать данные в модуль. Ошибка %d:%s\n",
                            err, LTR43_GetErrorString(err));
                } else {
                    LTRAPI_SLEEP_MS(500);
                    wr_val ^= 0xFFFFFFFF;
                }

#ifdef _WIN32
                /* проверка нажатия клавиши для выхода */
                if (err==LTR_OK) {
                    if (_kbhit())
                        f_out = 1;
                }
#endif
            }
        }

        if (LTR43_IsOpened(&hltr43) == LTR_OK)
            LTR43_Close(&hltr43);
    }
    return err;
}
