#ifndef LMATH_ACDC_H
#define LMATH_ACDC_H

#include "lmath_defs.h"
#include "lmath_window.h"


#ifdef __cplusplus
extern "C" {
#endif

/***************************************************************************//**
    Оценка постоянной и переменной (по RMS) составляющей входного временного
    сигнала. Для сглаживания разрывов на краях сигнала функция использует заранее
    рассчитанное с помощью lmath_window_ctx_create() окно.

    @param[in]  src         Массив с отсчетами исходного временного сигнала
                            (размер определяется размером окна)
    @param[in]  winctx      Контекст с параметрами окна
    @param[out] tmp_buf     Массив для хранения временных результатов, что позволяет
                            не выделять при каждом выполнение функции память.
                            Должен быть размером равным размеру окна.
                            Входное содержимое не имеет значения и затирается
                            промежуточными данными. Если не используется рассчет
                            AC (ac == NULL) и входной сигнал не нужен после
                            выполнения данной функции, то можно использовать
                            src в качестве временного буфера. Если выполняется
                            и рассчет AC, то в качестве данного буфера обязательно
                            должен быть отдельно созданный массив.
    @param[out] dc          В этой переменной возвращается рассчитанное значение
                            постоянной составляющей. Может быть NULL.
    @param[out] ac          В этой переменной возвращается рассчитанное значение
                            переменной составляющей (RMS). Может быть NULL.
    @return                 Код ошибки
*******************************************************************************/
t_lmath_errs lmath_acdc_estimation_with_window(const double *src,
                                              const t_lmath_window_ctx winctx,
                                              double *tmp_buf,
                                              double *dc, double *ac);


/***************************************************************************//**
    Оценка постоянной и переменной (по RMS) составляющей входного временного
    сигнала. Функция использует окно для сглаживания разрывов на краях сигнала.
    Упращенная версия lmath_acdc_estimation_with_window(), которая не требует
    отдельно создавать окно, но требует вычисление коэффициентов окна при каждом
    вызове функции и не позволяет явно определить окно.

    @param[in]  src     Массив с отсчетами исходного временного сигнала
    @param[in]  size    Количество отсчетов во входном массиве
    @param[out] dc      В этой переменной возвращается рассчитанное значение
                            постоянной составляющей. Может быть NULL.
    @param[out] ac      В этой переменной возвращается рассчитанное значение
                            переменной составляющей. Может быть NULL.
    @return             Код ошибки
*******************************************************************************/
t_lmath_errs lmath_acdc_estimation(const double *src, int size, double *dc, double *ac);

#ifdef __cplusplus
}
#endif

#endif // LMATH_ACDC_H
