#ifndef LMATH_SPECTRUM_FREQ_H
#define LMATH_SPECTRUM_FREQ_H

#include "lmath_window.h"
#include "lmath_config.h"

#ifdef __cplusplus
extern "C" {
#endif

/***************************************************************************//**
    Поиск значения частоты с максимальной мощностью и мощности сигнала на этой
    частоте по амплитудному спектру.

    @param[in]  amp_spectrum    Амплитудный спектр входного сигнала
    @param[in]  size            Количество отсчетов в спектре
    @param[in]  df              Изменение частоты между соседними отсчетами
                                  спектра в Гц
    @param[in]  winpar          Указатель на структуру с параметрами
                                   примененного перед вычислением спектра окна.
    @param[in]  search_idx_min  Если значение больше нуля, то ищется частота в
                                  частотном интетервале, начиная указанного
                                  отсчета amp_spectrum включительно.
                                  Если при этом search_idx_min равно search_idx_max,
                                  то ищется частота в районе указанной.
    @param[in]  search_idx_max  Если значение больше нуля, то ищется частота в
                                  частотном интетервале до казанного отсчета
                                  amp_spectrum включительно.
    @param[out] peak_freq       В этой переменной возвращается рассчитанное значение
                                  частоты, соответствующей максимальной мощности.
                                  Может быть NULL.
    @param[out] peak_pwr        В этой переменной возвращается рассчитанное значение
                                  мощности сигнала на найденной частоте.
                                  Может быть NULL.
    @return                     Код ошибки
*******************************************************************************/
t_lmath_errs lmath_spectrum_find_peak_freq(const double *amp_spectrum, int size, double df,
                                           const t_lmath_window_params *winpar,
                                           int search_idx_min, int search_idx_max,
                                           double *peak_freq, double *peak_pwr);


#ifdef LMATH_USE_FFT
/***************************************************************************//**
    Поиск значения частоты с максимальной мощностью и мощности сигнала на этой
    частоте по временному сигналу. Функция внутри себя применяет БПФ и анализирует
    спектр. Если уже рассчитан спектр, то стоит использовать
    lmath_find_peak_freq_spectrum()

    @param[in]  src             Массив с отсчетами исходного временного сигнала
    @param[in]  size            Количество отсчетов во входном массиве
    @param[in]  dt              Интервал времени между отсчетами входного
                                  сигнала в секундах
    @param[in]  search_freq     Если значение больше нуля, то ищется частота в
                                  окрестности указанной, иначе - во всей частотной
                                  области (кроме постоянной составляющей)
    @param[out] peak_freq       В этой переменной возвращается рассчитанное значение
                                  частоты, соответствующей максимальной мощности.
                                  Может быть NULL.
    @param[out] peak_pwr        В этой переменной возвращается рассчитанное значение
                                  мощности сигнала на найденной частоте.
                                  Может быть NULL.
    @return                     Код ошибки
*******************************************************************************/
t_lmath_errs lmath_find_peak_freq(const double *src, int size, double dt,
                                  double search_freq_min, double search_freq_max,
                                  t_lmath_wintype wintype,
                                  double *peak_freq, double *peak_pwr);
#endif
#ifdef __cplusplus
}
#endif

#endif // LMATH_SPECTRUM_FREQ_H
