#ifndef LTR216API_H_
#define LTR216API_H_

#include "ltrapi.h"

#ifdef __cplusplus
extern "C" {
#endif


#ifdef _WIN32
    #ifdef LTR216API_EXPORTS
        #define LTR216API_DllExport(type) __declspec(dllexport) type APIENTRY
    #else
        #define LTR216API_DllExport(type) __declspec(dllimport) type APIENTRY
    #endif
#elif defined __GNUC__
    #define LTR216API_DllExport(type) __attribute__ ((visibility("default"))) type
#else
    #define LTR216API_DllExport(type) type
#endif


/***************************************************************************//**
  @addtogroup const_list Константы и перечисления.
  @{
  *****************************************************************************/

/** Максимальное число измерительных каналов в одном модуле */
#define LTR216_CHANNELS_CNT      16

/** Количество пользовательских диапазонов измерения */
#define LTR216_RANGES_CNT   2

/** Количество калибруемых диапазонов, включая вспомогательные, не используемые
    для пользовательских измерений */
#define LTR216_CBR_RANGES_CNT   3



/** Маска, соответствующая всем каналам АЦП */
#define LTR216_CHANNEL_MASK_ALL     0xFFFF

/** Код АЦП, соответствующее максимальному пиковому значению диапазона */
#define LTR216_ADC_SCALE_CODE_MAX   8000000

/** Базовая частота, подающаяся на АЦП модуля в Гц, до делителя */
#define LTR216_ADC_CLOCK            32000000
/** Максимальное значение делителя частоты синхронизации АЦП в многоканальном режиме */
#define LTR216_SYNC_FDIV_MAX        0xFFFFFF
/** Максимальное значение кода, задающего силу тока питания датчиков */
#define LTR216_ISRC_CODE_MAX        4095
/** Максимальное количество каналов в логической таблице */
#define LTR216_MAX_LCHANNEL         1024

/** Размер поля с названием модуля */
#define LTR216_NAME_SIZE            8
/** Размер поля с серийным номером модуля */
#define LTR216_SERIAL_SIZE          16


/** Адрес, с которого начинается пользовательская область flash-памяти */
#define LTR216_FLASH_USERDATA_ADDR   0x0
/** Размер пользовательской области flash-памяти */
#define LTR216_FLASH_USERDATA_SIZE   0x100000

/** Порог для максимального значения \f$V_{adj}\f$, который служит для обнаружения неисправности */
#define LTR216_VADJ_THRESHOLD       7


/** Максимальное значение поля [AdcOdrCode](@ref TLTR216_CONFIG::AdcOdrCode) при использовании SINC3 фильтра */
#define LTR216_SINC3_ODR_MAX_DIV   32767
/** Значение поля [ShortThresholdR](@ref TLTR216_CONFIG::ShortThresholdR)
   по умолчанию, устанавливаемое  при вызове LTR216_Init() */
#define LTR216_DEFAULT_SHORT_THRESH_R             10
/** Внутренняя емкость входов LTR216 в пФ. Значение используется при расчете времени
    проверки обрыва линии. */
#define LTR216_CH_INTERNAL_CAPACITY             1450
/** Значение поля [CableCapacityPerUnit](@ref TLTR216_CONFIG::CableCapacityPerUnit)
    по умолчанию, устанавливаемое  при вызове LTR216_Init() */
#define LTR216_DEFAULT_CABLE_CAPACITY_PER_UNIT   150
/** Значение поля [CableLength](@ref TLTR216_CONFIG::CableLength) по умолчанию,
     устанавливаемое при вызове LTR216_Init() */
#define LTR216_DEFAULT_CABLE_LENGTH              100
/**  Величина тестового источника тока (А) для реализации функции контроля обрыва линий  */
#define LTR216_I_BURNOUT                        5e-6


/** @brief Коды ошибок, специфичные для модуля LTR216.

   Коды ошибок, которые определены и используются только в ltr216api.
   Остальные коды ошибок, которые используются разными модулями, определены в ltrapi.h */
typedef enum {
    LTR216_ERR_ADC_ID_CHECK             = -10700, /**< Не удалось обнаружить микросхему АЦП */
    LTR216_ERR_ADC_RECV_SYNC_OVERRATE   = -10701, /**< Частота синхронизации АЦП превысила частоту преобразования */
    LTR216_ERR_ADC_RECV_INT_CYCLE_ERROR = -10702, /**< Ошибка внутреннего цикла чтения данных с АЦП */
    LTR216_ERR_ADC_REGS_INTEGRITY       = -10703, /**< Нарушена целостность регистров АЦП */
    LTR216_ERR_INVALID_ADC_SWMODE       = -10704, /**< Задано неверное значение режима опроса каналов АЦП */
    LTR216_ERR_INVALID_FILTER_TYPE      = -10705, /**< Задано неверное значение типа фильтра АЦП */
    LTR216_ERR_INVALID_ADC_ODR_CODE     = -10706, /**< Задано неверное значение кода, определяющего скорость преобразования АЦП */
    LTR216_ERR_INVALID_SYNC_FDIV        = -10707, /**< Задано неверное значение делителя частоты синхронизации АЦП */
    LTR216_ERR_INVALID_LCH_CNT          = -10708, /**< Задано неверное количество логических каналов */
    LTR216_ERR_INVALID_ISRC_CODE        = -10709, /**< Задан неверный код, определяющий силу тока питания датчиков */
    LTR216_ERR_CH_NOT_FOUND_IN_LTABLE   = -10710, /**< Указанный канал не был найден в логической таблице */
    LTR216_ERR_NO_CH_ENABLED            = -10711, /**< Ни один канал не был разрешен */
    LTR216_ERR_TARE_CHANNELS            = -10712, /**< Не удалось найти действительное значение при тарировке некоторых каналов */
    LTR216_ERR_TOO_MANY_LTABLE_CH       = -10713, /**< Превышено максимальное число каналов в основном цикле опроса АЦП */
    LTR216_ERR_TOO_MANY_LTABLE_BG_CH    = -10714, /**< Превышено максимальное число каналов в фоновом цикле опроса АЦП */
    LTR216_ERR_UNSUF_SW_TIME            = -10715, /**< Полученное время на коммутацию меньше заданного предела */
    LTR216_ERR_BAD_INIT_MEAS_STATUS     = -10716, /**< Измеренное значение начального параметра недействительно */
    LTR216_ERR_INVALID_CH_RANGE         = -10717, /**< Задан неверный код диапазона канала АЦП*/
    LTR216_ERR_INVALID_CH_NUM           = -10718, /**< Задан неверный номер физического канала АЦП */
    LTR216_ERR_UREF_MEAS_REQ            = -10719  /**< Для выполнения операции требуется измерить действительное значение напряжения \f$U_{ref}\f$ */
} e_LTR216_ERROR_CODES;

/** Диапазоны измерения модуля LTR216 */
typedef enum {
    LTR216_RANGE_35 = 0, /**< Диапазон \f$\pm35\f$ мВ */
    LTR216_RANGE_70 = 1, /**< Диапазон \f$\pm70\f$ мВ */
} e_LTR216_RANGE;

/** Типы фильтров АЦП модуля LTR216 */
typedef enum {
    LTR216_FILTER_SINC5_1   = 0, /**< Каскад из SINC5 и SINC1 фильтра */
    LTR216_FILTER_SINC3     = 1, /**< SINC3 фильтр */
    LTR216_FILTER_ENH_50HZ  = 2  /**< Фильтр с усиленной фильтрацией частоты 50 Гц */
} e_LTR216_FILTER_TYPE;

/** Режим коммутации каналов модуля */
typedef enum {
    LTR216_ADC_SWMODE_MULTICH_SYNC = 0, /**< Многоканальный режим. В нем коммутация каналов
                                             происходит с частотой, определяемой настройкой
                                             [SyncFreqDiv](@ref TLTR216_CONFIG::SyncFreqDiv). */
    LTR216_ADC_SWMODE_SIGNLECH_CONT = 1 /**< Одноканальный режим без коммутации. */
} e_LTR216_ADC_SWMODE;

/** @brief Флаги конфигурации

    Флаги, которые могут быть установлены в поле [Flags](@ref TLTR216_CONFIG::Flags)
    структуры конфигурации модуля, для определения специальных режимов конфигурации
    или поведения модуля.
*/
typedef enum {
    LTR216_CONFIG_FLAG_RAWTABLE  =  1 << 0 /**< Признак, что нужно использовать
                                                низкоуровневые настройки из
                                                [RawCfg](@ref TLTR216::RawCfg),
                                                а не рассчитывать их на основе
                                                пользовательской конфигурации.
                                                Не используется в штатных режимах
                                                измерения. */
} e_LTR216_CONFIG_FLAGS;

/** @brief Фоновые измерения

    Флаги, определяющие, какие вспомогательные измерения будут выполняться.
    Каждый флаг соответствует одному измерению.

    Для задания нескольких измерений флаги каждого измерения объединяются через
    логическое "ИЛИ".

    Также в данном перечислении определены группы измерений, названия которых
    начинаются с LTR216_BG_MEASGROUP, включающие набор однотипных измерений.

    Данные флаги используются при настройке фоновых измерений (поле
    [BgMeas](@ref TLTR216_CONFIG::BgMeas)), а также при выполнении отдельной
    проверки состояния линий и условий измерения с помощью LTR216_CheckMeasStatus().
*/
typedef enum {    
    LTR216_BG_MEAS_OFFS         = 1 << 0, /**< Измерение смещения нуля каналов измерения разностного напряжения */
    LTR216_BG_MEAS_UREF         = 1 << 1, /**< Измерение падения напряжения на опорном тензодатчике \f$U_{ref}\f$ */
    LTR216_BG_MEAS_UREF_OFFS    = 1 << 2, /**< Измерение смещения нуля для канала измерения \f$U_{ref}\f$ */
    LTR216_BG_MEAS_VADJ         = 1 << 3, /**< Измерение напряжения \f$V_{adj}\f$ */
    LTR216_BG_MEAS_UNEG         = 1 << 4, /**< Измерение напряжения \f$U_-\f$ относительно земли */
    LTR216_BG_MEAS_UREF_SHORT   = 1 << 5, /**< Проверка КЗ канала измерения \f$U_{ref}\f$ */
    LTR216_BG_MEAS_UREF_OPEN    = 1 << 6, /**< Проверка обрыва канала измерения \f$U_{ref}\f$ */
    LTR216_BG_MEAS_UNEG_OPEN    = 1 << 7, /**< Проверка обрыва линии \f$U_-\f$ */
    LTR216_BG_MEAS_CH_SHORT     = 1 << 8, /**< Проверка КЗ каналов измерения разностного напряжения */
    LTR216_BG_MEAS_CH_OPEN      = 1 << 9, /**< Проверка обрыва каналов измерения разностного напряжения */
    LTR216_BG_MEAS_CH_UX        = 1 << 10, /**< Измерение абсолютного значения падения напряжения на тензодатчиках
                                                в измерительных каналах */
    LTR216_BG_MEAS_CH_CM        = 1 << 11, /**< Измерение синфазного напряжения каналов  */

    /** Группа из всех измерений для проверки обрыва различных цепей */
    LTR216_BG_MEASGROUP_OPEN = LTR216_BG_MEAS_CH_OPEN | LTR216_BG_MEAS_UNEG_OPEN | LTR216_BG_MEAS_UREF_OPEN,
    /** Группа из всех измерений для проверки КЗ различных цепей */
    LTR216_BG_MEASGROUP_SHORT= LTR216_BG_MEAS_CH_SHORT | LTR216_BG_MEAS_UREF_SHORT,

    /** Группа измерений, которые выполняются относительно земли в фоновом цикле опроса */
    LTR216_BG_MEASGROUP_SECTBL = LTR216_BG_MEAS_UREF | LTR216_BG_MEAS_UREF_OFFS |
        LTR216_BG_MEAS_VADJ | LTR216_BG_MEAS_UNEG | LTR216_BG_MEAS_CH_UX | LTR216_BG_MEAS_CH_CM |
        LTR216_BG_MEASGROUP_OPEN | LTR216_BG_MEASGROUP_SHORT,
    /** Группа измерений для полной проверки целостности линий измерения \f$ U_{ref} \f$ */
    LTR216_BG_MEASGROUP_UREF_CHECK = LTR216_BG_MEAS_UREF_SHORT | LTR216_BG_MEAS_UREF_OPEN |
                                     LTR216_BG_MEAS_UNEG_OPEN,
    /** Группа измерений для полной проверки целостности линий измерительных каналов  */
    LTR216_BG_MEASGROUP_CH_CHECK = LTR216_BG_MEAS_CH_SHORT | LTR216_BG_MEAS_CH_OPEN |
                                     LTR216_BG_MEAS_CH_UX | LTR216_BG_MEAS_CH_CM,

    /** Все фоновые измерения */
    LTR216_BG_MEASGROUP_ALL = LTR216_BG_MEAS_OFFS | LTR216_BG_MEASGROUP_SECTBL
} e_LTR216_BG_MEAS;


/** @brief Начальные измерения

   Флаги, определяющие, какие предварительные измерения будут выполнены
   при вызове LTR216_InitMeasParams().
   
   Для указания всех возможных начальных измерений можно использовать
   константу #LTR216_INIT_MEAS_ALL. */
typedef enum {
    LTR216_INIT_MEAS_OFFS = 1 << 0, /**< Измерение смещения нуля каналов измерения разностного напряжения */
    LTR216_INIT_MEAS_UREF = 1 << 1, /**< Измерение падения напряжения на опорном тензодатчике \f$U_{ref}\f$ */

    /** Все возможные предварительные измерения */
    LTR216_INIT_MEAS_ALL = LTR216_INIT_MEAS_OFFS | LTR216_INIT_MEAS_UREF
} e_LTR216_INIT_MEAS;


/** @brief Состояние вспомогательного измерения

    Константы определяют состояние соответствующего вспомогательного измерения.
    Проверка корректности работы и выполненных измерений может выполняться по
    анализу этого состояния для каждого разрешенного измерения.

    Если измерение не разрешено, то его состояние будет всегда соответствовать
    значению #LTR216_MEASSTATUS_NOT_USED. Если же разрешено, то изначально
    оно будет в состоянии #LTR216_MEASSTATUS_NOT_INIT. После получения
    необходимых данных для расчета значения измерения, его состояние изменится
    на #LTR216_MEASSTATUS_OK, что соответствует состоянию при нормальной работе
    модуля, или в одно из состояний, соответствующих обнаруженной проблеме.
  */
typedef enum {
    LTR216_MEASSTATUS_NOT_USED        = 0, /**< Измерение не используется в текущей конфигурации */
    LTR216_MEASSTATUS_OK              = 1, /**< Получено действительное значение измерения */
    LTR216_MEASSTATUS_NOT_INIT        = 2, /**< Измерение разрешено, но еще не были получены данные
                                                 для расчета его значения */
    LTR216_MEASSTATUS_ADC_OVERRANGE   = 3, /**< При выполнении данного измерения произошло
                                                 переполнение разрядной сетки АЦП.
                                                 Значение измерения не действительно */
    LTR216_MEASSTATUS_SHORT           = 4, /**< Обнаружено КЗ цепи, соответствующей
                                                 данному измерению. Значение измерения
                                                 не действительно */
    LTR216_MEASSTATUS_OPEN            = 5, /**< Обнаружен обрыв цепи, соответствующей
                                                 данному измерению. Значение измерения
                                                 не действительно */
    LTR216_MEASSTATUS_BAD_VALUE_RANGE = 6, /**< Фоновое измерение получено и его значение
                                                 действительно, однако оно выходит
                                                 за допустимый в нормальном режиме
                                                 работы модуля диапазон */
    LTR216_MEASSTATUS_CANT_CALC       = 7  /**< Фоновое измерение не может быть
                                                вычислено. Как правило это связано с
                                                тем, что не действительно другое измерение,
                                                которое необходимо для вычисления текущего */
} e_LTR216_MEAS_STATUS;


/** @brief Флаги обработки данных
   
     Набор флагов, которые влияют на работу функции LTR216_ProcessData(). 
 */
typedef enum {
    /** Флаг указывает, что в качестве результата будет возвращено
     *  значение разбаланса тензометрического полумоста, приведенного к
     *  номинальному опорному напряжению (2,5 В).
     *
     *  Это основной режим измерения модуля и он используются по умолчанию,
     *  когда не передан ни один флаг, определяющий тип выходного измерения. */
    LTR216_PROC_FLAG_MEAS_UNBALANCE     = 0 << 16,
    /** Флаг указывает, что в качестве результата будет возвращено разностное
     *  напряжение между падением на измеряемом и опорном тензодатчиках   */
    LTR216_PROC_FLAG_MEAS_UDIFF         = 1 << 16,
    /** Флаг указывает, что в качестве результата будет возвращено значение
     *  кода АЦП при измерении разностного напряжения между падением на измеряемом
     *  и опорном тензодатчиках (без перевода в Вольты) */
    LTR216_PROC_FLAG_MEAS_UDIFF_CODE    = 2 << 16,


    /** Флаг указывает, что не нужно применять заводские калибровочные коэффициенты.
     *  В штатном режиме работы они применяются всегда, независимо от того, используется
     *  ли пользовательская тарировка или нет */
    LTR216_PROC_FLAG_NO_FABRIC_CBR       = 1 << 0,

    /** По умолчанию LTR216_ProcessData() предполагает, что ей на обработку
        передаются все принятые данные и проверяет непрерывность счетчика не только
        внутри переданного блока данных, но и между вызовами.
        Если обрабатываются не все данные или одни и те же данные обрабатываются
        повторно, то нужно указать данный флаг, чтобы счетчик проверялся только
        внутри обрабатываемого блока */
    LTR216_PROC_FLAG_NONCONT_DATA = 0x0100
} e_LTR216_PROC_FLAGS;


/** @brief Флаги состояния канала при обработке данных

    Дополнительные флаги, определяющие состояние канала за обрабатываемый
    функцией LTR216_ProcessData() блок данных, помимо состояний, определяемых
    фоновыми измерениями. */
typedef enum {
    /**< Признак, что произошло переполнение разрядной сетки АЦП при измерении
         разностного напряжения данного канала */
    LTR216_CH_STATUS_FLAG_OVERRANGE         = 1 << 0
} e_LTR216_CH_STATUS_FLAGS;

/** @brief Флаги обнаруженных ошибок канала при обработке данных

    Данные флаги устанавливаются при обнаружении ошибок в статусе канала или
    фоновых измерений данного канала.

    В отличие от статусов, определяющих текущее состояние канала,
    данные флаги указывают, что отклонение было хотя бы раз за время обработки
    и сбрасываются только вручную пользователем. */
typedef enum {
    /** Признак, что возникало переполнение разрядной сетки АЦП при измерении
     * разностного напряжения для данного канала */
    LTR216_CH_DETECT_ERR_OVERRANGE         = 1 << 0,
    /** Признак, что возникал выход синфазного напряжения за пределы диапазона */
    LTR216_CH_DETECT_ERR_UCM_OUT_OF_RANGE  = 1 << 1,
    /** Признак, что возникал обрыв линии измерения разностного напряжения */
    LTR216_CH_DETECT_ERR_UX_OPEN           = 1 << 2,
    /** Признак, что возникало КЗ линии измерения разностного напряжения */
    LTR216_CH_DETECT_ERR_UX_SHORT          = 1 << 3
} e_LTR216_CH_DETECT_ERRS;

/** @brief Флаги состояния модуля при обработке данных

    Дополнительные флаги, определяющие состояние модуля за обрабатываемый  
    функцией LTR216_ProcessData() блок   данных, помимо состояний, 
    определяемых  фоновыми измерениями. */
typedef enum {
    /** Признак наличия превышения допустимого предела мощности нагрузки */
    LTR216_STATUS_FLAG_LOAD_POW_EXCEEDED = 1 << 0,
} e_LTR216_STATUS_FLAGS;

/** @brief Флаги обнаруженных ошибок модуля при обработке данных

    Данные флаги устанавливаются при обнаружении ошибок в статусе модуля или
    фоновых измерений не связанных с конкретным каналом.

    В отличие от статусов, определяющих текущее состояние модуля,
    данные флаги указывают, что отклонение было хотя бы раз за время обработки
    и сбрасываются только вручную пользователем. */
typedef enum {
    /** Признак, что было обнаружено превышение допустимого предела мощности нагрузки. */
    LTR216_DETECT_ERR_LOAD_POW_EXCEEDED = 1 << 0,
    /** Признак, что был обнаружен выход за шкалу АЦП при измерении напряжения на опорном 
     *    датчике. */
    LTR216_DETECT_ERR_UREF_OVERRANGE    = 1 << 1,
    /** Признак, что был обнаружен обрыв канала измерения напряжения на опорном датчике. */
    LTR216_DETECT_ERR_UREF_OPEN         = 1 << 2,
    /** Признак, что было обнаружено КЗ канала измерения напряжения на опорном датчике. */
    LTR216_DETECT_ERR_UREF_SHORT        = 1 << 3,
    /** Признак, что был обнаружен выход за допустимый диапазон измеренного значения \f$V_{adj}\f$. */
    LTR216_DETECT_ERR_VADJ_ERROR        = 1 << 4,
    /** Признак, что был обнаружен обрыв линии \f$U_-\f$. */
    LTR216_DETECT_ERR_UNEG_OPEN         = 1 << 5
} e_LTR216_DETECT_ERRS;

/** @brief Действительность информации в энергонезависимой памяти модуля

    Набор флагов, указывающий, какая информация была обнаружена в энергонезависимой
    flash-памяти и была прочитана из нее.
*/
typedef enum {
    /** Признак действительности серийного номера */
    LTR216_FLASH_INFO_FLAG_SERIAL_VALID    = 1 << 0,
    /** Признак действительности калибровки источников тока питания датчиков. */
    LTR216_FLASH_INFO_FLAG_ISRC_CBR_VALID  = 1 << 1,
    /** Признак действительности калибровки ЦАП (бит на диапазон). */
    LTR216_FLASH_INFO_FLAG_DAC_CBR_VALID   = 0x7 << 8,
    /** Признак действительности калибровки АЦП
     * (бит на диапазон). */
    LTR216_FLASH_INFO_FLAG_ADC_CBR_VALID   = 0x7 << 12,
} e_LTR216_FLASH_INFO_FLAGS;



/** @} */

/***************************************************************************//**
  @addtogroup type_list Типы данных.
  @{
  *****************************************************************************/

#pragma pack(4)

/** Калибровочные коэффиценты для одного канала. */
typedef struct {
    double Offset; /**< Смещение нуля. */
    double Scale;  /**< Множитель шкалы. */
} TLTR216_CBR_VALUE;


/** @brief Калибровочные коэффициенты источников тока питания датчиков

    Структура хранит заводские калибровочные коэффициенты для источников тока
    питания датчиков.
*/
typedef struct {
    /** Калибровочные коэффициенты для источника тока питания опорного датчика */
    TLTR216_CBR_VALUE Ref;
    /** Калибровочные коэффициенты для остальных источников тока.
     *   Не калибруются, всегда смещение равно 0, а шкала 1.
     *   Резерв для возможного дальнейшего использования. */
    TLTR216_CBR_VALUE Ch[LTR216_CHANNELS_CNT];
} TLTR216_ISRC_CBR;

/** @brief Заводские калибровочные коэффициенты

    Структура хранит все заводские калибровочные коэффициенты модуля, которые
    хранятся в энергонезависимой памяти.
*/
typedef struct {
    /** Калибровочные коэффициенты для каналов АЦП в разных диапазонах измерения */
    TLTR216_CBR_VALUE Adc[LTR216_CBR_RANGES_CNT];
    /** Калибровочные коэффициенты для каналов ЦАП в разных диапазонах измерения АЦП */
    TLTR216_CBR_VALUE Dac[LTR216_CBR_RANGES_CNT];

    /** Калибровочные коэффициенты для источников тока питания датчиков */
    TLTR216_ISRC_CBR ISrc;
    /** Резерв */
    double Reserved[48];
} TLTR216_FABRIC_CBR;


/** @brief Настройки тарировки канала.

  Структура содержит в себе часть настроек модуля, влияющих на тарировку, при которых
  выполнена тарировка канала. */
typedef struct {
    DWORD ISrcCode; /**< Код величины источника тока питания датчиков. */
    BYTE Range; /**< Диапазон измерения АЦП. Одно из значений #e_LTR216_RANGE. */
    DWORD Reserved[6]; /**< Резерв. */
} TLTR216_TARE_CH_CFG;

/** @brief Информация о тарировке одного канала.

    Структура содержит информацию о тарировке канала: сами коэффициенты, признаки
    их действительности, а также информацию о режиме, в котором тарировка была выполнена. */
typedef struct {
    BOOLEAN DacValid; /**< Признак, присутствует ли действительное значение ЦАП. */
    BOOLEAN OffsetValid; /**< Признак, присутствует ли действительное значение коэффициента смещения нуля. */
    BOOLEAN ScaleValid; /**< Признак, присутствует ли действительное значение коэффициента шкалы. */
    DWORD   Reserved1; /**< Резерв. */
    double DacValue;   /**< Напряжение для компенсации смещения нуля, выставляемое на ЦАП. */
    double Offset; /**< Коэффициент смещения нуля. */
    double Scale; /**< Коэффициент шкалы. */
    DWORD Reserved2[8]; /**< Резерв. */
    TLTR216_TARE_CH_CFG Cfg; /**< Настройки, при которых выполнена тарировка. */
} TLTR216_TARE_CH_COEFS;

/** @brief  Информация о модуле.

    Структура, содержащая информацию о версиях прошивок микросхем модуля и
    информацию, считанную из Flash-памяти модуля (серийный номер, калибровочные
    коэффициенты).

    Все поля заполняются при вызове LTR216_Open().
    Значение тарировочных коэффициентов изменяется также при вызове функций
    тарировки.
 */
typedef struct {
    CHAR        Name[LTR216_NAME_SIZE]; /**< Название модуля. Оканчивающуюся нулем ASCII-строка ("LTR216").  */
    CHAR        Serial[LTR216_SERIAL_SIZE]; /**< Серийный номер модуля. Оканчивающуюся нулем ASCII-строка.   */
    WORD        VerFPGA; /**< Версия прошивки ПЛИС. */
    DWORD       Reserved[4]; /**< Резерв. */
    DWORD       FlashInfoFlags; /**< Набор флагов из #e_LTR216_FLASH_INFO_FLAGS,
                                     объединенных по ИЛИ, указывающих, какая информация
                                     содержалась и была прочитана  из flash-памяти модуля */
    TLTR216_FABRIC_CBR FabricCbr; /**< Заводские калибровочные коэффициенты,
                                       считанные из flash-памяти модуля. */
    TLTR216_TARE_CH_COEFS Tare[LTR216_CHANNELS_CNT]; /**< Тарировочные коэффициенты,
                                                          считанные из flash-памяти, или
                                                          вычисленные при выполнении
                                                          тарировки. */
} TLTR216_MODULE_INFO;

/** @brief Параметры текущего состояния модуля.

    Структура, содержащая параметры модуля, которые пользователь должен использовать
    только для чтения, так как они изменяются только внутри функций ltr216api. */
typedef struct {
    BOOLEAN Configured;  /**< Признак, что была выполнена конфигурация устройства
                              через LTR216_SetADC() с момента установления соединения. */
    BOOLEAN Run;         /**< Признак, запущен ли сбор данных. */
    DWORD   FrameWordsCount;   /**< Количество слов в одном кадре.
                                   В LTR216_ProcessData() нужно передавать
                                   количество слов, кратное этому значению.
                                   Обновляется после вызова LTR216_SetADC(). */
    double  AdcFreq;     /**< Установленная частота АЦП. Обновляется после
                            вызова LTR216_SetADC(). */
    double  FrameFreq;   /**< Установленная частота следования кадров (соответствует
                             частоте на логический канал). Обновляется после
                            вызова LTR216_SetADC(). */
    double ISrcValue;    /**< Установленное значение источника тока питания датчиков в мА.
                              Обновляется после вызова LTR216_SetADC().*/
    DWORD Reserved[16];   /**< Резерв */
} TLTR216_STATE;

/** @brief Настройки логического канала

   Структура определяет настройки одного цикла опроса АЦП при пользовательских 
   измерениях разностного напряжения (в цепи последовательных
   измерений, определяемых логической таблицей). */
typedef struct {
    BYTE PhyChannel; /**< Номер физического канала (входа на разъеме), для которого
                          происходит измерение разностного напряжения  */
    BYTE Range;      /**< Используемый диапазон измерения напряжения.
                          Значение из #e_LTR216_RANGE */
    WORD Reserved;   /**< Резерв */
} TLTR216_LCHANNEL;



/** @brief Настройки модуля.

    Структура содержит все настройки модуля, которые должен заполнить
    пользователь перед вызовом LTR216_SetADC().
    Для заполнения части настроек существуют дополнительные вспомогательные
    функции подбора нужного значения.
*/
typedef struct {
    BYTE AdcSwMode; /**< Режим коммутации АЦП (одноканальный или многоканальный).
                         Значение из #e_LTR216_ADC_SWMODE. */
    DWORD SyncFreqDiv; /**< Делитель частоты синхронизации АЦП. В многоканальном
                            режиме определяет частоту коммутации АЦП, которая
                            получается делением частоты #LTR216_ADC_CLOCK на (значение
                            делителя + 1). Может заполняться функцией LTR216_FillSyncFreqDiv().
                            Может принимать значение до #LTR216_SYNC_FDIV_MAX.
                            
                            Не имеет значения в одноканальном режиме. */
    DWORD FilterType;  /**< Тип фильтра, который будет использоваться в АЦП.
                            Значение из #e_LTR216_FILTER_TYPE. Может заполняться с помощью
                            LTR216_FillFilterParams().*/
    DWORD AdcOdrCode;  /**< Код, определяющий время выполнения преобразования АЦП.
                            Зависит от выбранного типа фильтра
                            (поле [FilterType](@ref TLTR216_CONFIG::FilterType)).
                            В одноканальном режиме определяет частоту АЦП, в многоканальном
                            определяет, какое время цикла коммутации занимает преобразование,
                            а какое --- коммутация. Может заполняться с помощью
                            LTR216_FillFilterParams().*/

    double AdcMinSwTimeUs; /**< Необходимое время в мкс, которое требуется для коммутации
                             каналов в многоканальном режиме. Зависит от параметров
                             кабеля подключения и датчиков. Частота АЦП
                             и параметры фильтра должны быть выбраны таким образом,
                             чтобы оставшееся время на коммутацию было не менее указанного
                             параметра, что проверяется при вызове LTR216_SetADC(). 
                             
                             Не имеет значения в одноканальном режиме. */
    double AdcReqFrameFreq; /**< Данное поле определяет желаемое значение
                              частоты следования кадров (частоты на логический канал) в Гц.
                              Если данное значение больше нуля, то при расчете настроек
                              при вызове LTR216_SetADC() подбирается количество дополнительных
                             холостых циклов  таким образом, чтобы результирующая частота была 
                             максимально близкой  к данному значению (но не был превышен 
                             максимальный размер цикла опроса).
                              Если значение меньше или равно 0 (по умолчанию), то используется 
                              максимальная частота при заданных настройках. 
                              
                              Не имеет значения в одноканальном режиме. */
    DWORD ISrcCode;    /**< Код, задающий значение тока питания датчиков. Для заполнения
                            этого поля можно использовать функцию LTR216_FillISrcCode(). */
    DWORD BgMeas;      /**< Набор флагов из #e_LTR216_BG_MEAS, определяющий, какие
                            дополнительные измерения будут выполняться на фоне
                            основных измерений во время сбора данных. 
                            
                            Не имеет значения в одноканальном режиме. */
    BOOLEAN Ch16ForUref; /**< Признак использования 16-го канала для измерения падения
                              напряжения на опорном тензодатчике по 4-х проводной
                              схеме. Если истина, то
                              используется основная схема измерения с подключением
                              опорного тензадатчика по 4-х проводной схеме с использованием
                              16 канала. При этом для основных измерений доступны только
                              первые 15 канало. Если ложь --- то опорный тензорезистор
                              подключается по 2-х проводной схеме, а 16-ый канал может
                              быть использован для подключения еще одного тензодатчика. */
    BOOLEAN TareEnabled;  /**< Признак использования тарировочных коэффициентов.
                              Если истина, то для каналов, для которых действительны тарировочные 
                             коэффциенты, их значения будут использованы для настройки
                             ЦАП, а также для  расчета результирующих значений в
                              LTR216_ProcessData().*/
    DWORD Flags;          /**< Дополнительные флаги, определяющие особые режимы
                               конфигурации. Задается
                               объединением по или констант из #e_LTR216_CONFIG_FLAGS.
                               В текущей версии в штатных режимах работы не используется.*/

    DWORD LChCnt;         /**< Количество используемых логических каналов в логической
                               таблице @fieldref{TLTR216_CONFIG,LChTbl}. */
    TLTR216_LCHANNEL LChTbl[LTR216_MAX_LCHANNEL]; /**< Логическая таблица. Определяет,
                                какие каналы модуля должны быть опрошены в каком
                                порядке и с какими настройками, */
    double  RrefWireResistance; /**< Сопротивление проводов подключения опорного
                                датчика в Омах от модуля до датчика (в одну сторону).

                                Может быть задано в 16-канальном режиме
                                (запрещен [Ch16ForUref](@ref TLTR216_CONFIG::Ch16ForUref))
                                для компенсации падения напряжения на проводах
                                при измерении напряжения на опорном датчике по
                                2-х проводной схеме.
                                
                                При 4-х проводной схеме подключения опорного датчика 
                                не используется. */
    double  ShortThresholdR; /**< Порог сопротивления (Ом) для проверки КЗ
                                   линии. КЗ считается при измеренном сопротивлении
                                   линии ниже данного порога. */
    double  CableLength; /**< Максимальная длина кабеля (м), использующегося
                              для подключения датчиков.
                              Используется только при проверке обрыва
                              линий. */
    double  CableCapacityPerUnit; /**< Максимальная погонная емкость кабеля (пФ/м)
                                       подключения датчиков.
                                       Используется только при проверке обрыва линий. */

    double Reserved[32]; /**< Резерв */
} TLTR216_CONFIG;



/** @brief  Низкоуровневые настройки модуля

   Данная структура описывает реальный цикл опроса АЦП, включая
   как пользовательские измерения, соответствующие логической таблицы
   пользовательской конфигурации, так и служебные фоновые измерения и холостые
   такты для их согласования.
   
   Включает в себя таблицы, описывающие два цикла опроса. Основной выполняется
   каждый раз полностью за каждый кадр, а фоновый цикл выполняется за несколько
   кадров.
   
   Как правило пользователю  не требуется использовать эту структуру --- она автоматически заполняется
   при вызове LTR216_SetADC().

   Поля данной структуры можно использовать для информации о временных параметрах
   получившегося реального цикла опроса АЦП. */
typedef struct  {
    DWORD UserChCnt; /**< Количество логических каналов, соответствующих пользовательским
                          измерениям из логической таблицы [конфигурации модуля](@ref TLTR216_CONFIG).  */
    DWORD ProcChCnt; /**< Общее количество обрабатываемых каналов из основного цикла опроса (без холостых тактов). */
    DWORD MainLChCnt; /**< Количество действительных элементов в массиве [MainLChTbl](@ref TLTR216_RAWCONFIG::MainLChTbl).
                                                          Включает общее число измерений в основном цикле (без учета длительности шага фонового цикла). */
    DWORD BgStepSize; /**< Количество измерений, которое занимает  выполнение одного шага фоновой таблицы. */
    DWORD BgStepsCnt; /**< Количество шагов, за которое будет пройдена вся фоновая таблица. */
    DWORD BgFrameStepCnt; /**< Количество шагов фоновой таблицы в одном кадре.*/
    DWORD BgLChCnt; /**< Количество элементов в [BgLChTbl](@ref TLTR216_RAWCONFIG::BgLChTbl). */
    DWORD BgBoutLChCnt; /**< Количество каналов с включенным током проверки обрыва в фоновой таблице. */
    DWORD FrameSize;   /**< Общее количество измерений в одном кадре . */
    DWORD Reserved[7];  /**< Резерв. */
    DWORD MainLChTbl[LTR216_MAX_LCHANNEL];  /**< Массив, определяющий последовательность
                                                 измерений в основном цикле опроса */
    DWORD BgLChTbl[LTR216_MAX_LCHANNEL]; /**< Массив, определяющий последовательность
                                                 измерений в фоновом цикле опроса */
} TLTR216_RAWCONFIG;



/** @brief  Описатель модуля модуля.
 *
 *  Структура, которая хранит всю необходимую информацию о модуле и связи с ним:
 *  текущие настройки модуля, информацию о  его состоянии, структуру канала связи
 *  и т.д.
 *
 *  При работе с несколькими модулями одновременно для каждого модуля должен быть
 *  создан свой описатель.
 *
 *  Передается в большинство функций библиотеки.
 *  Некоторые поля структуры доступны для изменения пользователем
 *  для настройки параметров модуля.
 *
 *  Перед использованием требует инициализации с помощью функции LTR216_Init().
 */
typedef struct {
    INT         Size; /**< Размер структуры. Заполняется в LTR216_Init(). */
    /** Структура, содержащая состояние клиентского соединения со службой ltrd.
        Не используется напрямую пользователем. */
    TLTR        Channel;
    /** Указатель на непрозрачную структуру с внутренними параметрами,
      используемыми исключительно библиотекой и недоступными для пользователя. */
    PVOID       Internal;
    /** Настройки модуля. Заполняются пользователем перед вызовом LTR216_SetADC(). */
    TLTR216_CONFIG Cfg;
    /** Более детальные низкоуровневые настройки коммутации модуля. Рассчитываются
     * из основных настроек и заполняются автоматически при вызове LTR216_SetADC(). */
    TLTR216_RAWCONFIG RawCfg;
    /** Состояние модуля и рассчитанные параметры. Поля изменяются функциями
        библиотеки. Пользовательской программой могут использоваться
        только для чтения. */
    TLTR216_STATE  State;
    /** Информация о модуле. */
    TLTR216_MODULE_INFO ModuleInfo;
} TLTR216;


/** @brief Параметры фильтра

   Структура содержит расчетные параметры фильтра, полученные при заданных
   настройках.

   Поля [FilterType](@ref TLTR216_FILTER_OUT_PARAMS::FilterType) и
   [FilterType](@ref TLTR216_FILTER_OUT_PARAMS::AdcOdrCode) определяют настройки
   фильтра, расчетные параметры для которых содержит данная структура. Эти
   поля соответствуют одноименным полям [структуры конфигурации модуля](@ref TLTR216_CONFIG).

   Остальные поля содержат параметры фильтра, соответствующие указанным настройкам. */
typedef struct {
    DWORD FilterType; /**< Тип фильтра, которому соответствуют рассчитанные
                            пераметры.  Значение из #e_LTR216_FILTER_TYPE .*/
    DWORD AdcOdrCode;  /**< Код, определяющий время выполнения преобразования АЦП,
                             для которого получены рассчитанные параметры. */
    double Odr;        /**< Частота выдачи данных АЦП в Гц при его непрерывной работе.
                             Эта частота является обратной величиной времени
                             преобразования АЦП. */
    double NotchFreq;  /**< Первая частота режекции фильтра. На ней достигается
                            наибольшее подавление шумов. */
    double NotchDB;    /**< Уровень подавления на частоте режекции в децибелах,
                            если известен. Возвращается только для фильтра типа
                            #LTR216_FILTER_ENH_50HZ. */
    double Reserved[4]; /**< Резерв. */
} TLTR216_FILTER_OUT_PARAMS;

/** @brief Состояние фонового измерения.

    Данная структура содержит в себе информацию о состоянии и значении одного
    фонового измерения. */
typedef struct {
    BYTE    Status;  /**< Состояние измерения. Значение из #e_LTR216_MEAS_STATUS */
    BOOLEAN ValueValid; /**< Признак, действительно ли значение
                             в поле [Value](@ref TLTR216_MEASSTATE::Value) */
    BYTE    Reserved[6]; /**< Резерв */
    double  Value; /**< Значение фонового измерения. Действительно только при
                        установленном флаге [ValueValid](@ref TLTR216_MEASSTATE::ValueValid). */
} TLTR216_MEASSTATE;


/** @brief Состояние модуля во время сбора данных

   В данной структуре сохраняется информация об обнаруженных проблемах и результаты
   анализа  фоновых измерений, относящиеся ко всему модулю.
   Данная структура должна быть изначально проинициализирована с помощью LTR216_DataStatusInit(),
   после чего она может быть использована в функциях LTR216_ProcessData() и
   LTR216_CheckMeasStatus() для возвращения результатов анализа фоновых измерений. */
typedef struct {
    DWORD DetectedErrors; /**< При обнаружении проблем, не относящихся к отдельным
                               каналам, в данном поле устанавливаются
                              соответствующие флаги из #e_LTR216_DETECT_ERRS.
                              Данные флаги могут быть сброшены только вручную
                              пользователем, т.е. они описывают возникла ли
                              проблема с момента инициализации структуры или
                              ручного сброса данного поля. */
    DWORD StatusFlags; /**<  Объединение по "ИЛИ" флагов #e_LTR216_STATUS_FLAGS,
                            описывающих состояние модуля на текущий момент, которое
                            не описывается состоянием фоновых измерений. */
    TLTR216_MEASSTATE Uref; /**< Текущее состояние измерения падения
                              напряжения на опорном датчике и проводах до него. */
    TLTR216_MEASSTATE UrefR; /**< Текущее состояние измерения падения напряжения
                               только на опорном датчике. Данное измерение в 4-х проводной
                               схеме подключения опорного датчика измеряется явно,
                               а признаки обрыва и КЗ характеризуют состояние линий 16-го канала,
                               которые используются в этом режиме именно для измерения
                               данного значения. При 2-х проводной схеме подключения
                               опорного датчика данное измерение действительно только
                               если задано значение
                               [RrefWireResistance](@ref TLTR216_CONFIG::RrefWireResistance)
                               больше нуля и вычисляется на основе
                               [Uref](@ref TLTR216_DATA_STATUS::Uref). */
    TLTR216_MEASSTATE Vadj; /**< Текущее состояние измерения напряжения \f$ V_adj  \f$
                                  с учетом проверки порога. */
    TLTR216_MEASSTATE Uneg; /**< Измерение падения напряжения на линии \f$ U_- \f$ */
    DWORD Reserver[16]; /**< Резерв */
} TLTR216_DATA_STATUS;


/** @brief Состояние канала во время сбора данных

   В данной структуре сохраняется информация об обнаруженных проблемах и результаты
   анализа  фоновых измерений, относящиеся к отдельному измерительному каналу.
   
   Используется массив данных структур (по структуре на канал), количество элементов 
   которого должно  соответствовать количеству каналов в логической таблице.
   Данный массив  должен быть изначально проинициализирован с помощью 
   LTR216_DataChannelsStatusInit(),
   после чего может быть использована в функциях LTR216_ProcessData() и
   LTR216_CheckMeasStatus() для возвращения результатов анализа фоновых измерений. */
typedef struct {
    DWORD DetectedErrors; /**< При обнаружении проблем, относящихся к отдельным
                               каналам, в данном поле устанавливаются
                              соответствующие флаги из #e_LTR216_CH_DETECT_ERRS.
                              Данные флаги могут быть сброшены только вручную
                              пользователем, т.е. они описывают возникла ли
                              проблема с момента инициализации структуры или
                              ручного сброса данного поля. */
    DWORD StatusFlags; /**<  Объединение по "ИЛИ" флагов #e_LTR216_CH_STATUS_FLAGS,
                            описывающих состояние канала на текущий момент, которое
                            не описывается состоянием фоновых измерений. */
    TLTR216_MEASSTATE Ures; /**< Состояние результирующего измерения (в штатном
                                 режеме --- разбаланса).
                                 Действительность зависит от действительности
                                 измерений, на основе которых производится
                                 вычисление, а значение, при его действительности,
                                 соответствует последнему результату по данному
                                 каналу */
    TLTR216_MEASSTATE Ux; /**< Состояние измерения абсолютного напряжения на
                             тензодатчике, подключенному к каналу, и сопротивлении
                             проводов до датчика. */
    TLTR216_MEASSTATE Ucm; /**< Рассчитанное значение синфазного напряжения для данного канала */
    double UcmMin; /**< Минимально допустимое значение синфазного напряжения для данного канала.
                        Действительно только при действительном значении [Ucm](@ref TLTR216_DATA_CH_STATUS::Ucm). */
    double UcmMax; /**< Максимально допустимое значение синфазного напряжения для данного канала.
                        Действительно только при действительном значении [Ucm](@ref TLTR216_DATA_CH_STATUS::Ucm). */
    DWORD Reserver[16]; /**< Резерв. */
} TLTR216_DATA_CH_STATUS;

#pragma pack()

/** @} */

/***************************************************************************//**
    @addtogroup func_init Функции инициализации и работы с соединением с модулем.
    @{
*******************************************************************************/

/***************************************************************************//**
  @brief Инициализация описателя модуля.

  Функция инициализирует поля структуры описателя модуля значениями по умолчанию.
  Эта функция должна вызываться для каждой структуры #TLTR216 перед вызовом
  остальных функций.
   @param[in] hnd       Описатель модуля
   @return              [Код ошибки](@ref e_LTR216_ERROR_CODES).
 ******************************************************************************/
LTR216API_DllExport(INT) LTR216_Init(TLTR216 *hnd);

/***************************************************************************//**
   @brief Открытие соединения с модулем.

   Функция устанавливает соединение с модулем в соответствии с переданными
   параметрами, проверяет наличие модуля и считывает информацию о нем.
   Должна быть вызвана перед работой с модулем. После завершения работы
   необходимо закрыть соединение с помощью LTR216_Close().

   @param[in] hnd       Описатель модуля
   @param[in] ltrd_addr  IP-адрес машины, на которой запущена служба ltrd, в 32-битном
                        формате (описан в разделе "Формат задания IP-адресов"
                        @docref_ltrapi{руководства для библиотеки ltrapi}).
                        Если служба ltrd запущена на той же машине, что и программа,
                        вызывающая данную функцию, то в качестве адреса
                        можно передать LTRD_ADDR_DEFAULT.
   @param[in] ltrd_port TCP-порт для подключения к службе ltrd. По умолчанию
                        используется LTRD_PORT_DEFAULT.
   @param[in] csn       Серийный номер крейта, в котором находится интересующий
                        модуль. Представляет собой оканчивающуюся нулем ASCII-строку.
                        Для соединения с первым найденным крейтом можно передать
                        пустую строку или нулевой указатель.
   @param[in] slot      Номер слота крейта, в котором установлен интересующий модуль.
                        Значение от LTR_CC_CHNUM_MODULE1 до LTR_CC_CHNUM_MODULE16.
   @return              [Код ошибки](@ref e_LTR216_ERROR_CODES).
*******************************************************************************/
LTR216API_DllExport(INT) LTR216_Open(TLTR216 *hnd, DWORD ltrd_addr, WORD ltrd_port, const CHAR *csn, INT slot);
/***************************************************************************//**
   @brief Закрытие соединения с модулем.

   Функция закрывает ранее открытое с помощью LTR216_Open() соединение. Должна
   вызываться после завершения работы с модулем. При любом возвращенном значении
   после вызова этой функции соответствующий описатель уже нельзя использовать
   для работы с модулем без открытия нового соединения.

   @param[in] hnd       Описатель модуля
   @return              [Код ошибки](@ref e_LTR216_ERROR_CODES).
 ******************************************************************************/
LTR216API_DllExport(INT) LTR216_Close(TLTR216 *hnd);
/***************************************************************************//**
   @brief Проверка, открыто ли соединение с модулем.

   Функция проверяет, открыто ли в данный момент соединение с модулем. Если
   соединение открыто, функция возвращает LTR_OK, если закрыто --- код ошибки
   LTR_ERROR_CHANNEL_CLOSED.
   @param[in] hnd       Описатель модуля
   @return              [Код ошибки](@ref e_LTR216_ERROR_CODES) (LTR_OK, если соединение установлено)
 ******************************************************************************/
LTR216API_DllExport(INT) LTR216_IsOpened(TLTR216 *hnd);


/** @} */

/***************************************************************************//**
    @addtogroup func_config Функции для изменения настроек модуля
    @{
*******************************************************************************/


/***************************************************************************//**
   @brief Запись настроек в модуль.

   Функция передает настройки, соответствующие значениям полей 
   [структуры конфигурации](@ref TLTR216::Cfg) описателя модуля, в модуль.
   Должна вызываться хотя бы раз до первого запуска сбора, а также после любых изменений
   настроек модуля, чтобы они вступили в силу.

   @param[in] hnd       Описатель модуля
   @return              [Код ошибки](@ref e_LTR216_ERROR_CODES).
 ******************************************************************************/
LTR216API_DllExport(INT) LTR216_SetADC(TLTR216 *hnd);

/***************************************************************************//**
   @brief Заполнение делителя частоты АЦП

   Функция по заданному значению частоты АЦП подбирает делитель частоты
   таким образом, чтобы результирующая частота оказалась наиболее близка к заданной,
   и записывает полученное значение в поле [SyncFreqDiv](@ref TLTR216_CONFIG::SyncFreqDiv)
   в структуре конфигурации описателя модуля.

   Также может возвращать в выходном параметре результирующее значение частоты
   АЦП.
   
   Используется только для многоканального режима сбора.

   @param[in] hnd               Описатель модуля.
   @param[in] adcFreq           Требуемая частота АЦП в Гц.
   @param[out] resultAdcFreq    В данной переменной возвращается полученная в результате
                                подбора частота АЦП в Гц. Может быть передан
                                нулевой указатель, если значение не требуется.
   @return                      [Код ошибки](@ref e_LTR216_ERROR_CODES).
 ******************************************************************************/
LTR216API_DllExport(INT) LTR216_FillSyncFreqDiv(TLTR216 *hnd, double adcFreq, double *resultAdcFreq);

/***************************************************************************//**
   @brief Вычисление делителя частоты АЦП

   Функция аналогична LTR216_FillSyncFreqDiv(), но возвращает рассчитанный делитель явно
   в переменной. Позволяет производить вычисление без действительного описателя модуля.
   @param[in] adcFreq           Требуемая частота  АЦП.
   @param[out] div              В данной переменной возвращается рассчитанное значение
                                делителя частоты АЦП в Гц.
   @param[out] resultAdcFreq    В данной переменной возвращается полученная в результате
                                подбора частота  АЦП в Гц. Может быть передан
                                нулевой указатель, если значение не требуется.
   @return                      [Код ошибки](@ref e_LTR216_ERROR_CODES).
 ******************************************************************************/
LTR216API_DllExport(INT) LTR216_FindSyncFreqDiv(double adcFreq, DWORD *div, double *resultAdcFreq);

/***************************************************************************//**
   @brief Заполнение параметров фильтра АЦП

   Функция по режиму коммутации АЦП, частоте АЦП и требуемому времени на коммутацию 
   (последнее только в многоканальном режима) подбирает оптимальные параметры фильтра 
   и заполняет полученными значениями поля [FilterType](@ref TLTR216_CONFIG::FilterType) и
   [AdcOdrCode](@ref TLTR216_CONFIG::AdcOdrCode) в
   [структуре конфигурации](@ref TLTR216::Cfg) описателя модуля.
   
   Способ подбора зависит от того, настроен одноканальный или многоканальный режим сбора
   (перед вызовом данной функции должно быть заполнено поле
   [AdcSwMode](@ref TLTR216_CONFIG::AdcSwMode) в 
   [структуре конфигурации](@ref TLTR216::Cfg) описателя модуля.
   
   В одноканальном режиме сбора функция подбирает параметры фильтра так, чтобы
     результирующая частота преобразования АЦП (которая соответствует частоте АЦП
     в данном режиме) была наиболее близка к переданному значению adcFreq.
     Подобранную частоту преобразования можно узнать по значению поля
     [Odr](@ref TLTR216_FILTER_OUT_PARAMS::Odr) полученных параметров фильтра.
   
   В многоканальном режиме функция подбирает значения фильтра так, чтобы
     время преобразования АЦП было максимальным, но при этом для заданной частоты
     АЦП, определяемой по полю [SyncFreqDiv](@ref TLTR216_CONFIG::SyncFreqDiv)
     конфигурации, оставшееся время коммутации было не меньше
     [AdcMinSwTimeUs](@ref TLTR216_CONFIG::AdcMinSwTimeUs) мкс. Соответственно,
     оба поля в структуре конфигурации должны быть заполнены перед вызовом.
     Если нужного времени коммутации достичь не удалось, то функция вернет ошибку
     #LTR216_ERR_UNSUF_SW_TIME.

   @param[in] hnd               Описатель модуля.
   @param[in] adcFreq           Требуемая частота АЦП для одноканального режима.
                                Для многоканального это значение не используется,
                                а установленная частота определяется по значению
                                поля [SyncFreqDiv](@ref TLTR216_CONFIG::SyncFreqDiv)
                                конфигурации модуля.
   @param[out] filterParams     В данной структуре возвращаются параметры,
                                описывающие подобранные настройки фильтра.
                                Может быть передан нулевой указатель, если
                                данная информация не требуется.
   @param[out] swTimeRes        В многоканальном режиме в данной переменной
                                возвращается полученное время на коммутацию в мкс.
                                В одноканальном режиме не используется.
                                Может быть передан нулевой указатель, если
                                данная информация не требуется.
   @return                      [Код ошибки](@ref e_LTR216_ERROR_CODES).
 ******************************************************************************/
LTR216API_DllExport(INT) LTR216_FillFilterParams(TLTR216 *hnd, double adcFreq,
                                                 TLTR216_FILTER_OUT_PARAMS *filterParams,
                                                 double *swTimeRes);

/***************************************************************************//**
   @brief Подбор параметров фильтра АЦП

   Функция аналогична LTR216_FillFilterParams(), но только рассчитывает параметры
   без заполнения полей описателя. Позволяет выполнять расчет без создания
   описателя модуля.

   @param[in] adcSwMode         Режим коммутации АЦП. Значение из #e_LTR216_ADC_SWMODE.
   @param[in] adcFreq           Требуемая частота АЦП в Гц.
   @param[in] swTimeUs          Требуемое время на коммутацию в мкс в многоканальном
                                режиме. Не имеет значения для одноканального.
   @param[out] filterParams     В данной структуре возвращаются параметры,
                                описывающие подобранные настройки фильтра.
   @param[out] swTimeRes        В многоканальном режиме в данной переменной
                                возвращается полученное время на коммутацию в мкс.
                                В одноканальном режиме не используется.
                                Может быть передан нулевой указатель, если
                                данная информация не требуется.
   @return                      [Код ошибки](@ref e_LTR216_ERROR_CODES).
 ******************************************************************************/
LTR216API_DllExport(INT) LTR216_FindFilterParams(DWORD adcSwMode, double adcFreq,
                                                 double swTimeUs,
                                                 TLTR216_FILTER_OUT_PARAMS *filterParams,
                                                 double *swTimeRes);


/***************************************************************************//**
   @brief Получение параметров фильтра по настройкам

   Функция возвращает параметры фильтра по его настройкам в зависимости от режима
   коммутации АЦП.

   @param[in] adcSwMode         Режим коммутации АЦП. Значение из #e_LTR216_ADC_SWMODE.
   @param[in] filterType        Тип фильтра. Значение из #e_LTR216_FILTER_TYPE.
   @param[in] adcOdrCode        Код, определяющий время выполнения преобразования АЦП.
                                Значение аналогично полю [AdcOdrCode](@ref TLTR216_CONFIG::AdcOdrCode)
                                структуры конфигурации модуля.
   @param[out] filterParams     В данной структуре возвращаются параметры,
                                описывающие фильтр, соответствующий указанным настройкам.
   @return                      [Код ошибки](@ref e_LTR216_ERROR_CODES).
 ******************************************************************************/
LTR216API_DllExport(INT) LTR216_GetFilterOutParams(DWORD adcSwMode, DWORD filterType, DWORD adcOdrCode,
                                                   TLTR216_FILTER_OUT_PARAMS *filterParams);


/***************************************************************************//**
   @brief Заполнение поля для задания силы тока питания датчиков

   Функция по заданному значению силы тока питания датчиков подбирает и устанавливает
   значение поля [ISrcCode](@ref TLTR216_CONFIG::ISrcCode) конфигурации так,
   чтобы полученный ток питания датчиков был наиболее близок к заданному.

   Функция использует калибровочные коэффициенты источника тока для более точной
   установки требуемой величины.

   @param[in] hnd               Описатель модуля.
   @param[in] isrc              Требуемое значение источника тока в мА
   @param[out] resultValue      В данной переменной возвращается полученное значение
                                 тока питания датчиков в мА, соответствующее подобранному коду.
                                Может быть передан нулевой указатель, если
                                данная информация не требуется.
   @return                      [Код ошибки](@ref e_LTR216_ERROR_CODES).
 ******************************************************************************/
LTR216API_DllExport(INT) LTR216_FillISrcCode(TLTR216 *hnd, double isrc, double *resultValue);

/***************************************************************************//**
   @brief Подбор кода для задания силы тока питания датчиков

   Функция аналогична LTR216_FillISrcCode(), но только подбирает нужное значение
   кода, не заполняя конфигурацию описателя модуля.

   Функция позволяет выполнять расчет без создания описателя модуля.

   @param[in] cbr               Калибровочные коэффициенты источника тока. Может
                                быть передан нулевой указатель, если не требуется
                                применение калибровочных коэффициентов.
   @param[in] isrc              Требуемое значение источника тока в мА
   @param[out] code             Полученное значение кода задания величины
                                тока (которое может быть записано в поле
                                [ISrcCode](@ref TLTR216_CONFIG::ISrcCode)
                                конфигурации модуля).
                                Может быть передан нулевой указатель, если
                                данная информация не требуется.
   @param[out] resultValue      В данной переменной возвращается полученное значение
                                 тока питания датчиков в мА, соответствующее подобранному коду.
                                Может быть передан нулевой указатель, если
                                данная информация не требуется.
   @return                      [Код ошибки](@ref e_LTR216_ERROR_CODES).
 ******************************************************************************/
LTR216API_DllExport(INT) LTR216_FindISrcCode(const TLTR216_ISRC_CBR *cbr, double isrc,
                                             DWORD *code, double *resultValue);


/***************************************************************************//**
   @brief Рассчет величины силы тока по коду

   Функция позволяет по заданному значению поля [ISrcCode](@ref TLTR216_CONFIG::ISrcCode)
   получить результирующее значение силы тока питания датчиков.

   Функция может использовать калибровочные коэффициенты для более точного рассчета
   силы тока.
   @param[in] cbr               Калибровочные коэффициенты источника тока. Может
                                быть передан нулевой указатель, если не требуется
                                применение калибровочных коэффициентов.
   @param[in] code              Значение поля [ISrcCode](@ref TLTR216_CONFIG::ISrcCode),
                                для которого рассчитывается сила тока.
   @param[out] resultValue      В данной переменной возвращается значение силы
                                тока в мА, соответствующее заданному коду.
   @return                      [Код ошибки](@ref e_LTR216_ERROR_CODES).
 ******************************************************************************/

LTR216API_DllExport(INT) LTR216_CalcISrcValue(const TLTR216_ISRC_CBR *cbr,
                                              DWORD code, double *resultValue);


/***************************************************************************//**
   @brief Рассчет низкоуровневых параметров цикла опроса АЦП

   Функция по заданной пользовательской конфигурации рассчитывает низкоуровневые
   настройки, соответствующие результирующему циклу опроса АЦП, включая
   фоновые измерения, холостые такты и т.п.

   Данная функция как правило не требуется, т.к. действие этой функции выполняется
   при записи конфигурации в модуль вызовом LTR216_SetADC(). Данная функция
   позволяет сделать этот расчет без записи настроек в модуль, в частности
   если связь не установлена или описатель не создан.

   @param[in] minfo           Структура с информацией о модуле, включая его
                              калибровочные коэффициенты. Может быть передан
                              нулевой указатель, если не требуется их учет.
   @param[in] cfg             Пользовательская конфигурация модуля
   @param[out] rawCfg         В данной структуре сохраняется полученные низкоуровневые
                              параметры цикла опроса АЦП.
   @return                    [Код ошибки](@ref e_LTR216_ERROR_CODES).
 ******************************************************************************/
LTR216API_DllExport(INT) LTR216_FillRawTables(const TLTR216_MODULE_INFO *minfo,
                                              const TLTR216_CONFIG *cfg,
                                              TLTR216_RAWCONFIG *rawCfg);
/** @} */



/***************************************************************************//**
    @addtogroup func_stream Функции для управления сбором данных
    @{
*******************************************************************************/

/** @brief Измерение начальных параметров перед запуском сбора

   Функция позволяет выполнить начальные измерения, необходимые для расчета
   результатов при дальнейшем сборе, перед запуском самого сбора.

   Какие именно измерения будут выполняться, зависит от переданных флагов в качестве
   поля meas. Для выполнения всех требуемых измерений можно передать значение
   #LTR216_INIT_MEAS_ALL.
   
   Помимо измерений начальных параметров на фоне могут быть использованы дополнительные
   фоновые измерения для одновременной проверки линий, связанных с выполняемым
   измерением. Эти измерения задаются с помощью флагов, аналогично полю 
   [BgMeas](@ref TLTR216_CONFIG::BgMeas) конфигурации модуля.    
   Для измерения смещения нуля не используются, а для измерения \f$ U_{ref} \f$ могут 
   использоваться набор измерений  из группы #LTR216_BG_MEASGROUP_UREF_CHECK.
   Если измерение начального значения не может быть выполнено из-за обнаруженных
   ошибок, то функция вернет ошибку #LTR216_ERR_BAD_INIT_MEAS_STATUS и более 
   подробную информацию можно будет получить по полям структуры 
    #TLTR216_DATA_STATUS, возвращаемой в параметре status.

   При завершении выполнения функция обновляет внутренние поля, явно не доступные
   пользователю, необходимые для расчетов при запущенном сборе данных. Также
   результаты измерения напряжения на опорном датчике можно получить через
   соответствующие поля структуры  #TLTR216_DATA_STATUS, возвращаемой
   в параметре status.

   @param[in] hnd   Описатель модуля
   @param[in] meas  Флаги, определяющие, какие начальные измерения должны быть
                    выполнены. Может быть использован флаг #LTR216_INIT_MEAS_ALL
                    для выполнения всех измерений.
   @param[in] bgMeas Дополнительные фоновые измерения, которые могут использоваться
                    для проверки целостности линий, связанных с выбранными
                    начальными измерениями.
   @param[out] status В данной структуре могут быть возвращены результаты выполненных
                    измерений (если они доступны пользователю), а также их состояние.
                    Если данная информация не требуется, то может быть передан нулевой
                    указатель.
   @return         [Код ошибки](@ref e_LTR216_ERROR_CODES).
 ******************************************************************************/
LTR216API_DllExport(INT) LTR216_InitMeasParams(TLTR216 *hnd, DWORD meas,
                                               DWORD bgMeas, TLTR216_DATA_STATUS *status);


/***************************************************************************//**
    @brief Запуск сбора данных

    При вызове данной функции запускается сбор данных с АЦП и модуль начинает
    передавать в ПК полученные отсчеты, которые необходимо вычитывать
    с помощью LTR216_Recv().
    При завершении измерений для остановки сбора данных необходимо вызвать
    LTR216_Stop().

   @param[in] hnd       Описатель модуля
   @return              [Код ошибки](@ref e_LTR216_ERROR_CODES).
 ******************************************************************************/
LTR216API_DllExport(INT) LTR216_Start(TLTR216 *hnd);

/***************************************************************************//**
    @brief Останов сбора данных.

    При вызове данной функции модуль останавливает сбор и выдачу данных.

    При этом вычитываются и отбрасываются все переданные, но непрочитанные
    данные от модуля.

   @param[in] hnd       Описатель модуля
   @return              [Код ошибки](@ref e_LTR216_ERROR_CODES).
 ******************************************************************************/
LTR216API_DllExport(INT) LTR216_Stop(TLTR216 *hnd);


/***************************************************************************//**
    @brief Инициализация структуры состояния модуля при обработке данных.

   Данная функция заполняет начальными значениями поля структуры #TLTR216_DATA_STATUS.
   Необходимо вызвать данную функцию после создания структуры  до передачи ее в другие
   функции.

   @param[in] status    Структура состояния модуля, которая заполняется начальными значениями
   @return              [Код ошибки](@ref e_LTR216_ERROR_CODES).
 ******************************************************************************/
LTR216API_DllExport(INT) LTR216_DataStatusInit(TLTR216_DATA_STATUS *status);

/***************************************************************************//**
    @brief Инициализация структур состояния каналов модуля при обработке данных.

   Данная функция заполняет начальными значениями поля в массиве структур #TLTR216_DATA_STATUS.
   Может быть проинициализирована и одна структура, если передан указатель на нее и количество 
   элементов указано равным 1.
   Необходимо вызвать данную функцию после создания данных структур до передачи их
   в другие функции.

   @param[in] chStatusList  Массив из chCnt структур состояния каналов, которые необходимо
                            проинициализировать.
   @param[in] chCnt         Количество элементов в массиве chStatusList
   @return                  [Код ошибки](@ref e_LTR216_ERROR_CODES).
 ******************************************************************************/
LTR216API_DllExport(INT) LTR216_DataChannelsStatusInit(TLTR216_DATA_CH_STATUS *chStatusList, DWORD chCnt);

/***************************************************************************//**
   @brief Прием данных от модуля.

   Функция принимает запрашиваемое число слов от модуля. Возвращаемые слова
   находятся в специальном формате, который включает в себя служебную информацию.
   Возвращаемые слова соответствуют как основным пользовательским измерениям, так
   и фоновым измерениям в соответствие с настройками модуля.

   Для обработки принятых слов, получения измеренных значений и  проверки
   состояния модуля и каналов используется функция LTR216_ProcessData().

   Функция возвращает управление либо когда примет запрошенное количество слов,
   либо по истечению таймаута. При этом реально принятое количество слов можно
   узнать по возвращенному значению.

   @param[in]  hnd      Описатель модуля.
   @param[out] data     Массив, в который будут сохранены принятые слова. Должен
                        быть размером на size 32-битных слов.
   @param[out] tmark    Указатель на массив размером на size 32-битных слов,
                        в который будут сохранены значения счетчиков синхрометок,
                        соответствующие принятым данным. Генерация меток
                        настраивается для крейта или специального модуля отдельно.
                        Синхрометки подробнее описаны в разделе "Синхрометки"
                        @docref_ltrapi{руководства для библиотеки ltrapi}.
                        Если синхрометки не используются, то можно передать
                        в качестве параметра нулевой указатель.
   @param[in]  size     Запрашиваемое количество 32-битных слов на прием.
   @param[in]  timeout  Таймаут на выполнение операции в миллисекундах. Если в течение
                        заданного времени не будет принято запрашиваемое количество
                        слов, то функция все равно вернет управление, возвратив
                        в качестве результата реально принятое количество слов.
   @return              Значение меньше нуля соответствует коду ошибки. Значение
                        больше или равное нулю соответствует количеству реально
                        принятых и сохраненных в массив data слов.
*******************************************************************************/
LTR216API_DllExport(INT) LTR216_Recv(TLTR216 *hnd, DWORD *data, DWORD *tmark,
                                     DWORD size, DWORD timeout);


/****************************************************************************//**
   @brief Обработка принятых от модуля слов.

   Функция используется для обработки слов, принятых от модуля с помощью LTR216_Recv().
   Функция проверяет служебные поля принятых слов, извлекает полезную информацию
   с отсчетами и выполняет расчет результирующих измерений с применением при
   необходимости калибровочных и тарировочных коэффициентов.

   Кроме того, данная функция выполняет обработку фоновых измерений и использует
   их результаты либо для учета в расчете результатов, либо для проверки линий
   подключения и условий измерений, в зависимости от назначения фонового измерения.
   Результаты указанный проверок и значения этих фоновых измерений сохраняются
   в структуры, переданные в качестве параметров status и chStatusList.

   В функцию необходимо передавать слова, выравненные на начало кадра и
   в количестве кратном количеству слов в кадре (можно узнать по значению поля
   [FrameWordsCount](@ref TLTR216_STATE::FrameWordsCount)).

   Функция изначально считает, что все данные, принятые LTR216_Recv() полностью
   последовательно обрабатываются LTR216_ProcessData().
   Если это не так, то необходимо передать флаг #LTR216_PROC_FLAG_NONCONT_DATA.

   @param[in] hnd      Описатель модуля.
   @param[in] src      Указатель на массив, содержащий слова, принятые от модуля
                       с помощью LTR216_Recv(), которые нужно обработать.
   @param[out] dest    Указатель на массив, в который будут сохранены результирующие
                        значения измерений. Порядок следования соответствует порядку во входном
                       массиве: сперва первый отсчет первого логического канала,
                        затем первый отсчет второго логического канала и т.д.
   @param[in,out] size  На входе принимает размер массива src для обработки. На
                        выходе, при успешном завершении, возвращает количество
                        сохраненных отсчетов в массиве dest.
   @param[in]  flags   Флаги из #e_LTR216_PROC_FLAGS, управляющие работой функции.
                        Может быть объединено несколько флагов через логическое
                        ИЛИ.
   @param[out] status   В данной структуре сохраняются результаты обработки фоновых
                        измерений и проверки корректности условий измерения, не
                        связанные с конкретным каналом.
                        Может быть передан нулевой указатель, если информация
                        не требуется.
   @param[out] chStatusList   Массив структур, количество элементов в котором должно быть
                        равно количеству элементов в логической таблице
                        ([LChCnt](@ref TLTR216_CONFIG::LChCnt)).
                        В нем сохраняются результаты обработки фоновых измерений и
                        проверки корректности условий измерения, связанные с соответствующим
                        логическим каналом.
                        Может быть передан нулевой указатель, если информация
                        не требуется.
   @return              [Код ошибки](@ref e_LTR216_ERROR_CODES).
  *****************************************************************************/
LTR216API_DllExport(INT) LTR216_ProcessData(TLTR216 *hnd, const DWORD *src, double *dest,
                                            INT *size, DWORD flags,
                                            TLTR216_DATA_STATUS *status, TLTR216_DATA_CH_STATUS *chStatusList);

/****************************************************************************//**
   @brief Проверка линий подключения и условий измерения

   Функция позволяет выполнить вспомогательные измерения, для проверки целостности
   линий и корректности условий измерения, не запуская основной цикл сбора данных.

   Функция принимает в качестве параметра свой набор из флагов, определяющих,
   какие вспомогательные измерения должны быть проведены. Назначение флагов аналогично
   фоновым измерениям, но сам набор может отличаться от того, на который настроен
   модуль.

   Результаты проверок и значения этих фоновых измерений сохраняются
   в структуры, переданные в качестве параметров status и chStatusList, аналогично
   LTR216_ProcessData().

   @param[in] hnd      Описатель модуля.
   @param[in] measList Набор флагов из #e_LTR216_BG_MEAS, определяющий, какие
                       вспомогательные измерения нужно провести.
   @param[out] status   В данной структуре сохраняются результаты обработки фоновых
                        измерений и проверки корректности условий измерения, не
                        связанные с конкретным каналом.
                        Может быть передан нулевой указатель, если информация
                        не требуется.
   @param[out] chStatusList   Массив структур, количество элементов в котором должно быть
                        равно количеству элементов в логической таблице
                        ([LChCnt](@ref TLTR216_CONFIG::LChCnt)).
                        В нем сохраняются результаты обработки фоновых измерений и
                        проверки корректности условий измерения, связанные с соответствующим
                        логическим каналом.
                        Может быть передан нулевой указатель, если информация
                        не требуется.
   @return              [Код ошибки](@ref e_LTR216_ERROR_CODES).
  *****************************************************************************/
LTR216API_DllExport(INT) LTR216_CheckMeasStatus(TLTR216 *hnd, DWORD measList,
                                                TLTR216_DATA_STATUS *status,
                                                TLTR216_DATA_CH_STATUS *chStatusList);
/** @} */



/***************************************************************************//**
    @addtogroup func_tare Функции для пользовательской тарировки каналов
    @{
*******************************************************************************/

/** @brief Тарировка смещения нуля выбранных каналов.

   Функция выполняет тарировку смещения нуля для выбранных с помощью параметра
   chsMask каналов.

   Для каждого выбранного канала должен присутствовать элемент логической таблицы
   с соответствующим номером физического канала. Соответствующий логический канал
   и определяет диапазон измерения, на котором будет выполняться тарировка
   соответствующего физического канала.

   Также должны быть предварительно заполнены все остальные настройки модуля.

   Перед вызовом этой функции на каждый тарируемый канал необходимо подать сигнал,
   который должен соответствовать нулевому значению после выполнения тарировки.

   При выполнении тарировки сперва подбирается значение ЦАП для компенсации смещения
   нуля, величиной до двух диапазонов измерения. После этого выполняется подбор
   дополнительного калибровочного коэффициента для тонкой компенсации нуля.

   В случае успешного завершения тарировки данная функция обновляет информацию
   в полях [DacValue](@ref TLTR216_TARE_CH_COEFS::DacValue) и
   [Offset](@ref TLTR216_TARE_CH_COEFS::Offset) (а также признаки их действительности)
   структуры #TLTR216_TARE_CH_COEFS с информацией о тарировке соответствующего
   канала в поле [ModuleInfo](@ref TLTR216::ModuleInfo) описателя модуля.

   В случае, если не по всем заданным каналам удалось измерить смещение, то
   функция вернет ошибку #LTR216_ERR_TARE_CHANNELS и по возвращаемой маске
   в параметре chsOk можно определить, тарировка каких каналов выполнена успешно.

   
   @param[in]   hnd      Описатель модуля.
   @param[in]   chsMask  Маска каналов, для которых должна быть выполнена тарировка.
                         Каждому физическому каналу соответствует свой бит, начиная
                         с младшего. Если бит установлен, то по соответствующему каналу будет 
                         выполнена тарировка.
   @param[out]  chsOk    В данной переменной возвращается битовая маска, указывающая,
                         по каким каналам успешно выполнена тарировка (бит установлен
                         в 1, если тарировка по соответствующему каналу выполнена успешно).
   @return              [Код ошибки](@ref e_LTR216_ERROR_CODES).
  *****************************************************************************/
LTR216API_DllExport(INT) LTR216_TareOffset(TLTR216 *hnd, DWORD chsMask, DWORD *chsOk);

/** @brief Тарировка шкалы выбранных каналов.

   Функция выполняет тарировку шкалы для выбранных с помощью параметра
   chsMask каналов.

   Тарировка шкалы выполняется после тарировки нуля, если необходимо выполнить
   обе тарировки.

   Для каждого выбранного канала должен присутствовать элемент логической таблицы
   с соответствующим номером физического канала. Соответствующий логический канал
   и определяет диапазон измерения, на котором будет выполняться тарировка
   соответствующего физического канала.

   Также должны быть предварительно заполнены все остальные настройки модуля.

   Перед выполнением этой функции необходимо, чтобы значение \f$ U_{ref} \f$ уже было
   измерено с помощью LTR216_InitMeasParams() с флагом #LTR216_INIT_MEAS_UREF
   при текущий уловиях и настройках, т.к. шкала калибруется по значению разбаланса,
   который вычисляется с учетом опорного напряжения.

   Перед вызовом этой функции на каждый тарируемый канал необходимо подать сигнал,
   который должен соответствовать максимальному измеряемому значению пользовательской
   шкалы.

   В случае успешного завершения тарировки данная функция обновляет информацию
   в поле [Scale](@ref TLTR216_TARE_CH_COEFS::Scale) (а также признак
   действительности этого поля) в соответствующем каналу элементе массива
   [Tare](@ref TLTR216_MODULE_INFO::Tare) в структуре [ModuleInfo](@ref TLTR216::ModuleInfo)
   описателя модуля.

   В случае, если не по всем заданным каналам удалось измерить смещение, то
   функция вернет ошибку #LTR216_ERR_TARE_CHANNELS и по возвращаемой маске
   в параметре chsOk можно определить, тарировка каких каналов выполнена успешно.

   @param[in]   hnd      Описатель модуля.
   @param[in]   chsMask  Маска каналов, для которых должна быть выполнена тарировка.
                         Каждому физическому каналу соответствует свой бит, начиная
                         с младшего. Если бит установлен, то по соответствующему каналу будет 
                         выполнена тарировка.
   @param[out]  chsOk    В данной переменной возвращается битовая маска, указывающая,
                         по каким каналам успешно выполнена тарировка (бит установлен
                         в 1, если тарировка по соответствующему каналу выполнена успешно).
   @return              [Код ошибки](@ref e_LTR216_ERROR_CODES).
  *****************************************************************************/
LTR216API_DllExport(INT) LTR216_TareScale(TLTR216 *hnd, DWORD chsMask, DWORD *chsOk);

/** @brief Запись тарировочных коэффициентов во flash-память

   Функция выполняет запись полученных тарировочных коэффициентов для заданных
   каналов из соответствующих элементов массива
   [Tare](@ref TLTR216_MODULE_INFO::Tare) в структуре [ModuleInfo](@ref TLTR216::ModuleInfo)
   описателя модуля в энергонезависимую flash-память модуля.

   Информация во flash-памяти по неуказанным каналам остается без изменений.

   Запись коэффициентов во flash-память позволяет их автоматически восстановить
   при последующих сеансах работы с модулем (при вызове LTR216_Open())  или вручную 
   через LTR216_ReadTareInfo().

   @param[in]   hnd      Описатель модуля.
   @param[in]   chsMask  Маска каналов, для которых должна быть выполнена запись
                         информации во flash-память.
   @return              [Код ошибки](@ref e_LTR216_ERROR_CODES).
  *****************************************************************************/
LTR216API_DllExport(INT) LTR216_WriteTareInfo(TLTR216 *hnd, DWORD chsMask);


/** @brief Чтение тарировочных коэффициентов из flash-памяти

   Функция выполняет чтение тарировочных коэффициентов для заданных каналов
   из энергонезависимой flash-памяти модуля и сохранение их в соответствующих элементах массива
   [Tare](@ref TLTR216_MODULE_INFO::Tare) в структуре [ModuleInfo](@ref TLTR216::ModuleInfo)
   описателя модуля.

   Информация в элементах массива [Tare](@ref TLTR216_MODULE_INFO::Tare), соответствующим
   неуказанным каналам, остается без изменений.

   Чтение тарировочных коэффициентов происходит автоматически по всем каналам после
   открытия соединения с модулем функцией LTR216_Open(). Данная функция позволяет
   их повторно прочитать для восстановления, в случае если они были изменены
   вручную или при выполнении тарировки без записи во flash-память.

   @param[in]   hnd      Описатель модуля.
   @param[in]   chsMask  Маска каналов, для которых должно быть выполнено чтение
                         информации из flash-памяти.
   @return              [Код ошибки](@ref e_LTR216_ERROR_CODES).
  *****************************************************************************/
LTR216API_DllExport(INT) LTR216_ReadTareInfo(TLTR216 *hnd, DWORD chsMask);

/** @} */

/***************************************************************************//**
    @addtogroup func_flash Функции для работы с flash-памятью модуля
    @{
*******************************************************************************/


/***************************************************************************//**
    @brief Чтение блока данных из Flash-памяти.

    Функция считывает массив данных из Flash-памяти модуля в массив, переданный
    пользователем. Для считывания не нужно специальное разрешение --- данная
    операция всегда доступна.
    
    @param[in]  hnd     Описатель модуля.
    @param[in]  addr    Адрес начала блока во flash-памяти.
    @param[out] data    Массив, куда будут сохранены считанные данные
                          (должен быть не менее size байт).
    @param[in]  size    Количество байт для чтения.
    @return             Код ошибки.
 ******************************************************************************/
LTR216API_DllExport(INT) LTR216_FlashRead(TLTR216 *hnd, DWORD addr, BYTE *data, DWORD size);
/***************************************************************************//**
    @brief Запись блока данных во Flash-память модуля.

    Функция записывает переданный массив данных во Flash-память модуля.
    Эта область должна быть предварительно стерта с помощью LTR216_FlashErase() и
    изменения в пользовательской области должны быть ранее разрешены через 
    LTR216_FlashWriteEnable().

    Пользователю для записи доступна только область памяти размером
    #LTR216_FLASH_USERDATA_SIZE с адреса #LTR216_FLASH_USERDATA_ADDR.
    @param[in]  hnd     Описатель модуля.
    @param[in]  addr    Адрес начала блока.
    @param[in]  data    Массив с записываемыми данными (должен быть не меньше
                        size байт).
    @param[in]  size    Количество байт для записи.
    @return             Код ошибки.
 ******************************************************************************/
LTR216API_DllExport(INT) LTR216_FlashWrite(TLTR216 *hnd, DWORD addr, const BYTE *data, DWORD size);

/***************************************************************************//**
    @brief Стирание блока во Flash-памяти.

    Функция стирает блок во Flash-памяти модуля (все ячейки будут читаться как
    0xFF). Адрес и размер должны быть кратны 256 байт.
    
    Стирание необходимо перед записью данных с помощью LTR216_FlashWrite().
    
    Для работы данной функции должны быть разрешены  изменения в пользовательской 
    области flash-памяти с помощью LTR216_FlashWriteEnable().
    @param[in]  hnd     Описатель модуля.
    @param[in]  addr    Адрес начала блока (должен быть кратен 256).
    @param[in]  size    Количество байт для стирания (кратно 256).
    @return             Код ошибки.
 ******************************************************************************/
LTR216API_DllExport(INT) LTR216_FlashErase(TLTR216 *hnd, DWORD addr, DWORD size);


/***************************************************************************//**
    @brief Разрешение записи в пользовательскую область Flash-памяти.

    Функция разрешает запись в пользовательскую область Flash-памяти. 
    Пользовательская область начинается с адреса #LTR216_FLASH_USERDATA_ADDR и
    занимает #LTR216_FLASH_USERDATA_SIZE байт.
    
    Должна быть вызвана перед использованием
    LTR216_FlashErase() и LTR216_FlashWrite() для изменения
    содержимого пользовательской области памяти. 
    
    После завершения изменений следует вызвать LTR216_FlashWriteDisable().
    @param[in]  hnd     Описатель модуля.
    @return              [Код ошибки](@ref e_LTR216_ERROR_CODES).
 ******************************************************************************/
LTR216API_DllExport(INT) LTR216_FlashWriteEnable(TLTR216 *hnd);
/***************************************************************************//**
    @brief Запрет записи в пользовательскую область Flash-памяти.

    Функция запрещает запись в пользовательскую область Flash-памяти модуля.
    Должна быть вызвана после того, как
    нужные данные в пользовательской области были изменены с помощью
    LTR216_FlashErase() и LTR216_FlashWrite(), чтобы защитить пользовательскую
    область от случайного изменения в дальнейшем.
    @param[in]  hnd     Описатель модуля.
    @return              [Код ошибки](@ref e_LTR216_ERROR_CODES).
 ******************************************************************************/
LTR216API_DllExport(INT) LTR216_FlashWriteDisable(TLTR216 *hnd);

/** @} */


/***************************************************************************//**
    @addtogroup func_misc Функции вспомогательного характера
    @{
*******************************************************************************/

/***************************************************************************//**
   @brief Разрешение или запрет источников тока питания датчиков.

   Данная функция позволяет вручную управлять разрешением 
   источников тока питания датчиков.

   Источники тока включены, если они разрешены и сконфигурированы.

   По умолчанию источники тока разрешены и будут включены после первого вызова
   LTR216_SetADC() и до последующего сброса модуля путем закрытия и открытия
   нового соединения с ним. В большинстве случаев это поведение соответствует ожидаемому
   и явное управление разрешением не требуется.

   Но, если по какой-то причине это необходимо, пользователь может явно запретить
   источники тока, вызвав данную функцию со значением параметра enabled равным ЛОЖЬ,
   в результате чего они будет выключены до следующего включения
   этой же функцией c enabled равным ИСТИНА. 
   
   При этом при разрешении источников тока  будет установлен код, соответствующий
   значению полю [ISrcCode](@ref TLTR216_CONFIG::ISrcCode) конфигурации модуля.

    @param[in] hnd       Описатель модуля.
    @param[in] enabled   Параметр указывает, должны быть источники тока разрешены или 
                         запрещены.
    @return              [Код ошибки](@ref e_LTR216_ERROR_CODES).
******************************************************************************/
LTR216API_DllExport(INT) LTR216_SetISrcEnabled(TLTR216 *hnd, BOOLEAN enabled);


/***************************************************************************//**
   @brief Получение сообщения об ошибке.

   Функция возвращает строку, соответствующую переданному коду ошибки, в кодировке
   CP1251 для ОС Windows или UTF-8 для ОС Linux. Функция может обработать как ошибки
   из ltr216api, так и общие коды ошибок из ltrapi.

   @param[in] err       Код ошибки
   @return              Указатель на строку, содержащую сообщение об ошибке.
 ******************************************************************************/
LTR216API_DllExport(LPCSTR) LTR216_GetErrorString(INT err);


/** @} */

#ifdef __cplusplus
}
#endif

#endif
