#ifndef LTR24API_H_
#define LTR24API_H_


#include "ltrapi.h"


#ifdef __cplusplus
extern "C" {
#endif


/*===============================================================================================*/
#ifdef _WIN32
    #ifdef LTR24API_EXPORTS
        #define LTR24_EXPORT(type) __declspec(dllexport) type APIENTRY
    #else
        #define LTR24_EXPORT(type) __declspec(dllimport) type APIENTRY
    #endif
#elif defined __GNUC__
    #define LTR24_EXPORT(type) __attribute__ ((visibility("default"))) type
#else
    #define LTR24_EXPORT(type) type
#endif

/** @cond obsoleted */
#define LTR24_VERSION_CODE      0x02010000UL
#define LTR24_CHANNEL_NUM       LTR24_CHANNEL_CNT
#define LTR24_RANGE_NUM         LTR24_RANGE_CNT
#define LTR24_ICP_RANGE_NUM     LTR24_ICP_RANGE_CNT
#define LTR24_FREQ_NUM          LTR24_FREQ_CNT
#define LTR24_I_SRC_VALUE_NUM   LTR24_I_SRC_VALUE_CNT
/** @endcond */


/***************************************************************************//**
  @addtogroup const_list Константы и перечисления.
  @{
  *****************************************************************************/

/** Количество каналов АЦП модуля LTR24 */
#define LTR24_CHANNEL_CNT       4

/** Количество диапазонов в режиме дифференциального входа */
#define LTR24_RANGE_CNT         2

/** Количество диапазонов в режиме ICP-входа */
#define LTR24_ICP_RANGE_CNT     2

/** Количество частот дискретизации */
#define LTR24_FREQ_CNT          16

/** Количество значений источника тока */
#define LTR24_I_SRC_VALUE_CNT    2

/** Размер поля с названием модуля */
#define LTR24_NAME_SIZE          8

/** Размер поля с серийным номером модуля */
#define LTR24_SERIAL_SIZE       16

/** @brief Коды ошибок, специфичные для модуля LTR24.

   Коды ошибок, которые определены и используются только в ltr24api.
   Остальные коды ошибок, которые используются разными модулями, определены в ltrapi.h */
typedef enum {
    LTR24_ERR_INVAL_FREQ        = -10100, /**< Задана некорректная частота дискретизации */
    LTR24_ERR_INVAL_FORMAT      = -10101, /**< Задан некорректный формат данных */
    LTR24_ERR_CFG_UNSUP_CH_CNT  = -10102, /**< Для заданной частоты и 24-битного формата не
                                               поддерживается установленное количество каналов */
    LTR24_ERR_INVAL_RANGE       = -10103, /**< Задан некорректный диапазон канала */
    LTR24_ERR_WRONG_CRC         = -10104, /**< Неверная контрольная сумма EEPROM */
    LTR24_ERR_VERIFY_FAILED     = -10105, /**< Ошибка верификации записи в EEPROM */
    LTR24_ERR_DATA_FORMAT       = -10106, /**< Неверный формат данных в обработанных отсчетах */
    LTR24_ERR_UNALIGNED_DATA    = -10107, /**< Не выровненные данные */
    LTR24_ERR_DISCONT_DATA      = -10108, /**< Сбой счетчика данных в обработанных отсчетах */
    LTR24_ERR_CHANNELS_DISBL    = -10109, /**< Ни один канал не был разрешен */
    LTR24_ERR_UNSUP_VERS        = -10110, /**< Версия формата управляющей структуры модуля не поддерживается */
    LTR24_ERR_FRAME_NOT_FOUND   = -10111, /**< Начало кадра не найдено */
    LTR24_ERR_UNSUP_FLASH_FMT   = -10116, /**< Не поддерживаемый формат информации во Flash-памяти */
    LTR24_ERR_INVAL_I_SRC_VALUE = -10117, /**< Задано некорректное значение источника тока */
    LTR24_ERR_UNSUP_ICP_MODE    = -10118, /**< Данная модификация модуля не поддерживает ICP-режим */
} e_LTR24_ERROR_CODES;



/** Коды частот сбора АЦП */
typedef enum {
    LTR24_FREQ_117K     = 0, /**< 117.1875 кГц */
    LTR24_FREQ_78K      = 1, /**< 78.125 кГц */
    LTR24_FREQ_58K      = 2, /**< 58.59375 кГц */
    LTR24_FREQ_39K      = 3, /**< 39.0625 кГц */
    LTR24_FREQ_29K      = 4, /**< 29.296875 кГц */
    LTR24_FREQ_19K      = 5, /**< 19.53125 кГц */
    LTR24_FREQ_14K      = 6, /**< 14.6484375 кГц */
    LTR24_FREQ_9K7      = 7, /**< 9.765625 кГц */
    LTR24_FREQ_7K3      = 8, /**< 7.32421875 кГц */
    LTR24_FREQ_4K8      = 9, /**< 4.8828125 кГц */
    LTR24_FREQ_3K6      = 10, /**< 3.662109375 кГц */
    LTR24_FREQ_2K4      = 11, /**< 2.44140625 кГц */
    LTR24_FREQ_1K8      = 12, /**< 1.8310546875 кГц */
    LTR24_FREQ_1K2      = 13, /**< 1.220703125 кГц */
    LTR24_FREQ_915      = 14, /**< 915.52734375 Гц */
    LTR24_FREQ_610      = 15, /**< 610.3515625 Гц */
} e_LTR24_FREQS;

/** Коды диапазонов в режиме дифференциального входа */
typedef enum {
    LTR24_RANGE_2  = 0, /**< \f$\pm2\f$ В  */
    LTR24_RANGE_10 = 1  /**< \f$\pm10\f$ В */
} e_LTR24_RANGES;

/** Коды диапазонов в режиме ICP входа */
typedef enum {
    LTR24_ICP_RANGE_1 = 0, /**< \f$~1\f$ В (RMS) */
    LTR24_ICP_RANGE_5 = 1  /**< \f$~5\f$ В (RMS) */
} e_LTR24_ICP_RANGES;

/** Значения источника тока */
typedef enum {
    LTR24_I_SRC_VALUE_2_86   = 0, /**< 2.86 мА. */
    LTR24_I_SRC_VALUE_10     = 1  /**< 10 мА. */
} e_LTR24_I_SOURCES;

/** Форматы данных от модуля */
typedef enum {
    LTR24_FORMAT_20 = 0, /**< 20-битный целочисленный (1 слово на отсчет) */
    LTR24_FORMAT_24 = 1, /**< 24-битный целочисленный (2 слова на отсчет) */
} e_LTR24_FORMATS;


/** @brief Дополнительные возможности, поддерживаемые модулем

    Флаги, указывающие, какие дополнительные возможности реализованы в данной
    версии модуля. */
typedef enum {
    /** Расширенная полоса пропускания в области низких частот в ICP режиме.
     *  Относится к модулям LTR24-2, выпущенным в 2020 году и позже.
     *  Флаг указывает, что в RC-цепи, использующейся для удаления постоянной
     *  составляющей сигнала на ICP-входе, установлено сопротивление 110 кОм
     *  вместо 25.5 кОм, что приводит к расширению нижней границы полосы
     *  частот пропускания, а также к меньшему влиянию корректирующего фильтра
     *  фазы на АЧХ в области нижних частот и меньшей зависимости падения
     *  напряжения от выходного сопротивления датчика. */
    LTR24_FEATURE_ICP_EXT_BANDWIDTH_LF =    (1 << 0)
} e_LTR24_FEATURES;

/** @brief Флаги обработки данных

    Флаги, управляющие работой функции LTR24_ProcessData() */
typedef enum {
    /** Признак, что нужно выполнить калибровку, применив к данным калибровочные
     *  коэффициенты, как описано в разделе @ref sect_gen_cbr. */
    LTR24_PROC_FLAG_CALIBR          = 0x00000001,
    /** Признак, что нужно перевести коды АЦП в Вольты. */
    LTR24_PROC_FLAG_VOLT            = 0x00000002,
    /** Признак, что необходимо выполнить коррекцию АЧХ. Данный флаг оставлен
        для совместимости со старыми приложениями. В новых  программах
        рекомендуется использовать #LTR24_PROC_FLAG_AFC_COR_EX. */
    LTR24_PROC_FLAG_AFC_COR         = 0x00000004,
    /** Признак, что необходимо выполнить коррекцию АЧХ
        с помощью нового набора фильтров, которые не влияют
        на ФЧХ модуля, как описано в разделе @ref sec_gen_afc_cor. */
    LTR24_PROC_FLAG_AFC_COR_EX      = 0x00000008,
    /** Признак, что необходимо выполнить коррекцию фазы каналов модуля, работающих
     * в ICP режиме, с помощью специального фильтра, как описано
     * в разделе @ref sec_sensor_pha_cor. */
    LTR24_PROC_FLAG_ICP_PHASE_COR   = 0x00000010,

    /** По умолчанию функция LTR24_ProcessData() предполагает, что ей на обработку
        передаются все принятые данные и проверяет непрерывность счетчика не только
        внутри переданного блока данных, но и между вызовами, а также использует
        состояние фильтров с прошлого вызова.
        Если обрабатываются не все данные или одни и те же данные обрабатываются
        повторно, то нужно указать данный флаг. */
    LTR24_PROC_FLAG_NONCONT_DATA    = 0x00000100
} e_LTR24_PROC_FLAGS;

/** @} */

/***************************************************************************//**
  @addtogroup type_list Типы данных.
  @{
  *****************************************************************************/

#pragma pack(4)


/** @brief Калибровочные коэффициенты.

    Структура, хранящая калибровочные коэффициенты для одного канала
    и диапазона на заданной частоте дискретизации. */
typedef struct {
    float   Offset; /**< Код смещения */
    float   Scale;  /**< Коэффициент шкалы */
} TLTR24_CBR_COEF;

/** @brief Коэффициенты БИХ-фильтра коррекции АЧХ */
typedef struct {
    double a0; /**< Параметр, используемых для расчета коэффициентов БИХ-фильтра */
    double a1; /**< Параметр, используемых для расчета коэффициентов БИХ-фильтра */
    double b0; /**< Параметр, используемых для расчета коэффициентов БИХ-фильтра */
} TLTR24_AFC_IIR_COEF;

/** @brief Набор коэффициентов для коррекции АЧХ модуля */
typedef struct {
    /** Частота сигнала, для которой снято отношение амплитуд из FirCoef */
    double AfcFreq;
    /**  Набор отношений измеренной амплитуды синусоидального сигнала
         к реальной амплитуде для макс. частоты дискретизации и частоты сигнала
         из AfcFreq для каждого канала и каждого диапазона */
    double FirCoef[LTR24_CHANNEL_CNT][LTR24_RANGE_NUM];
    /** Коэффициенты БИХ-фильтра для коррекции АЧХ АЦП на частотах
     *  #LTR24_FREQ_39K и ниже */
    TLTR24_AFC_IIR_COEF AfcIirCoef;
} TLTR24_AFC_COEFS;

/** @brief Набор параметров для коррекции ФЧХ модуля */
typedef  struct {
    /** Частота, на которой измерен сдвиг фаз каналов модуля */
    double PhaseShiftRefFreq;
    double PhaseShift[LTR24_CHANNEL_CNT]; /**< Сдвиг фазы для каждого канала модуля в градусах */
} TLTR24_ICP_PHASE_SHIFT_COEFS;



/** @brief  Информация о модуле.

    Структура, содержащая информацию о данном экземпляре модуля.

    Поля @fieldref{TLTR24_MODULE_INFO,VerPLD} и @fieldref{TLTR24_MODULE_INFO,SupportICP}
    заполняются после установления связи с модулем с помощью LTR24_Open() на
    основе информации, полученной от модуля.

    Остальные поля заполняются информацией из энергонезависимой памяти после
    вызова функции LTR24_GetConfig().
 */
typedef struct {    
    CHAR        Name[LTR24_NAME_SIZE]; /**<  Название модуля ("LTR24").  */
    CHAR        Serial[LTR24_SERIAL_SIZE]; /**< Серийный номер модуля. */
    BYTE        VerPLD; /**< Версия прошивки PLD. */
    BOOL        SupportICP; /**< Признак, поддерживает ли модуль режим измерения
                                 сигнала с ICP датчиков. */
    DWORD       SupportedFeatures; /**< Флаги из #e_LTR24_FEATURES, указывающие,
                                        какие дополнительные возможности
                                         поддерживает данная версия модуля */
    DWORD       Reserved[7]; /**< Зарезервированные поля.  */
    /** Массив заводских калибровочных коэффициентов. */
    TLTR24_CBR_COEF CalibCoef[LTR24_CHANNEL_CNT][LTR24_RANGE_NUM][LTR24_FREQ_NUM];
    /** Коэффициенты для корректировки АЧХ модуля. */
    TLTR24_AFC_COEFS AfcCoef;
    /** Измеренные значения источников токов для каждого канала
        (только для LTR24-2). */
    double ISrcVals[LTR24_CHANNEL_CNT][LTR24_I_SRC_VALUE_NUM];
} TLTR24_MODULE_INFO;


/** @brief Настройки канала АЦП.

    Структура, содержащая настройки одного канала АЦП. */
typedef struct {
    BOOL    Enable; /**< Признак, разрешен ли сбор по данному каналу. Если равно TRUE,
                         то модулем будут передаваться слова, соответствующие отсчетам
                         от данного канала, если FALSE --– нет. */
    BYTE    Range; /**< Диапазон измерения канала. Устанавливается равным одному
                        из значений перечисления #e_LTR24_RANGES (если выбран
                        дифференциальный режим) или из #e_LTR24_ICP_RANGES
                        (если выбран ICP режим) в зависимости от значения
                        поля @fieldref{TLTR24_CHANNEL_CONFIG,ICPMode}.*/
    BOOL    AC; /**< Включение режима отсечки постоянной составляющей. Если
                     поле равно TRUE, то на входе канала влючается аналоговый
                     фильтр для отсечки постоянной составляющей. Имеет значение
                     только при @fieldref{TLTR24_CHANNEL_CONFIG,ICPMode}
                     равном FALSE. */
    BOOL    ICPMode; /**< Включение режима измерения сигнала с ICP датчиков.
                          Если FALSE –-- используется режим измерения
                          "Дифференциальный вход" или "Измерение нуля"
                          (в зависимости от поля @fieldref{TLTR24,TestMode}),
                          если TRUE – режим "ICP вход" или "ICP тест".
                          Для включения режима измерения с ICP датчиков необходимо,
                          чтобы модуль поддерживал данный режим.
                          Режим измерения с ICP датчиков предполагает подключение
                          сигнала к другим контактам разъема модуля. */
    float SensorROut; /**< Выходное сопротивление датчика, подключенного к
                            данному входу ICP, с возможным учетом сопртивления
                            проводов подключения датчика, которое будет использовано
                            для коррекции падения напряжения из-за его взаимодействия
                            с входным сопротивлением ICP-входа.

                            Нулевое значение (по умолчанию) или значение меньше
                            нуля указывает, что коррекция падения напряжения
                            не требуется. */
    DWORD   Reserved[3]; /**< Резервные поля (не должны изменяться пользователем) */
}  TLTR24_CHANNEL_CONFIG;


/** @brief  Управляющая структура модуля.

    Хранит текущие настройки модуля, информацию о
    его состоянии, структуру канала связи. Передается в большинство функций
    библиотеки. Некоторые поля структуры доступны для изменения пользователем
    для настройки параметров модуля. Перед использованием требует инициализации
    с помощью функции LTR24_Init().
 */
typedef struct {
    INT         Size; /**< Размер структуры. Заполняется при вызове LTR24_Init(). */
    /** Структура, содержащая состояние клиентского соединения со службой ltrd.
        Не используется напрямую пользователем. */
    TLTR        Channel;
    /** Признак, запущен ли сбор данных. Значение изменяется функциями модуля
         и не должно меняться вручную. */
    BOOL        Run;
    /** Настройка частоты дискретизации. Должно быть равно одному из значений
        перечисления #e_LTR24_FREQS.

        Значение устанавливается пользователем при конфигурации модуля. */
    BYTE        ADCFreqCode;
    /** Значение частоты дискретизации в Гц.
        Заполняется автоматически значением частоты дискретизации, соответствующему
        коду в поле @fieldref{TLTR24,ADCFreqCode}, после вызова функции LTR24_SetADC(). */
    double      ADCFreq;
    /** Настройка формата передачи данных. Должно быть равно одному из значений
        перечисления #e_LTR24_FORMATS.

        Значение устанавливается пользователем при конфигурации модуля. */
    BYTE        DataFmt;
    /** Настройка значения источника тока для всех каналов подключения ICP-датчиков.
        Должно быть равно одному из значений перечисления #e_LTR24_I_SOURCES.
        Имеет значение только в случае, если модуль поддерживает режим измерения
        с сигнала с ICP датчиков.

        Значение устанавливается пользователем при конфигурации модуля. */
    BYTE        ISrcValue;
    /**  Включение тестовых режимов ("Измерение нуля" или "ICP-тест" в зависимости от
         значения значения поля @fieldref{TLTR24_CHANNEL_CONFIG,ICPMode} для каждого канала)
         для всех каналов модуля.

         Значение устанавливается пользователем при конфигурации модуля. */
    BOOL        TestMode;
    /** Резервные поля. Не должны изменяться пользователем */
    DWORD       Reserved[16];
    /** Настройки каналов АЦП.

        Все поля устанавливаются пользователем при конфигурации модуля. */
    TLTR24_CHANNEL_CONFIG ChannelMode[LTR24_CHANNEL_CNT];
    /** Информация о модуле.

        Значения полей обновляются при вызовах LTR24_Open() и LTR24_GetConfig()
        и не предназначены для ручного изменения пользователем.
        */
    TLTR24_MODULE_INFO ModuleInfo;
    /** Массив используемых калибровочных коэффициентов по каждому каналу,
        диапазону и частоте.

        Данные коэффициенты применяются для коррекции данных в функции
        LTR24_ProcessData().
        При вызове LTR24_GetConfig() в данные поля копируются заводские
        калибровочные коэффициенты (те же, что и в поля @fieldref{TLTR24,ModuleInfo}).
        При необходимости, пользователь может записать в данные поля
        свои коэффициенты. */
    TLTR24_CBR_COEF CalibCoef[LTR24_CHANNEL_CNT][LTR24_RANGE_NUM][LTR24_FREQ_NUM];
    /**  Коэффициенты для корректировки АЧХ, применяемые в функции
         LTR24_ProcessData().

         При вызове LTR24_GetConfig() в поля данной структуры копируются значения 
         из ПЗУ модуля (те же, что и в поля @fieldref{TLTR24,ModuleInfo}). */
    TLTR24_AFC_COEFS AfcCoef;
    /** Указатель на непрозрачную структуру с внутренними параметрами,
        используемыми исключительно библиотекой и недоступными для пользователя. */
    PVOID       Internal;
} TLTR24;

#pragma pack()


/*================================================================================================*/
/** @} */

/***************************************************************************//**
    @addtogroup func_init Функции инициализации и работы с соединением с модулем.
    @{
*******************************************************************************/
/***************************************************************************//**
  @brief Инициализация описателя модуля.

  Функция инициализирует поля структуры описателя модуля значениями по умолчанию.
  Эта функция должна вызываться для каждой структуры #TLTR24 перед вызовом
  остальных функций.
   @param[in] hnd       Описатель модуля.
   @return              [Код ошибки](@ref e_LTR24_ERROR_CODES).
 ******************************************************************************/
LTR24_EXPORT(INT) LTR24_Init(TLTR24 *hnd);

/***************************************************************************//**
   @brief Открытие соединения с модулем.

   Функция устанавливает соединение с модулем в соответствии с переданными
   параметрами и проверяет наличие и тип модуля.
   Должна быть вызвана перед работой с модулем. После завершения работы
   необходимо закрыть соединение с помощью LTR24_Close().

   @param[in] hnd       Описатель модуля.
   @param[in] ltrd_addr  IP-адрес машины, на которой запущена служба ltrd, в 32-битном
                        формате (описан в разделе "Формат задания IP-адресов"
                        @docref_ltrapi{руководства для библиотеки ltrapi}).
                        Если служба ltrd запущена на той же машине, что и программа,
                        вызывающая данную функцию, то в качестве адреса
                        можно передать LTRD_ADDR_DEFAULT.
   @param[in] ltrd_port TCP-порт для подключения к службе ltrd. По умолчанию
                        используется LTRD_PORT_DEFAULT.
   @param[in] csn       Серийный номер крейта, в котором находится интересующий
                        модуль. Представляет собой оканчивающуюся нулем ASCII-строку.
                        Для соединения с первым найденным крейтом можно передать
                        пустую строку или нулевой указатель.
   @param[in] slot      Номер слота крейта, в котором установлен интересующий модуль.
                        Значение от LTR_CC_CHNUM_MODULE1 до LTR_CC_CHNUM_MODULE16.
   @return              [Код ошибки](@ref e_LTR24_ERROR_CODES).
*******************************************************************************/
LTR24_EXPORT(INT) LTR24_Open(TLTR24 *hnd, DWORD ltrd_addr, WORD ltrd_port, const CHAR *csn, INT slot);

/** @cond kd_extension */
LTR24_EXPORT(INT) LTR24_OpenEx(TLTR24 *hnd, DWORD net_addr, WORD net_port, const CHAR *crate_sn,
                               INT slot, DWORD in_flags, DWORD *out_flags);
/** @endcond */

/***************************************************************************//**
   @brief Закрытие соединения с модулем.

   Функция закрывает ранее открытое с помощью LTR24_Open() соединение. Должна
   вызываться после завершения работы с модулем. При любом возвращенном значении
   после вызова этой функции соответствующий описатель уже нельзя использовать
   для работы с модулем без открытия нового соединения.

   @param[in] hnd       Описатель модуля.
   @return              [Код ошибки](@ref e_LTR24_ERROR_CODES).
 ******************************************************************************/
LTR24_EXPORT(INT) LTR24_Close(TLTR24 *hnd);

/***************************************************************************//**
   @brief Проверка, открыто ли соединение с модулем.

   Функция проверяет, открыто ли в данный момент соединение с модулем. Если
   соединение открыто, функция возвращает LTR_OK, если закрыто --- код ошибки
   LTR_ERROR_CHANNEL_CLOSED.
   @param[in] hnd       Описатель модуля.
   @return              [Код ошибки](@ref e_LTR24_ERROR_CODES)
                        (LTR_OK, если соединение установлено).
 ******************************************************************************/
LTR24_EXPORT(INT) LTR24_IsOpened(TLTR24 *hnd);

/***************************************************************************//**
    @brief Чтение информации и калибровочных коэффициентов.

    Функция считывает информацию из энергонезависимой памяти модуля и
    обновляет поля @fieldref{TLTR24,ModuleInfo}, @fieldref{TLTR24,CalibCoef} и
    @fieldref{TLTR24,AfcCoef} в управляющей структуре модуля.

    В отличие от некоторых других модулей для LTR24 эта операция не выполняется
    при открытии и как правило эта функция должна быть вызвана вручную после
    вызове LTR24_Open(). Может быть вызвана также повторно для восстановления
    измененных коэффициентов.

   @param[in] hnd      Описатель модуля.
   @return             [Код ошибки](@ref e_LTR24_ERROR_CODES).
 ******************************************************************************/
LTR24_EXPORT(INT) LTR24_GetConfig(TLTR24 *hnd);

/** @} */

/***************************************************************************//**
    @addtogroup func_config Функции для изменения настроек модуля
    @{
*******************************************************************************/

/***************************************************************************//**
   @brief Запись настроек в модуль.

   Функция передает настройки, соответствующие значениям конфигурационных полей
   [описателя модуля](@ref TLTR24) в модуль.
   Должна вызываться перед первым запуском сбора данных с помощью
   LTR24_Start().

   @param[in] hnd       Описатель модуля.
   @return              [Код ошибки](@ref e_LTR24_ERROR_CODES).
 ******************************************************************************/
LTR24_EXPORT(INT) LTR24_SetADC(TLTR24 *hnd);


/** @brief Управление режимом измерения нуля во время сбора.

   Функция включает или выключает тестовый режим (измерение нуля или ICP-тест)
   одновременно для всех каналов.
   Данная функция используется только во время сбора данных.
   Для настройки в режиме конфигурирования используется поле @fieldref{TLTR24,TestMode}
   управляющей структуры модуля.

   @param[in] hnd       Описатель модуля.
   @param[in] enable    Признак включения тестовых режимов
   @return              [Код ошибки](@ref e_LTR24_ERROR_CODES).
 ******************************************************************************/
LTR24_EXPORT(INT) LTR24_SetZeroMode(TLTR24 *hnd, BOOL enable);


/** @brief Управление режимом отсечки постоянной составляющей во время сбора.

   Функция включает или выключает режим отсечки постоянной составляющей
   выбранного канала. Канал должен быть настроен на измерение в дифференциальном
   режиме (т.к. в режиме ICP отсечка постоянной составляющей всегда включена).
   Данная функция используется только во время сбора данных.
   Для настройки в режиме конфигурирования используется поле
   @fieldref{TLTR24_CHANNEL_CONFIG,AC} в настройках соответствующего канала
   управляющей структуры модуля.

   @param[in] hnd       Описатель модуля.
   @param[in] chan      Номер канала (0-3).
   @param[in] ac_mode   Признак включения отсечки постоянной составляющей
   @return              [Код ошибки](@ref e_LTR24_ERROR_CODES).
 ******************************************************************************/
LTR24_EXPORT(INT) LTR24_SetACMode(TLTR24 *hnd, BYTE chan, BOOL ac_mode);

/** @} */



/***************************************************************************//**
    @addtogroup func_stream Функции для управления сбором данных
    @{
*******************************************************************************/

/***************************************************************************//**
    @brief Запуск сбора данных.

    При вызове данной функции запускается сбор данных с АЦП модуля. После успешного
    завершения этой функции запускается АЦП и модуль начинает передавать в ПК
    полученные отсчеты, которые необходимо вычитывать с помощью LTR24_Recv().
    При завершении измерения для остановки сбора данных необходимо вызвать
    LTR24_Stop().

    Хотя бы один из каналов АЦП должен быть до этого разрешен и модуль должен быть
    хотя бы раз сконфигурирован с помощью LTR24_SetADC().

   @param[in] hnd       Описатель модуля.
   @return              [Код ошибки](@ref e_LTR24_ERROR_CODES).
 ******************************************************************************/
LTR24_EXPORT(INT) LTR24_Start(TLTR24 *hnd);

/***************************************************************************//**
    @brief Останов сбора данных.

    При вызове данной функции модуль останавливает сбор и выдачу данных АЦП.

    При этом принимаются и отбрасываются все переданные, но непрочитанные
    данные от модуля.

   @param[in] hnd       Описатель модуля.
   @return              [Код ошибки](@ref e_LTR24_ERROR_CODES).
 ******************************************************************************/
LTR24_EXPORT(INT) LTR24_Stop(TLTR24 *hnd);

/***************************************************************************//**
   @brief Прием данных от модуля.

   Функция принимает запрашиваемое число слов от модуля. Возвращаемые слова
   находятся в специальном формате, который включает в себя служебную информацию.
   Формат и количество слов на один отсчет определяется настройкой
   @fieldref{TLTR24,DataFmt}.

   Для обработки принятых слов и получения значений АЦП используется функция
   LTR24_ProcessData().

   Функция возвращает управление либо когда примет запрошенное количество слов,
   либо по истечению таймаута. При этом реально принятое количество слов можно
   узнать по возвращенному значению.

   @param[in]  hnd      Описатель модуля.
   @param[out] data     Массив, в который будут сохранены принятые слова. Должен
                        быть размером на size 32-битных слов.
   @param[out] tmark    Указатель на массив размером на size 32-битных слов,
                        в который будут сохранены значения счетчиков синхрометок,
                        соответствующие принятым данным. Генерация меток
                        настраивается для крейта или специального модуля отдельно.
                        Синхрометки подробнее описаны в разделе "Синхрометки"
                        @docref_ltrapi{руководства для библиотеки ltrapi}.
                        Если синхрометки не используются, то можно передать
                        в качестве параметра нулевой указатель.
   @param[in]  size     Запрашиваемое количество 32-битных слов на прием.
   @param[in]  timeout  Таймаут на выполнение операции в миллисекундах. Если в течение
                        заданного времени не будет принято запрашиваемое количество
                        слов, то функция все равно вернет управление, возвратив
                        в качестве результата реально принятое количество слов
   @return              Значение меньше нуля соответствует коду ошибки. Значение
                        больше или равное нулю соответствует количеству реально
                        принятых и сохраненных в массив data слов.
*******************************************************************************/
LTR24_EXPORT(INT) LTR24_Recv(TLTR24 *hnd, DWORD *data, DWORD *tmark, DWORD size, DWORD timeout);

/** @cond kd_extension */
/* Аналогично LTR24_Recv принимает сырые данные от модуля. В массив time
 * складываются абсолютные метки времени для каждого отсчета в формате unixtime.
 */
LTR24_EXPORT(INT) LTR24_RecvEx(TLTR24 *ltr24, DWORD *data, DWORD *tmark, DWORD size, DWORD timeout,
    LONGLONG *time_vals);
/** @endcond */

/****************************************************************************//**
   @brief Обработка принятых от модуля слов.

   Функция используется для обработки слов, принятых от модуля с помощью LTR24_Recv().
   Функция проверяет служебные поля принятых слов, извлекает полезную информацию
   с отсчетами, выполняет их калибровку и переводит отсчеты в
   Вольты (при указании соответствующих флагов).

   Функция предполагает, что передаваемые слова выровнены на начало кадра (первое
   слово первого разрешенного канала). Если это не так, то неполный кадр в начале
   будет отброшен и функция вернет ошибку.

   При использовании 24-битного режима функция также анализирует
   признаки перегрузки разрядной сетки АЦП для каждого отсчета и может сохранить
   их в массив, переданный в качестве параметра ovload.

   По умолчанию функция предполагает, что все принятые данные обрабатываются
   и только один раз, сохраняя состояния фильтров между вызовами.
   Если это условие не выполняется, то нужно передать флаг #LTR24_PROC_FLAG_NONCONT_DATA.

   @param[in] hnd      Описатель модуля.
   @param[in] src      Указатель на массив, содержащий слова, принятые от модуля
                       с помощью LTR24_Recv(), которые нужно обработать.
   @param[out] dest    Указатель на массив, в который будут сохранены обработанные
                       данные. Порядок следования соответствует порядку во входном
                       массиве (т.е. сперва первый отсчет первого разрешенного канала,
                       затем первый отсчет второго канала и т.д.). Массив должен
                       иметь достаточно места для сохранения как минимум size
                       элементов для 20-битного режима или size/2 для 24-битного.
   @param[in,out] size  На входе принимает размер массива src для обработки. На
                        выходе, при успешном завершении, возвращает количество
                        сохраненных отсчетов в массиве dest.
   @param[in]  flags   Флаги из #e_LTR24_PROC_FLAGS, управляющие работой функции.
                        Может быть объединено несколько флагов через логическое
                        ИЛИ.
   @param[out] ovload   Массив для сохранения признаков перегрузки разрядной
                        сетки АЦП для каждого отсчета. Размер массива должен
                        соответствовать размеру массива dest.

                        Информация о перегрузке разрядной сетки АЦП доступна
                        только в 24-битном режиме, при 20-битном режиме
                        всегда сохраняются значения, соответствующие отсутствию признака.
                        Может быть передан нулевой
                        указатель, если эта информация не нужна.
   @return              [Код ошибки](@ref e_LTR24_ERROR_CODES).
  *****************************************************************************/
LTR24_EXPORT(INT) LTR24_ProcessData(TLTR24 *hnd, const DWORD *src, double *dest, INT *size,
                                    DWORD flags, BOOL *ovload);


/****************************************************************************//**
   @brief Поиск начала первого кадра.

   Функция находит в принятом массиве сырых данных от модуля индекс первого
   слова, соответствующего началу кадра. Может использоваться для выравнивания на
   начало кадра данных в случае произошедшего сбоя без останова сбора.


   @param[in] hnd      Описатель модуля.
   @param[in] data     Указатель на массив, содержащий слова, принятые от модуля
                       с помощью LTR24_Recv(), в которых ищется начало кадра.
   @param[in] size     Количество слов в массиве data
   @param[out] index   В этой переменной возвращается индекс элемента,
                       соответствующего началу первого кадра, если функция
                       завершилась успешно.
   @return             [Код ошибки](@ref e_LTR24_ERROR_CODES).
  *****************************************************************************/
LTR24_EXPORT(INT) LTR24_FindFrameStart(TLTR24 *hnd, const DWORD *data, INT size, INT *index);

/** @} */

/***************************************************************************//**
    @addtogroup func_misc Функции вспомогательного характера
    @{
*******************************************************************************/
/***************************************************************************//**
   @brief Получение сообщения об ошибке.

   Функция возвращает строку, соответствующую переданному коду ошибки, в кодировке
   CP1251 для ОС Windows или UTF-8 для ОС Linux. Функция может обработать как ошибки
   из ltr24api, так и общие коды ошибок из ltrapi.

   @param[in] err       [Код ошибки](@ref e_LTR24_ERROR_CODES).
   @return              Указатель на строку, содержащую сообщение об ошибке.
 ******************************************************************************/
LTR24_EXPORT(LPCSTR) LTR24_GetErrorString(INT err);


/****************************************************************************//**
   @brief Получение коэффициентов коррекции ФЧХ в ICP режиме.

   Функция возвращает структуру с информацией, которая используется для коррекции
   ФЧХ модуля в ICP режиме. Данная информация считывается из энергонезависимой памяти модуля
   при вызове LTR24_GetConfig(), поэтому вызов данной функции имеет смысл выполнять
   только после вызова LTR24_GetConfig().

   Если в энергонезависимой памяти не будет найдена необходимая информация, то
   используются коэффициенты по умолчанию, рассчитанные для модуля LTR24.

   @param[in] hnd           Описатель модуля.
   @param[out] PhaseCoefs   Структура, в которой возвращается информация о коэффициентах
                            для коррекции ФЧХ модуля в ICP режиме.
   @return                  [Код ошибки](@ref e_LTR24_ERROR_CODES).
  *****************************************************************************/
LTR24_EXPORT(INT) LTR24_GetICPPhaseCoefs(TLTR24 *hnd, TLTR24_ICP_PHASE_SHIFT_COEFS *PhaseCoefs);

/****************************************************************************//**
   @brief Установка коэффициентов коррекции ФЧХ в ICP режиме.

   Функция устанавливает новые коэффициенты для коррекции ФЧХ модуля в ICP режиме.
   Функция изменяет значения в описателе модуля, которые будут использоваться
   при обработке данных. По умолчанию данные коэффициенты считываются из
   энергонезависимой памяти модуля или устанавливаются в значения по умолчанию,
   если данной информации нет в памяти. Явный вызов данной функции в штатном 
   режиме работы не требуется.

   @param[in] hnd           Описатель модуля.
   @param[out] PhaseCoefs   Структура с коэффициентами для коррекции ФЧХ модуля
                            в ICP-режиме.
   @return                  [Код ошибки](@ref e_LTR24_ERROR_CODES).
  *****************************************************************************/
LTR24_EXPORT(INT) LTR24_SetICPPhaseCoefs(TLTR24 *hnd,  const TLTR24_ICP_PHASE_SHIFT_COEFS *PhaseCoefs);

/** @cond obsoleted */
/* Возвращает текущую версию библиотеки. */
LTR24_EXPORT(DWORD) LTR24_GetVersion(void);
/** @endcond */

#ifdef LTRAPI_USE_KD_STORESLOTS
/** @cond kd_extension */
/* Определяет данные в слоте для хранения управляющей структуры
 * как некорректные.
 */
LTR24_EXPORT(INT) LTR24_StoreConfig(TLTR24 *hnd, TLTR_CARD_START_MODE start_mode);
/** @endcond */
#endif

/** @} */

#ifndef LTRAPI_DISABLE_COMPAT_DEFS
/** @cond obsoleted */
typedef TLTR24_MODULE_INFO TINFO_LTR24;
/** @endcond */
#endif
#ifdef __cplusplus
}
#endif

#endif
