#ifndef LTRK416API_H_
#define LTRK416API_H_

#include "ltr01api.h"

#ifdef __cplusplus
extern "C" {                                 // only need to export C interface if
                                             // used by C++ source code
#endif

#ifdef _WIN32
    #ifdef LTRK416API_EXPORTS
      #define LTRK416API_DllExport(type) __declspec(dllexport) type APIENTRY
    #else
      #define LTRK416API_DllExport(type) __declspec(dllimport) type APIENTRY
    #endif
#elif defined __GNUC__
    #define LTRK416API_DllExport(type) __attribute__ ((visibility("default"))) type
#else
    #define LTRK416API_DllExport(type)  type
#endif


/** Размер строки с именем модуля в структуре #TLTRK416_MODULE_INFO */
#define LTRK416_NAME_SIZE               16
/** Размер строки с серийным номером модуля в структуре #TLTRK416_MODULE_INFO */
#define LTRK416_SERIAL_SIZE             24
/** Количество каналов RS-232 */
#define LTRK416_RS_CH_CNT               4
/** Количество каналов МКП MIL-1553 */
#define LTRK416_MIL_CH_CNT              2
/** Максимальный размер в словах буфера на канал вывода по RS-232 */
#define LTRK416_RS_CH_BUF_SIZE          1024
/** Максимальный размер буфера в словах на канал вывода по RS-232 */
#define LTRK416_MIL_CH_BUF_SIZE         128
/** Максимальное значение кода паузы, вставляемой после последнего передаваемого слова буфера */
#define LTRK416_PAUSE_MAX               254
/** Код паузы, означающуй однократную передачу буфера */
#define LTRK416_PAUSE_SINGLE            255

/** Максимальное значение величины, выставляемой на ЦАП */
#define LTRK416_DAC_VALUE_MAX           5
/** Максимальный код ЦАП */
#define LTRK416_DAC_CODE_MAX            0x0FFF

/** Стартовый адрес области пользовательских данных во flash-памяти  */
#define LTRK416_FLASH_USER_ADDR         512
/** Размер области пользовательских данных во flash-памяти */
#define LTRK416_FLASH_USER_SIZE         (3*512)



/** Коды ошибок, специфичные для LTRK415. */
typedef enum {
    LTRK416_ERR_INVALID_CH_NUM          = -11120, /**< Указан недействительный номер канала */
    LTRK416_ERR_INSUF_OUT_CH_BUF_SIZE   = -11121, /**< Превышен размер буфера канала на вывод */
} e_LTRK416_ERROR_CODES;

/** Дополнительные флаги в словах данных MIL-1553 на передачу */
typedef enum {
    LTRK416_MIL_DATA_FLAG_CMD           = 0x00010000U /**< Флаг указывает, что данное слово является командой, а не данными */
} e_LTRK416_MIL_DATA_FLAGS;


/** Флаги, использующиеся при выводе значения на ЦАП */
typedef enum {
    LTRK416_DAC_FLAG_VALUE   = 0x0001, /**< Указывает, что значение задает величину в диапазоне от 0 до #LTRK416_DAC_VALUE_MAX.
                                              Если не указан, то значение задано в кодах ЦАП. */
    LTRK416_DAC_FLAG_CALIBR  = 0x0002, /**< Применение калибровочных коэффициентов к заданному значению. */
} e_LTRK416_DAC_FLAGS;


/** Тип канала. Определяет каналу какого интерфейса соответствуют принятые данные после разбора */
typedef enum {
    LTRK416_CH_TYPE_RS                  = 0, /**< Слово принято по RS-232 */
    LTRK416_CH_TYPE_MIL                 = 1, /**< Слово принято по MIL-1553 */
} e_LTRK416_CH_TYPE;


/** Коды скорости приема/передачи по RS-232 */
typedef enum {
    LTRK416_RS_BAUDRATE_110     = 1,  /**< 110 бод/с */
    LTRK416_RS_BAUDRATE_300     = 2,  /**< 300 бод/с */
    LTRK416_RS_BAUDRATE_600     = 3,  /**< 600 бод/с */
    LTRK416_RS_BAUDRATE_1200    = 4,  /**< 1200 бод/с */
    LTRK416_RS_BAUDRATE_2400    = 5,  /**< 2400 бод/с */
    LTRK416_RS_BAUDRATE_4800    = 6,  /**< 4800 бод/с */
    LTRK416_RS_BAUDRATE_9600    = 7,  /**< 9600 бод/с */
    LTRK416_RS_BAUDRATE_19200   = 8,  /**< 19200 бод/с */
    LTRK416_RS_BAUDRATE_38400   = 9,  /**< 38400 бод/с */
    LTRK416_RS_BAUDRATE_57600   = 10, /**< 57690 бод/с */
    LTRK416_RS_BAUDRATE_115200  = 11, /**< 115200 бод/с */
    LTRK416_RS_BAUDRATE_230400  = 12, /**< 230400 бод/с */
    LTRK416_RS_BAUDRATE_460800  = 13, /**< 460800 бод/с */
    LTRK416_RS_BAUDRATE_921600  = 14  /**< 921600 бод/с */
} e_LTRK416_RS_BAUDRATE;

/** Тип принятого слова по интерфейсу MIL-1553 */
typedef enum {
    LTRK416_MIL_WORD_TYPE_DATA       = 0, /**< Принято слово данных */
    LTRK416_MIL_WORD_TYPE_CMD        = 1, /**< Принято командное слово или слово статуса */
    LTRK416_MIL_WORD_TYPE_RX_ERR     = 2, /**< Слово принято с ошибками и в данных содержится код ошибки вместо принятой информации */
} e_LTRK416_MIL_WORD_TYPE;


/** Коды ошибок при приеме слова по интерфейсу MIL-1553 */
typedef enum {
    LTRK416_MIL_WORD_ERR_CODE_SYNC_HALF_1 = 1, /**< Первая половина синхроимпульса */
    LTRK416_MIL_WORD_ERR_CODE_SYNC_RECV   = 2, /**< Прием синхроимпульса */
    LTRK416_MIL_WORD_ERR_CODE_SYNC_HALF_2 = 3, /**< Вторая половина синхроимпульса */
    LTRK416_MIL_WORD_ERR_CODE_WORD_RECV   = 4, /**< Ошибка приема слова */
    LTRK416_MIL_WORD_ERR_CODE_PARITY_RECV = 5, /**< Ошибка приема бита четности */
    LTRK416_MIL_WORD_ERR_CODE_PARITY_CHECK= 6, /**< Ошибка проверки бита четности */
} e_LTRK416_MIL_WORD_ERR_CODE;

#pragma pack (4)

/** @brief Калибровочные коэффициенты
    Структура, хранящая калибровочные коэффициенты для ЦАП. */
typedef struct {
    float   Offset; /**< Код смещения. */
    float   Scale;  /**< Коэффициент шкалы. */
} TLTRK416_CBR_COEF;

/** Информация о модуле */
typedef struct {
    CHAR        Name[LTRK416_NAME_SIZE];      /**< Название модуля (оканчивающаяся нулем ASCII-строка) */
    CHAR        Serial[LTRK416_SERIAL_SIZE];  /**< Серийный номер модуля (оканчивающаяся нулем ASCII-строка) */
    BYTE        VerPLD;                       /**< Версия прошивки PLD  */
    TLTRK416_CBR_COEF DacCbrCoefs;            /**< Калибровочные коэффициенты ЦАП. */
    DWORD       Reserved[13];                 /**< Резерв */
} TLTRK416_MODULE_INFO;


/** Стандартные настройки канала интерфейса приема-передачи (общие для RS-232 и MIL-1553) */
typedef struct {
    BOOLEAN TxEn; /**< Разрешение передачи. Если TRUE, то после запуска обмена модуль
                       будет циклически передавать предварительно загруженные данные */
    BOOLEAN RxEn; /**< Разрешение приема. Если TRUE, то после запуска обмена, по данному
                       каналу модуль будет принимать данные и передавать их в ПК */
    BYTE    PauseLen; /**< Данное поле задает длительность паузы при выводе межу последним словом циклического буфера
                           и первым словом следующего цикла. Значение 0 соответствует передаче следующего буфера
                           без паузы. Длительность паузы соответствующей единицы кода соответствует
                           выводу 11 бит на скорости 115200 или 95,4861 мкс. Максимальная пауза достигается при
                           значении поля равном #LTRK416_PAUSE_MAX. Значение поля, равное #LTRK416_PAUSE_SINGLE
                           имеет специальный смысл, при данной настройке буфер передается только один раз,
                           после чего передача вообще останавливается до останова и нового запуска обмена */
    BYTE    RSBaudrate; /**< Частота приема/передачи. Действительно только для RS-232.
                              Может принимать значения из #e_LTRK416_RS_BAUDRATE */
    DWORD Reserved[7]; /** < Резерв */
} TLTRK416_CH_CONFIG;



/** Настройки интерфейса RS-232 */
typedef struct {
    TLTRK416_CH_CONFIG Ch[LTRK416_RS_CH_CNT]; /**< Настройки каналов RS-232, по элементу на каждый канал */
    DWORD Reserved[8]; /**< Резерв */
} TLTRK416_RS_CONFIG;

/** Настройки интерфейса MIL-1553 */
typedef struct {
    TLTRK416_CH_CONFIG Ch[LTRK416_MIL_CH_CNT]; /**< Настройки каналов MIL-1553, по элементу на каждый канал */
    DWORD Reserved[8]; /**< Резерв */
} TLTRK416_MIL_CONFIG;

/** Настройки модуля LTRK416 */
typedef struct {
    TLTRK416_RS_CONFIG   RS; /**< Настройки RS-232 */
    TLTRK416_MIL_CONFIG  MIL; /**< Настройки MIL-1553 */
    DWORD Reserved[16]; /**< Резерв */
} TLTRK416_CONFIG;



/** Состояние выхода ЦАП */
typedef struct {
    double Data; /**< Значение для вывода на ЦАП в кодах или величинах, в зависимости от флагов. */
    WORD Flags;  /**< Набор флагов из #e_LTRK416_DAC_FLAGS, определяющих получение результирующего
                     кода ЦАП по заданному значению. */
    WORD ResCode; /**< Результирующий код ЦАП, который был записан. */
    DWORD Reserved[3]; /**< Резерв. */
} TLTRK416_DAC_STATE;

/** Состояние обмена по интерфейсам */
typedef struct {
    BOOLEAN Active; /**< Признак, что был запущен обмен данными по интерфейсам.
                         Изменяется библиотекой автоматически при вызовах
                         LTRK416_ExchangeStart() и LTRK416_ExchangeStop(). */
    DWORD Reserved[7]; /**< Резерв. */
} TLTRK416_EXCHANGE_STATE;

/** Параметры состояния модуля */
typedef struct {
    TLTRK416_DAC_STATE Dac; /**< Состояние выхода ЦАП */
    TLTRK416_EXCHANGE_STATE Exchange; /**< Состояние обмена по интерфейсам */
    DWORD Reserved[32]; /**< Резерв */
} TLTRK416_STATE;



/** Описатель модуля */
typedef struct {
    INT size;     /**< Размер структуры. Заполняется в LTRK416_Init(). */
    /** Структура, содержащая состояние соединения с сервером.
        Не используется напрямую пользователем. */
    TLTR Channel;
    /** Указатель на непрозрачную структуру с внутренними параметрами,
        используемыми исключительно библиотекой и недоступными для пользователя. */
    PVOID Internal;    
    DWORD Reserved[16]; /**< Резерв */
    TLTRK416_CONFIG Cfg;
    /** Состояние модуля и рассчитанные параметры. Поля изменяются функциями
        библиотеки. Пользовательской программой могут использоваться
        только для чтения. */
    TLTRK416_STATE State;
    TLTRK416_MODULE_INFO ModuleInfo; /**< Информация о модуле */
} TLTRK416;


/** Информация о принятом слове по одному из интерфейсов. Получается на основе
    стандартного слова LTR в результате обработки данных с помощью
    LTRK416_ProcessData() */
typedef struct {
    BYTE ChType; /**< Тип канала. Одно из значений из перечисления #e_LTRK416_CH_TYPE.
                      Используемый вариант далее идущих параметров из объединения
                      определяется данным типом. */
    BYTE ChNum; /**< Номер канала типа ChType, по которому было принято слово. */
    union {
        struct {
            BYTE Data; /**< Байт данных, принятых по каналу RS-232 */
            BOOLEAN ParityVal; /**< Значение бита четности, принятого по RS-232 */
            BOOLEAN ParityOk; /**< Признак корректности бита четности принятого по RS-232 слова */
        } RS; /**< Принятые данные по каналу типа #LTRK416_CH_TYPE_RS */
        struct {
            BYTE WordType; /**< Тип принятого слова по MIL-1553. Значение из
                                #e_LTRK416_MIL_WORD_TYPE. Данное значение определяет
                                используемый вариант оставшихся полей. Если данное поле равно
                                #LTRK416_MIL_WORD_TYPE_DATA или #LTRK416_MIL_WORD_TYPE_CMD,
                                то в поле Data указывается принятых 16-битный
                                код слова данных или каманды/статуса. Если данное
                                поле равно #LTRK416_MIL_WORD_TYPE_RX_ERR, то слово
                                принято с ошибкой, и вместо данных далее содержится
                                информация об ошибки в структуре Error. */
            union {
                WORD Data; /**< Принятое 16-битное слово данных или команды/статуса.
                                Действительно только для WordType #LTRK416_MIL_WORD_TYPE_DATA
                                или #LTRK416_MIL_WORD_TYPE_CMD */
                struct {
                    BYTE Code;  /**< Код ошибки приема слова. Коды ошибок определены
                                     в перечислении #e_LTRK416_MIL_WORD_ERR_CODE. */
                    BOOLEAN Parity; /**< Значение принятого бита четности. Присутствует
                                         только при наличии ошибки приема. Модуль сам
                                         проверяет бит четности при приеме по MIL-1553,
                                         и при несоответствии возвращает слово ошибки
                                         с кодом #LTRK416_MIL_WORD_ERR_CODE_PARITY_CHECK,
                                         отбрасывая полезные данные. Если же были приняты
                                         данные или команда/статус без ошибки, то
                                         это означает, что бит четности был успешно проверен,
                                         в этом случае его значение не передается, а передается
                                         только само 16-битное слово. */
                } Error; /**< Информация об ошибке приема слова. Действительна только
                              для типа принятого слова #LTRK416_MIL_WORD_TYPE_RX_ERR */
            };
        } MIL;  /**< Принятые данные по каналу типа #LTRK416_CH_TYPE_MIL */
    };
} TLTRK416_RX_DATA;

#pragma pack ()

/*============================================================================*/

LTRK416API_DllExport(INT) LTRK416_Init(TLTRK416 *hnd);
LTRK416API_DllExport(INT) LTRK416_Open(TLTRK416 *hnd, DWORD net_addr, WORD net_port,
                                       const CHAR *csn, WORD slot);
LTRK416API_DllExport(INT) LTRK416_Close(TLTRK416 *hnd);
LTRK416API_DllExport(INT) LTRK416_IsOpened(TLTRK416 *hnd);



/***************************************************************************//**
   @brief Установка значения ЦАП.

   Функция позволяет установить значение на ЦАП модуля. При этом т.к. значение
   ЦАП нельзя менять сразу на большую величину, данная функция выполняет постепенное
   изменение текущего значения ЦАП, в результате чего функция может выволняться
   значительное время до нескольких секунд.

   При успешном выводе обновляется значение состояния ЦАП в описателе
   модуля.

   @param[in] hnd       Описатель модуля.
   @param[in] data      Значение для вывода на ЦАП в кодах или величинах, в зависимости
                        от флагов.
   @param[in] flags     Набор флагов из #e_LTRK416_DAC_FLAGS.
   @return              Код ошибки.
 ******************************************************************************/
LTRK416API_DllExport(INT) LTRK416_SetDAC(TLTRK416 *hnd, double data, WORD flags);

LTRK416API_DllExport(INT) LTRK416_SetDACEx(TLTRK416 *hnd, double data, WORD flags, INT inc, DWORD cmd_delay);

/***************************************************************************//**
   @brief Вычисление значения кода ЦАП.

   Функция позволяет вычислить код ЦАП без непосредственно вывода его на ЦАП.

   @param[in] dac_cbr   Структура с калибровочными коэффициентами модуля.
   @param[in] data      Значение для вывода на ЦАП в кодах или величинах, в зависимости
                        от флагов.
   @param[in] flags     Набор флагов из #e_LTRK416_DAC_FLAGS.
   @return              Код ошибки.
 ******************************************************************************/
LTRK416API_DllExport(INT) LTRK416_CalcDacCode(const TLTRK416_CBR_COEF *dac_cbr, double data, DWORD flags, WORD *res_code);


/***************************************************************************//**
   @brief Загрузка значений циклического буфера RS-232 на вывод.

   Функция позволяет загрузить в буфер значения для последующего циклического вывода
   по интерфейсу RS-232.

   Каждый экземляр  интерфейса RS-232 имеет свой независимый циклический буфер
   размером в #LTRK416_RS_CH_BUF_SIZE слов и в случае резрешения вывода
   должен быть загружен индивидуально для каждого интерефса, для которого разрешен
   вывод, перед запуском обмена с помощью LTRK416_ExchangeStart().

   В каждом загружаемом слове действительны только младшие 8 бит, которые и задают
   код, который будует выведен по интерфейсу. Бит четности явно не задется, он
   вычисляется всегда самим модулем так, что общее число единиц в байте данных и
   бите четности будет нечетное.

   Весь буфер должен быть загружен за один вызов. Повторный вызов функции приводит
   к замене текущих данных буфера на новые.

   @param[in] hnd       Описатель модуля.
   @param[in] ch_num    Номер канала RS-232 от 0 до #LTRK416_RS_CH_CNT - 1.
   @param[in] data      Массив из size 32-битных слов, в которых действительны только
                        младшие 8 бит, определющих выводимый код по RS-232.
   @param[in] size      Количество слов в буфере data для загрузки в модуль. Может
                        быть от 1 до #LTRK416_RS_CH_BUF_SIZE.
   @return              Код ошибки.
 ******************************************************************************/
LTRK416API_DllExport(INT) LTRK416_LoadRsBuf(TLTRK416 *hnd, BYTE ch_num, const DWORD *data, WORD size);

/***************************************************************************//**
   @brief Загрузка значений циклического буфера MIL-1553 на вывод.

   Функция позволяет загрузить в буфер значения для последующего циклического вывода
   по интерфейсу MIL-1553.

   Каждый экземляр  интерфейса MIL-1553 имеет свой независимый циклический буфер
   размером в #LTRK416_MIL_CH_BUF_SIZE слов и в случае резрешения вывода
   должен быть загружен индивидуально для каждого интерефса, для которого разрешен
   вывод, перед запуском обмена с помощью LTRK416_ExchangeStart().

   В каждом загружаемом слове младшие 16 бит определяют содержимое самого слова,
   которое будет выведено по интерфейсу. Старшие же 16 бит включают дополнительные флаги,
   в частности служит для определения, что должно передаваться, данные или команда.
   Если должна быть передана команда, то 16 бит должны быть объединены по или
   с флагом #LTRK416_MIL_DATA_FLAG_CMD, в противном случае данные 16 бит будут
   интерпретироваться как слово данных.

   Бит четности явно не задется, он вычисляется всегда самим модулем так,
   что общее число единиц в 16-битных данных и бите четности будет нечетное.

   Весь буфер должен быть загружен за один вызов. Повторный вызов функции приводит
   к замене текущих данных буфера на новые.

   @param[in] hnd       Описатель модуля.
   @param[in] ch_num    Номер канала MIL-1553 от 0 до #LTRK416_MIL_CH_CNT - 1.
   @param[in] data      Массив из size 32-битных слов, в которых содержится 16-битный
                        передаваемый код и признак, данные это или команда/статус.
   @param[in] size      Количество слов в буфере data для загрузки в модуль. Может
                        быть от 1 до #LTRK416_MIL_CH_BUF_SIZE.
   @return              Код ошибки.
 ******************************************************************************/
LTRK416API_DllExport(INT) LTRK416_LoadMilBuf(TLTRK416 *hnd, BYTE ch_num, const DWORD *data, WORD size);


/***************************************************************************//**
   @brief Запуск обмена данными по интерфейсам.

   Функция выполняет запуск передачи и/или приема данных по каналам RS-232 и
   MIL-1553 в соответствии с заполненными настройками структуры #TLTRK416_CONFIG
   описателя модуля.

   Для каждого канала, для которого разрешена передача данных
   (@fieldref{TLTRK416_CH_CONFIG,TxEn} = TRUE) перед вызовом
   данной функции должен быть заполнен данными на передачу циклический буфер
   соответствующих каналов с помощью LTRK416_LoadRsBuf() и/или LTRK416_LoadMilBuf().
   После вызова LTRK416_ExchangeStart() для этих каналов модуль сам будет выдавать
   данные из циклического буфера по кругу. Данные внути буфера передаются подряд
   слово за словом (для RS-232 разделены одним битовым интервалом). Между завершением
   передачи одного цикла и началом следующего может быть вставлена пауза, заданная
   в поле @fieldref{TLTRK416_CH_CONFIG,PauseLen} настроек канала. Если значение
   @fieldref{TLTRK416_CH_CONFIG,PauseLen} равно #LTRK416_PAUSE_SINGLE, то буфер
   будет выдан один раз, после чего передача завершится и может быть возвобновлена
   только при останове обмена и повторном запуске.

   Для каждого канала, для которого разрешен прием данных
   (@fieldref{TLTRK416_CH_CONFIG,RxEn} = TRUE), модуль прослушивает
   соответствующий канал и в случае приема слова передает в ПК одно слово данных
   протокола LTR. Эти слова должны быть приняты с помощью LTRK416_Recv() и
   обработаны с помощью LTRK416_ProcessData().

   После вызова данной функции не должны измняться настройки модуля, а также
   до завершения обмена с помощью LTRK416_ExchangeStop()
   вызываться любые функции для данного модуля, кроме LTRK416_Recv() и
   обработаны с помощью LTRK416_ProcessData().


   @param[in] hnd       Описатель модуля.
   @return              Код ошибки.
 ******************************************************************************/
LTRK416API_DllExport(INT) LTRK416_ExchangeStart(TLTRK416 *hnd);

/***************************************************************************//**
   @brief Завершение обмена данными по интерфейсам.

   Данная функция останавливает прием и/или передачу данных по интерфейсам,
   запущенную с помощью LTRK416_ExchangeStart().

   После завершения данной функции конфигурация модуля и выводимые данные могут
   быть изменены, после чего обмен снова возобновлен.

   @param[in] hnd       Описатель модуля.
   @return              Код ошибки.
 ******************************************************************************/
LTRK416API_DllExport(INT) LTRK416_ExchangeStop(TLTRK416 *hnd);


/***************************************************************************//**
   @brief Прием данных от модуля

   Функция принимает запрашиваемое число слов от модуля. Возвращаемые слова
   находятся в специальном формате, который включает в себя служебную информацию.

   Для обработки принятых слов и получения значений измерений используется функция
   LTRK415_ProcessData().

   Функция возвращает управление либо когда примет запрошенное количество слов,
   либо по истечению таймаута. При этом реально принятое количество слов можно
   узнать по возвращенному значению.

   @param[in]  hnd      Описатель модуля.
   @param[out] data     Массив, в который будут сохранены принятые слова. Должен
                        быть размером на size 32-битных слов.
   @param[out] tmark    Указатель на массив размером на size 32-битных слов,
                        в который будут сохранены значения счетчиков синхрометок,
                        соответствующие принятым данным. Генерация меток
                        настраивается для крейта или специального модуля отдельно.
                        Синхрометки подробнее описаны в разделе "Синхрометки"
                        @docref_ltrapi{руководства для библиотеки ltrapi}.
                        Если синхрометки не используются, то можно передать
                        в качестве параметра нулевой указатель.
   @param[in]  size     Запрашиваемое количество 32-битных слов на прием.
   @param[in]  timeout  Таймаут на выполнение операции в миллисекундах. Если в течение
                        заданного времени не будет принято запрашиваемое количество
                        слов, то функция все равно вернет управление, возвратив
                        в качестве результата реально принятое количество слов
   @return              Значение меньше нуля соответствует коду ошибки. Значение
                        больше или равное нулю соответствует количеству реально
                        принятых и сохраненных в массив data слов.
*******************************************************************************/
LTRK416API_DllExport(INT) LTRK416_Recv(TLTRK416 *hnd, DWORD *data, DWORD *tstamp, DWORD size, DWORD timeout);

/***************************************************************************//**
    @brief Обработка принятых данных

    Функция позволяет на основе слов, принятых с помощью LTRK416_Recv(), получить
    информацию о номере канала, типе интерфейса, данных и дополнительных признаков,
    содержащихся в этом слове.

    На каждое слово функция возвращает структуру типа #TLTRK416_RX_DATA,
    в которой содержится вся инфомрация из исходного принятого слова.


    @param[in] hnd       Описатель модуля.
    @param[in] src_data  Массив из size 32-битных слов, принятых с помощью LTRK416_Recv().
    @param[out] dst_data Массив из структур типа #TLTRK416_ACQ_DATA, в котором будет
                         сохранен результат обработки. Количество элементов массива
                         должно быть не меньше переданного значения size. После
                         завершения функции данные количество заполненных элементов
                         будет указанно в измененном значении size.
    @param[in, out] size При вызове фунции в данном параметре должно быть записано
                         количество слов в src_data, которое необходимо обработать.
                         После завершения функции значение size будет обновлено и
                         содержать количество обработанных и сохраненных в dst_data
                         слов. Как правило эти значения должны совпадать, но могут
                         отличаться в случае обнаружения слов неизвестного формата.
    @return              Код ошибки.
 ******************************************************************************/
LTRK416API_DllExport(INT) LTRK416_ProcessData(TLTRK416 *hnd, const DWORD *src_data, TLTRK416_RX_DATA *dst_data, INT *size);


/***************************************************************************//**
   @brief Запись данных во flash-память модуля

   Функция записывает данные во flash-память модуля по заданному
   адресу. Пользователю выделяется область памяти с адреса
   #LTRK416_FLASH_USER_ADDR размером #LTRK416_FLASH_USER_SIZE байт.
   Предварительное стирание не требуется.

   @param[in] hnd      Описатель модуля.
   @param[in] addr     Адрес памяти, начиная с которого необходимо выполнить запись
   @param[in] data     Массив из size байт с данными, которые будут записаны
   @param[in] size     Количество данных в байтах, которое необходимо записать
   @return             Код ошибки.
 ******************************************************************************/
LTRK416API_DllExport(INT) LTRK416_FlashWrite(TLTRK416 *hnd, WORD addr, const BYTE *data, WORD size);

/***************************************************************************//**
   @brief Чтение данных из flash-памяти модуля

   Функция вычитывает данные, записанные во flash-памяти модуля по заданному
   адресу. Пользователю выделяется область памяти с адреса
   #LTRK416_FLASH_USER_ADDR размером #LTRK416_FLASH_USER_SIZE байт.

   @param[in] hnd      Описатель модуля.
   @param[in] addr     Адрес памяти, начиная с которого необходимо прочитать данные
   @param[out] data    Массив на size байт, в который будут записаны прочитанные
                       из Flash-памяти данные
   @param[in] size     Количество данных в байтах, которое необходимо прочитать
   @return             Код ошибки.
 ******************************************************************************/
LTRK416API_DllExport(INT) LTRK416_FlashRead(TLTRK416 *hnd, WORD addr, BYTE *data, WORD size);



/***************************************************************************//**
   @brief Получение сообщения об ошибке.

   Функция возвращает строку, соответствующую переданному коду ошибки, в кодировке
   CP1251 для Windows или UTF-8 для Linux. Функция может обработать как ошибки
   из LTRK416API, так и общие коды ошибок из ltrapi.

   @param[in] err       Код ошибки
   @return              Указатель на строку, содержащую сообщение об ошибке.
 ******************************************************************************/
LTRK416API_DllExport(LPCSTR) LTRK416_GetErrorString(INT err);


#ifdef __cplusplus
}                                          // only need to export C interface if
                                           // used by C++ source code
#endif

#endif                      /* #ifndef LTR11API_H_ */


