#include "l502api.h"
#include "e502api.h"

#include "../../devs/e502/e502_fpga_regs.h"
#include "dev_funcs.h"

#ifdef _WIN32
#include <locale.h>
#include <conio.h>

#else
#include <signal.h>
#include <unistd.h>
#include <string.h>
#include <sys/time.h>
#include <unistd.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <stdio.h>

#define TCP_CONNECTION_TOUT 5000
#define EVENT_CONNECTION_TOUT 100

#define DEV_E502_DEF  "E502"
#define DEV_E16_DEF  "E16"
#define DEV_LTA37_DEF  "LTA37"

static char const *DEV_E502 = DEV_E502_DEF;
static char const *DEV_E16 = DEV_E16_DEF;
static char const *DEV_LTA37 = DEV_LTA37_DEF;

typedef struct {
    uint32_t ip_addr;
    char const *devname;
} ip_dev_list_t;


/* Функция находит все подключенные модули по интерфейсам PCI-Express и USB и
 * сохраняет записи о этих устройствах в выделенный массив.
 * Также создаются записи по переданным IP-адресам модулей и добавляются в конец
 * массива.
 * Указатель на выделенный массив, который должен быть потом очищен, сохраняется
 * в pdevrec_list, а количество действительных элементов (память которых должна
 * быть в дальнейшем освобождена с помощью X502_FreeDevRecordList()) возвращается
 * как результат функции */
static uint32_t f_get_all_devrec(t_x502_devrec **pdevrec_list, ip_dev_list_t *ip_dev_list, unsigned ip_cnt) {
    int32_t fnd_devcnt = 0;
    uint32_t pci_devcnt = 0;
    uint32_t usb_devcnt = 0;
    uint32_t e16_usb_devcnt = 0;
    uint32_t eth_devcnt = 0;

    t_x502_devrec *rec_list = NULL;

    /* получаем количество подключенных устройств по интерфейсам PCI и USB */
    L502_GetDevRecordsList(NULL, 0, 0, &pci_devcnt);
    E502_UsbGetDevRecordsList(NULL, 0, 0, &usb_devcnt);
    E16_UsbGetDevRecordsList(NULL, 0, 0, &e16_usb_devcnt);
    E502_SearchEthForDevicesIPv4Addr(NULL, 0, 0, &eth_devcnt, EVENT_CONNECTION_TOUT, TCP_CONNECTION_TOUT);

    if ((pci_devcnt+usb_devcnt + e16_usb_devcnt + ip_cnt + eth_devcnt) != 0) {
        /* выделяем память для массива для сохранения найденного количества записей */
        rec_list = malloc((pci_devcnt + usb_devcnt +e16_usb_devcnt + ip_cnt + eth_devcnt) * sizeof(t_x502_devrec));

        if (rec_list != NULL) {
            unsigned i;
            /* получаем записи о модулях L502, но не больше pci_devcnt */
            if (pci_devcnt!=0) {
                int32_t res = L502_GetDevRecordsList(&rec_list[fnd_devcnt], pci_devcnt, 0, NULL);
                if (res >= 0) {
                    fnd_devcnt += res;
                }
            }
            /* добавляем записи о модулях E502, подключенных по USB, в конец массива */
            if (usb_devcnt!=0) {
                int32_t res = E502_UsbGetDevRecordsList(&rec_list[fnd_devcnt], usb_devcnt, 0, NULL);
                if (res >= 0) {
                    fnd_devcnt += res;
                }
            }

            /* добавляем записи о модулях E16, подключенных по USB, в конец массива */
            if (e16_usb_devcnt!=0) {
                int32_t res = E16_UsbGetDevRecordsList(&rec_list[fnd_devcnt], e16_usb_devcnt, 0, NULL);
                if (res >= 0) {
                    fnd_devcnt += res;
                }
            }

            if(eth_devcnt != 0) {
                int32_t res = E502_SearchEthForDevicesIPv4Addr(&rec_list[fnd_devcnt], 0, eth_devcnt, NULL, EVENT_CONNECTION_TOUT, TCP_CONNECTION_TOUT);
                if (res >= 0) {
                    fnd_devcnt += res;
                }
            }

            /* создаем записи для переданного массива ip-адресов */
            for (i=0; i < ip_cnt; i++) {
                if (E502_MakeDevRecordByIpAddr2(&rec_list[fnd_devcnt], ip_dev_list[i].ip_addr, 0, TCP_CONNECTION_TOUT, ip_dev_list[i].devname) == X502_ERR_OK) {
                    fnd_devcnt++;
                }
            }
        }
    }

    if (fnd_devcnt != 0) {
        /* если создана хотя бы одна запись, то сохраняем указатель на выделенный массив */
        *pdevrec_list = rec_list;
    } else {
        *pdevrec_list = NULL;
        free(rec_list);
    }

    return fnd_devcnt;
}


static t_x502_hnd f_dev_select_open(int argc, char** argv, uint32_t flags_clear) {
    t_x502_hnd hnd = NULL;
    uint32_t fnd_devcnt,i, dev_ind;
    t_x502_devrec *devrec_list = NULL;
    ip_dev_list_t *ip_dev_list = NULL;
    uint32_t ip_cnt = 0;

    /* если есть аргументы командной строки, то предполагаем, что это могут быть
       ip-адреса интересующих устройств. */
    if (argc > 1) {
        ip_dev_list = malloc((argc - 1) * sizeof(ip_dev_list[0]));
        if (ip_dev_list == NULL) {
            fprintf(stderr, "Ошибка выделения памяти!\n");
        } else {
            for (i=1; (int)i < argc; i++) {
                int a[4];
                if (sscanf(argv[i], "%d.%d.%d.%d", &a[0], &a[1], &a[2], &a[3])==4 ||
                    sscanf(argv[i], DEV_E502_DEF":%d.%d.%d.%d", &a[0], &a[1], &a[2], &a[3])==4
                ) {
                    ip_dev_list[ip_cnt].devname = DEV_E502;
                    ip_dev_list[ip_cnt++].ip_addr = ((a[0] & 0xFF) << 24) |
                                             ((a[1] & 0xFF) << 16) |
                                             ((a[2] & 0xFF) <<  8) |
                                             (a[3] & 0xFF);
                } else
                if (sscanf(argv[i], DEV_E16_DEF":%d.%d.%d.%d", &a[0], &a[1], &a[2], &a[3])==4) {
                    ip_dev_list[ip_cnt].devname = DEV_E16;
                    ip_dev_list[ip_cnt++].ip_addr = ((a[0] & 0xFF) << 24) |
                                             ((a[1] & 0xFF) << 16) |
                                             ((a[2] & 0xFF) <<  8) |
                                             (a[3] & 0xFF);
                } else
                if (sscanf(argv[i], DEV_LTA37_DEF":%d.%d.%d.%d", &a[0], &a[1], &a[2], &a[3])==4) {
                    ip_dev_list[ip_cnt].devname = DEV_LTA37;
                    ip_dev_list[ip_cnt++].ip_addr = ((a[0] & 0xFF) << 24) |
                                             ((a[1] & 0xFF) << 16) |
                                             ((a[2] & 0xFF) <<  8) |
                                             (a[3] & 0xFF);
                }
            }
        }
    }

    /* получаем список модулей для выбора */
    fnd_devcnt = f_get_all_devrec(&devrec_list, ip_dev_list, ip_cnt);

    if (fnd_devcnt == 0) {
        printf("Не найдено ни одного модуля\n");
    } else {
        /* выводим информацию по списку модулей */
        printf("Доступны следующие модули:\n");
        for (i=0; i < fnd_devcnt; i++) {
            printf("Модуль № %d: %s, %-9s", i, devrec_list[i].devname,
                   devrec_list[i].iface == X502_IFACE_PCI ? "PCI/PCIe" :
                   devrec_list[i].iface == X502_IFACE_USB ? "USB" :
                   devrec_list[i].iface == X502_IFACE_ETH ? "Ethernet" : "?");

            /* при подключении по сети по IP-адресу серийный номер можно узнать
               только после открытия соединения. При этом поле location
               содержит строку с описанием адреса устройства */
            if (devrec_list[i].iface != X502_IFACE_ETH) {
                printf("Сер. номер: %s\n", devrec_list[i].serial);
            } else {
                printf("Адрес: %s\n", devrec_list[i].location);
            }
        }

        /* выбираем нужный по введенному номеру модуля по порядку с клавиатуры */
        printf("Введите номер модуля, с которым хотите работать (от 0 до %d)\n", fnd_devcnt-1);
        fflush(stdout);
        scanf("%d", &dev_ind);

        if (dev_ind >= fnd_devcnt) {
            printf("Неверно указан номер модуля...\n");
        } else {
            /* если ввели номер правильно - создаем описатель */
            hnd = X502_Create();
            if (hnd==NULL) {
                fprintf(stderr, "Ошибка создания описателя модуля!");
            } else {
                devrec_list[dev_ind].flags &= ~flags_clear;
                /* устанавливаем связь с модулем по записи */
                int32_t err = X502_OpenByDevRecord(hnd, &devrec_list[dev_ind]);
                if (err != X502_ERR_OK && err != X502_ERR_FPGA_NOT_LOADED) {
                    fprintf(stderr, "Ошибка установления связи с модулем: %s!", X502_GetErrorString(err));
                    X502_Free(hnd);
                    hnd = NULL;
                }
            }
        }

        /* освобождение ресурсов действительных записей из списка */
        X502_FreeDevRecordList(devrec_list, fnd_devcnt);
        /* очистка памяти самого массива */
        free(devrec_list);
    }

    /* освобождаем выделенный массив под IP-адреса (если был выделен) */
    free(ip_dev_list);

    return hnd;
}

t_x502_hnd select_dev_from_list(int argc, char** argv, uint32_t flags_clear) {
    t_x502_hnd hnd;
    uint32_t ver;
    int32_t err = X502_ERR_OK;
    /* получаем версию библиотеки */
    ver = X502_GetLibraryVersion();
    printf("Версия библиотеки: %d.%d.%d\n", (ver >> 24)&0xFF, (ver>>16)&0xFF, (ver>>8)&0xFF);

    /********** Получение списка устройств и выбор, с каким будем работать ******************/
    hnd = f_dev_select_open(argc, argv, flags_clear);

    /********************************** Работа с модулем **************************/
    /* если успешно выбрали модуль и установили с ним связь - продолжаем работу */
    if (hnd == NULL) {
        return NULL;
    }

    /* получаем информацию */
    t_x502_info info;
    err = X502_GetDevInfo(hnd, &info);
    if (err != X502_ERR_OK) {
        fprintf(stderr, "Ошибка получения серийного информации о модуле: %s!", X502_GetErrorString(err));
    } else {
        /* выводим полученную информацию */
        printf("Установлена связь со следующим модулем:\n");
        printf(" Серийный номер          : %s\n", info.serial);
        printf(" Наличие ЦАП             : %s\n", info.devflags & X502_DEVFLAGS_DAC_PRESENT ? "Да" : "Нет");
        printf(" Наличие BlackFin        : %s\n", info.devflags & X502_DEVFLAGS_BF_PRESENT ? "Да" : "Нет");
        printf(" Наличие гальваноразвязки: %s\n", info.devflags & X502_DEVFLAGS_GAL_PRESENT ? "Да" : "Нет");
        printf(" Индустриальное исп.     : %s\n", info.devflags & X502_DEVFLAGS_INDUSTRIAL ? "Да" : "Нет");
        printf(" Наличие интерф. PCI/PCIe: %s\n", info.devflags & X502_DEVFLAGS_IFACE_SUPPORT_PCI ? "Да" : "Нет");
        printf(" Наличие интерф. USB     : %s\n", info.devflags & X502_DEVFLAGS_IFACE_SUPPORT_USB ? "Да" : "Нет");
        printf(" Наличие интерф. Ethernet: %s\n", info.devflags & X502_DEVFLAGS_IFACE_SUPPORT_ETH ? "Да" : "Нет");
        printf(" Версия ПЛИС             : %d.%d\n", (info.fpga_ver >> 8) & 0xFF, info.fpga_ver & 0xFF);
        printf(" Версия PLDA             : %d\n", info.plda_ver);
        if (info.mcu_firmware_ver != 0) {
            printf(" Версия прошивки ARM     : %d.%d.%d.%d\n",
                    (info.mcu_firmware_ver >> 24) & 0xFF,
                    (info.mcu_firmware_ver >> 16) & 0xFF,
                    (info.mcu_firmware_ver >>  8) & 0xFF,
                    info.mcu_firmware_ver & 0xFF);
        }
    }

    if(err && hnd) {
        X502_Free(hnd);
        hnd = NULL;
    }
    return hnd;
}

